unit linkedlist;

{$mode objfpc}{$h+}

interface

uses
  SysUtils; { for CompareMem }

type
  PLinkedArrayList = ^TLinkedArrayList;
  TLinkedArrayList = record
    Prev: PLinkedArrayList;
    Next: PLinkedArrayList;
    LastItem: PByte;
    Items: Byte;
  end;

  TLinkedList = class(TObject)
  private
    FFirst: PLinkedArrayList;
    FCurrent: PLinkedArrayList;
    FLast: PLinkedArrayList;
    FCurrentItem: PByte;
    FCount: integer;
    FItemSize: integer;
    FArraySize: integer;

    procedure Init;
    procedure Grow;
    procedure Shrink;

    function GetItem: PtrInt;
    function GetItemPointer: Pointer;
    procedure SetItem(NewItem: PtrInt);
  public
    constructor Create;
    destructor Destroy; override;

    procedure Clear;

    function First: boolean;
    function Prior: boolean;
    function Next: boolean;
    function Last: boolean;

    function Add(const AData: PtrInt): Pointer;
    function Insert(After: boolean): Pointer;
    function FindItem(AData: PtrInt): boolean;
    function FindData(AData: Pointer): boolean;
    procedure Delete;

    property ArraySize: integer read FArraySize write FArraySize;
    property Count: integer read FCount;
    property ItemSize: integer read FItemSize write FItemSize;
    property Item: PtrInt read GetItem write SetItem;
    property ItemPointer: Pointer read GetItemPointer;
  end;

implementation

const
  DefaultSize = 14 * sizeof(PtrInt);

constructor TLinkedList.Create;
begin
  inherited;

  FItemSize := sizeof(PtrInt);
  FArraySize := DefaultSize;
end;

destructor TLinkedList.Destroy;
begin
  Clear;

  inherited;
end;

procedure TLinkedList.Init;
begin
  GetMem(FCurrent, FArraySize);
  FFirst := FCurrent;
  FLast := FCurrent;
  FCurrentItem := @FCurrent^.Items;
  FCurrent^.Prev := nil;
  FCurrent^.Next := nil;
  FCurrent^.LastItem := FCurrentItem-FItemSize;
end;

procedure TLinkedList.Clear;
var
  lList, lNext: PLinkedArrayList;
begin
  lList := FFirst;
  while lList <> nil do	
  begin
    lNext := lList^.Next;
    FreeMem(lList);
    lList := lNext;
  end;
  FFirst := nil;
  FCurrent := nil;
  FLast := nil;
  FCurrentItem := nil;
  FCount := 0;
end;

function TLinkedList.GetItem: PtrInt; inline;
begin
  Result := FCurrentItem^;
end;

function TLinkedList.GetItemPointer: Pointer; inline;
begin
  Result := FCurrentItem;
end;

procedure TLinkedList.SetItem(NewItem: PtrInt); inline;
begin
  FCurrentItem^ := NewItem;
end;

function TLinkedList.First: boolean;
begin
  FCurrent := FFirst;
  Result := FFirst <> nil;
  if not Result then exit;
  FCurrentItem := @FCurrent^.Items;
end;

function TLinkedList.Prior: boolean;
begin
  Result := (FCurrentItem <> nil) 
    and ((FCurrentItem > @FCurrent^.Items) or (FCurrent^.Prev <> nil));
  if not Result then exit;

  if FCurrentItem = @FCurrent^.Items then
  begin
    FCurrent := FCurrent^.Prev;
    FCurrentItem := FCurrent^.LastItem;
  end else
    Dec(FCurrentItem, FItemSize);
end;

function TLinkedList.Next: boolean;
begin
  Result := (FCurrentItem <> nil) 
    and ((FCurrentItem < FCurrent^.LastItem) or (FCurrent^.Next <> nil));
  if not Result then exit;

  if FCurrentItem = FCurrent^.LastItem then
  begin
    FCurrent := FCurrent^.Next;
    FCurrentItem := @FCurrent^.Items;
  end else
    Inc(FCurrentItem, FItemSize);
end;

function TLinkedList.Last: boolean;
begin
  FCurrent := FLast;
  Result := FLast <> nil;
  if not Result then exit;
  FCurrentItem := FCurrent^.LastItem;
end;

function TLinkedList.Add(const AData: PtrInt): Pointer;
begin
  Last;
  Result := Insert(true);
  PtrInt(Result^) := AData;
end;

procedure TLinkedList.Grow;
  { post: one item "present" in new array listitem after FCurrent }
var
  lList: PLinkedArrayList;
begin
  GetMem(lList, FArraySize);
  if FCurrent^.Next <> nil then
    FCurrent^.Next^.Prev := lList
  else
    FLast := lList;
  lList^.Prev := FCurrent;
  lList^.Next := FCurrent^.Next;
  lList^.LastItem := @lList^.Items;
  FCurrent^.Next := lList;
end;

function TLinkedList.Insert(After: boolean): Pointer;
var
  numBytes: integer;
begin
  if FCurrentItem = nil then 
  begin
    Init;
    { no items: don't skip non-existing item }
    After := false;
  end;
  if FCurrent^.LastItem+2*FItemSize > PByte(FCurrent)+FArraySize then 
  begin
    Grow;
    Move(FCurrent^.LastItem^, FCurrent^.Next^.Items, FItemSize);
  end else begin
    Inc(FCurrent^.LastItem, FItemSize);
  end;
  if After then Next;
  { LastItem points to item after last item }
  numBytes := FCurrent^.LastItem - FCurrentItem;
  if numBytes > 0 then
    Move(FCurrentItem^, (FCurrentItem+FItemSize)^, numBytes);
  Inc(FCount);  
  Result := FCurrentItem;
end;

procedure TLinkedList.Shrink;
var
  lList: PLinkedArrayList;
begin
  lList := FCurrent;
  if FCurrent^.Prev <> nil then
  begin
    FCurrent^.Prev^.Next := FCurrent^.Next;
  end else begin
    FFirst := FCurrent^.Next;
  end;
  if FCurrent^.Next <> nil then
  begin
    FCurrent^.Next^.Prev := FCurrent^.Prev;
    FCurrentItem := @FCurrent^.Next^.Items;
    FCurrent := FCurrent^.Next;
  end else begin
    FLast := FCurrent^.Prev;
    if FLast <> nil then
    begin
      FCurrentItem := FLast^.LastItem;
      FCurrent := FLast;
    end else begin
      FCurrentItem := nil;
      FCurrent := nil;
    end;
  end;
  FreeMem(lList);
end;

procedure TLinkedList.Delete;
var
  numBytes: integer;
begin
  if FCurrentItem = nil then exit;
  numBytes := FCurrent^.LastItem - FCurrentItem;
  Dec(FCurrent^.LastItem, FItemSize);
  Dec(FCount);
  if numBytes > 0 then
    Move((FCurrentItem+FItemSize)^, FCurrentItem^, numBytes)
  else if FCurrent^.LastItem < @FCurrent^.Items then
    Shrink
  else if FCurrent^.Next <> nil then
  begin
    FCurrent := FCurrent^.Next;
    FCurrentItem := @FCurrent^.Items;
  end else 
    Dec(FCurrentItem, FItemSize);
end;

function TLinkedList.FindItem(AData: PtrInt): boolean;
begin
  Result := FCurrentItem <> nil;
  if not Result then exit;

  repeat
    Result := PtrInt(FCurrentItem^) = AData;
    if Result then exit;
  until not Next;
end;

function TLinkedList.FindData(AData: Pointer): boolean;
begin
  Result := FCurrentItem <> nil;
  if not Result then exit;
  
  repeat
    Result := CompareMem(AData, FCurrentItem, FItemSize);
    if Result then exit;
  until not Next;
end;

end.
