/*****************************************************************************
*
* Filename:      ks-959.c
* Version:       0.1
* Description:   Irda KingSun KS-959 USB Dongle
* Status:        Experimental
* Author:        Alex Villacís Lasso <a_villacis@palosanto.com>
*
*  	Based on stir4200, mcs7780, kingsun-sir drivers.
*
*	This program is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation; either version 2 of the License.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****************************************************************************/

/*
 * Following is my most current (2007-07-14) understanding of how the ks959
 * dongle is supposed to work. This information was deduced by reverse-engineering
 * and examining the USB traffic captured with USBSnoopy from the WinXP driver. 
 * Feel free to update here as more of the dongle is known.*
 *
 * General: This dongle exposes one interface with one interrupt IN endpoint. 
 * However, the interrupt endpoint is NOT used at all for this dongle. Instead,
 * this dongle uses control transfers for everything, including sending and 
 * receiving the IrDA frame data. Apparently the interrupt endpoint is just a
 * dummy to ensure the dongle has a valid interface to present to the PC. And I 
 * thought the DonShine dongle was weird... In addition, this dongle uses 
 * obfuscation (?!?!?), applied at the USB level, to hide the traffic, both sent
 * and received, from the dongle. I call it obfuscation because the XOR keying 
 * and padding required to produce an USB traffic acceptable for the dongle can
 * not be explained by any other technical requirement.
 *
 * Transmission: To transmit an IrDA frame, the driver must prepare a control 
 * URB with the following as a setup packet:
 *	bRequestType	0x00
 *	bRequest	0x09 USB_REQ_SET_CONFIGURATION, might be HID_REQ_SET_REPORT
 *	wValue		<length of valid data before padding, little endian>
 *	wIndex		0x0000
 *	wLength		0x0000
 * The payload packet must be manually wrapped and escaped (as in stir4200.c),
 * then sent padded and obfuscated. Both padding and obfuscation are implemented
 * in the procedure obfuscate_tx_buffer(). Suffice to say, the designer/programmer
 * of the dongle used his name as a source for the obfuscation. WTF?!
 *	
 * Reception: To receive data, the driver must poll the dongle regularly (like
 * kingsun-sir.c) with control URBs and the following as a setup packet:
 *	bRequestType	0x00
 *	bRequest	0x01 USB_REQ_CLEAR_FEATURE, might be HID_REQ_GET_REPORT
 *	wValue		0x0200 little endian
 *	wIndex		0x0000
 *	wLength		0x0000 (changes to 0x0800 little endian on return)
 * If there is data to be read, it will be returned as the response payload.
 * This data is (apparently) not padded, but it is obfuscated. To de-obfuscate
 * it, the driver must XOR every byte, in sequence, with a value that starts at 
 * 1 and is incremented with each byte processed, and then with 0x55. The value
 * incremented with each byte processed overflows as an unsigned char. The 
 * resulting bytes form a wrapped SIR frame that is then unwrapped and unescaped
 * as in stir4200.c The incremented value is NOT reset with each frame, but is
 * kept across the entire session with the dongle.
 *
 * Speed chage: To change the speed of the dongle, the driver prepares a control
 * URB with the following as a setup packet:
 *	bRequestType	0x00
 *	bRequest	0x09 USB_REQ_SET_CONFIGURATION, might be HID_REQ_SET_REPORT
 *	wValue		0x0200 little endian
 *	wIndex		0x0001 little endian
 *	wLength		0x0000
 * The payload is a 8-byte record, apparently identical to the one used in 
 * drivers/usb/serial/cypress_m8.c to change speed:
 * 	__u32 baudSpeed;
 *	unsigned int dataBits : 2;	// 0 - 5 bits 3 - 8 bits
 *	unsigned int : 1;
 *	unsigned int stopBits : 1;
 *	unsigned int parityEnable : 1;
 *	unsigned int parityType : 1;
 *	unsigned int : 1;
 *	unsigned int reset : 1;
 *	unsigned char reserved[3];	// set to 0
 *
 * For now only SIR speeds have been observed with this dongle. Therefore, 
 * nothing is known on what changes (if any) must be done to frame wrapping /
 * unwrapping for higher than SIR speeds. This driver assumes no change is
 * necessary and announces support for all the way to 4MBps.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <linux/usb.h>
#include <linux/device.h>
#include <linux/crc32.h>

#include <asm/unaligned.h>
#include <asm/byteorder.h>
#include <asm/uaccess.h>

#include <net/irda/irda.h>
#include <net/irda/wrapper.h>
#include <net/irda/crc.h>

#define KS959_VENDOR_ID 0x07d0
#define KS959_PRODUCT_ID 0x4959

/* These are the currently known USB ids */
static struct usb_device_id dongles[] = {
    /* KingSun Co,Ltd  IrDA/USB Bridge */
    { USB_DEVICE(KS959_VENDOR_ID, KS959_PRODUCT_ID) },
    { }
};

MODULE_DEVICE_TABLE(usb, dongles);

#define KINGSUN_MTT 0x07
#define KINGSUN_REQ_RECV 0x01
#define KINGSUN_REQ_SEND 0x09

#define KINGSUN_FIFO_SIZE		2048
#define CTRL_TIMEOUT		100	   /* milliseconds */

struct ks959_speedparams {
	__le32 baudrate;
	unsigned int data_bits : 2;	/* data bits - 5 (5..8) */
	unsigned int : 1;
	unsigned int stop_bits : 1;
	unsigned int parity_enable : 1;
	unsigned int parity_type : 1;
	unsigned int : 1;
	unsigned int reset : 1;
	__u8 reserved[3];
} __attribute__ ((packed));

struct ks959_cb {
	struct usb_device *usbdev;      /* init: probe_irda */
	struct net_device *netdev;      /* network layer */
	struct irlap_cb   *irlap;       /* The link layer we are binded to */
	struct net_device_stats stats;	/* network statistics */
	struct qos_info   qos;

	struct usb_ctrlrequest *	tx_setuprequest;
	struct urb *			tx_urb;		/* this is a CONTROL urb, not BULK/INTERRUPT */
	__u8 *				tx_buf_clear;	/* transmit buffer, before obfuscation */
	__u8 *				tx_buf_xored;	/* transmit buffer, after obfuscation */
							/* max length available is KINGSUN_FIFO_SIZE */

	struct usb_ctrlrequest *	rx_setuprequest;
	struct urb *			rx_urb;		/* this is a CONTROL urb, not BULK/INTERRUPT */
	__u8 *				rx_buf;		/* receive buffer, before de-obfuscation */
							/* de-obfuscation is done while unwrapping frame */
							/* max length available is KINGSUN_FIFO_SIZE */
	__u8				rx_variable_xormask;	/* variable portion of rx xormask */
	iobuff_t  			rx_unwrap_buff;	/* receive unwrap state machine */
	struct timeval			rx_time;
	
	struct ks959_speedparams	speedparams;

	spinlock_t lock;
	int receiving;
};

/* Procedure to perform the obfuscation/padding expected by the dongle 
 *
 * buf_cleartext	(IN) Cleartext version of the IrDA frame to transmit
 * len_cleartext	(IN) Length of the cleartext version of IrDA frame
 * buf_xoredtext	(OUT) Points to obfuscated version of frame build by proc
 * len_maxbuf		(OUT) Maximum space available at buf_xoredtext
 *
 * (return) 		length of obfuscated frame with padding
 *
 * If not enough space (as indicated by len_maxbuf vs. required padding), 
 * zero is returned
 *
 * The value of lookup_string is actually a required portion of the algorithm. 
 * It seems the designer of the dongle wanted to state who exactly is responsible
 * for implementing obfuscation. Send your best (or other) wishes to him ]:-)
 */
static unsigned int obfuscate_tx_buffer(
	const __u8 * buf_cleartext, unsigned int len_cleartext,
	__u8 * buf_xoredtext, unsigned int len_maxbuf)
{
	unsigned int len_xoredtext;

	/* Calculate required length with padding, check for necessary space */
	len_xoredtext = ((len_cleartext + 7) & ~0x7) + 0x10;
	if (len_xoredtext <= len_maxbuf) {
		static const __u8 lookup_string[] = "wangshuofei19710";
		__u8 xor_mask;
		
		/* Unlike the WinXP driver, we *do* clear out the padding */
		memset(buf_xoredtext, 0, len_xoredtext);

		xor_mask = lookup_string[(len_cleartext & 0x0f) ^ 0x06] ^ 0x55;
		
		while (len_cleartext-- > 0) *buf_xoredtext++ = *buf_cleartext++ ^ xor_mask;
	} else {
		len_xoredtext = 0;
	}
	return len_xoredtext;
}

/* Callback transmission routine */
static void ks959_send_irq(struct urb *urb)
{
	struct ks959_cb *kingsun = urb->context;
	struct net_device *netdev = kingsun->netdev;

	/* in process of stopping, just drop data */
	if (!netif_running(kingsun->netdev)) {
		err("ks959_send_irq: Network not running!");
		return;
	}

	/* unlink, shutdown, unplug, other nasties */
	if (urb->status != 0) {
		err("ks959_send_irq: urb asynchronously failed - %d", urb->status);
	}
	netif_wake_queue(netdev);
}

/*
 * Called from net/core when new frame is available.
 */
static int ks959_hard_xmit(struct sk_buff *skb, struct net_device *netdev)
{
	struct ks959_cb *kingsun;
	unsigned int wraplen;
	unsigned int padlen;
	int ret = 0;

	if (skb == NULL || netdev == NULL)
		return -EINVAL;

	netif_stop_queue(netdev);

	/* the IRDA wrapping routines don't deal with non linear skb */
	SKB_LINEAR_ASSERT(skb);

	kingsun = netdev_priv(netdev);

	spin_lock(&kingsun->lock);

	/* Append data to the end of whatever data remains to be transmitted */
	wraplen = async_wrap_skb(skb, kingsun->tx_buf_clear, KINGSUN_FIFO_SIZE);

	/* Perform dongle obfuscation */
	padlen = obfuscate_tx_buffer(kingsun->tx_buf_clear, wraplen,
		kingsun->tx_buf_xored, KINGSUN_FIFO_SIZE);

	/* Calculate how much data can be transmitted in this urb */
	kingsun->tx_setuprequest->bRequestType = USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE;
	kingsun->tx_setuprequest->bRequest = KINGSUN_REQ_SEND;
	kingsun->tx_setuprequest->wValue = cpu_to_le16(wraplen);
	kingsun->tx_setuprequest->wIndex = cpu_to_le16(0x0000);
	kingsun->tx_setuprequest->wLength = cpu_to_le16(padlen);
	usb_fill_control_urb(kingsun->tx_urb, kingsun->usbdev,
			usb_sndctrlpipe(kingsun->usbdev, 0),
			(unsigned char *)kingsun->tx_setuprequest,
			kingsun->tx_buf_xored, padlen,
			ks959_send_irq, kingsun);	
	kingsun->tx_urb->status = 0;

	if ((ret = usb_submit_urb(kingsun->tx_urb, GFP_ATOMIC))) {
		err("ks959_hard_xmit: failed tx_urb submit: %d", ret);
		switch (ret) {
		case -ENODEV:
		case -EPIPE:
			break;
		default:
			kingsun->stats.tx_errors++;
			netif_start_queue(netdev);
		}
	} else {
		kingsun->stats.tx_packets++;
		kingsun->stats.tx_bytes += skb->len;
	}
	
	dev_kfree_skb(skb);
	spin_unlock(&kingsun->lock);
	
	return ret;
}

/* Receive callback function */
static void ks959_rcv_irq(struct urb *urb)
{
	struct ks959_cb *kingsun = urb->context;
	int ret;

	/* in process of stopping, just drop data */
	if (!netif_running(kingsun->netdev)) {
		kingsun->receiving = 0;
		return;
	}

	/* unlink, shutdown, unplug, other nasties */
	if (urb->status != 0) {
		err("kingsun_rcv_irq: urb asynchronously failed - %d", urb->status);
		kingsun->receiving = 0;
		return;
	}

	if (urb->actual_length > 0) {
		__u8 *bytes = urb->transfer_buffer;
		unsigned int i;
		
		for (i = 0; i < urb->actual_length; i++) {
			/* De-obfuscation implemented here: variable portion of
			   xormask is incremented, and then used with the encoded
			   byte for the XOR. The result of the operation is used
			   to unwrap the SIR frame. */
			kingsun->rx_variable_xormask++;
			async_unwrap_char(kingsun->netdev, &kingsun->stats,
				&kingsun->rx_unwrap_buff, 
				bytes[i] ^ kingsun->rx_variable_xormask ^ 0x55u);
		}
		kingsun->netdev->last_rx = jiffies;
		do_gettimeofday(&kingsun->rx_time);
		kingsun->receiving = (kingsun->rx_unwrap_buff.state != OUTSIDE_FRAME) ? 1 : 0;
	}
	
	/* This urb has already been filled in kingsun_net_open. Setup
	   packet must be re-filled, but it is assumed that urb keeps the
	   pointer to the initial setup packet, as well as the payload buffer. */
	kingsun->rx_setuprequest->bRequestType = USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE;
	kingsun->rx_setuprequest->bRequest = KINGSUN_REQ_RECV;
	kingsun->rx_setuprequest->wValue = cpu_to_le16(0x0200);
	kingsun->rx_setuprequest->wIndex = cpu_to_le16(0x0000);
	kingsun->rx_setuprequest->wLength = cpu_to_le16(0);
	ret = usb_submit_urb(urb, GFP_ATOMIC);
}

/* Send a control request to change speed of the dongle */
static int ks959_change_speed(struct ks959_cb * kingsun, unsigned speed)
{
	static unsigned int supported_speeds[] = {2400, 9600, 19200, 38400,
		57600, 115200, 576000, 1152000, 4000000, 0};
	int err;
	unsigned int i;
	
	/* Check that requested speed is among the supported ones */
	for (i = 0; supported_speeds[i] != 0 && supported_speeds[i] != speed; i++);
	if (supported_speeds[i] == 0) return -EOPNOTSUPP;

	memset(&(kingsun->speedparams), 0, sizeof(struct ks959_speedparams));
	kingsun->speedparams.baudrate = speed;
	kingsun->speedparams.data_bits = 3;	/* 8 data bits */
		
	err = usb_control_msg(
		kingsun->usbdev,
		usb_sndctrlpipe(kingsun->usbdev, 0),
		KINGSUN_REQ_SEND,
		USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE,
		0x0200, 
		0x0001,
		&(kingsun->speedparams), sizeof(struct ks959_speedparams),
		CTRL_TIMEOUT);
	if (err > 0) err = 0;
	return err;
}

/*
 * Function kingsun_net_open (dev)
 *
 *    Network device is taken up. Usually this is done by "ifconfig irda0 up"
 */
static int ks959_net_open(struct net_device *netdev)
{
	struct ks959_cb *kingsun = netdev_priv(netdev);
	int err = -ENOMEM;
	char hwname[16];

	/* At this point, urbs are NULL, and skb is NULL (see kingsun_probe) */
	kingsun->receiving = 0;

	/* Initialize speed for dongle */
	err = ks959_change_speed(kingsun, 9600);
	if (err < 0) goto err_exit;

	/* Initialize for SIR to copy data directly into skb.  */
	kingsun->rx_unwrap_buff.in_frame = FALSE;
	kingsun->rx_unwrap_buff.state = OUTSIDE_FRAME;
	kingsun->rx_unwrap_buff.truesize = IRDA_SKB_MAX_MTU;
	kingsun->rx_unwrap_buff.skb = dev_alloc_skb(IRDA_SKB_MAX_MTU);
	if (!kingsun->rx_unwrap_buff.skb) 
		goto free_mem;

	skb_reserve(kingsun->rx_unwrap_buff.skb, 1);
	kingsun->rx_unwrap_buff.head = kingsun->rx_unwrap_buff.skb->data;
	do_gettimeofday(&kingsun->rx_time);

	kingsun->rx_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!kingsun->rx_urb) 
		goto free_mem;

	kingsun->tx_urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!kingsun->tx_urb) 
		goto free_mem;
	
	/*
	 * Now that everything should be initialized properly,
	 * Open new IrLAP layer instance to take care of us...
	 */
	sprintf(hwname, "usb#%d", kingsun->usbdev->devnum);
	kingsun->irlap = irlap_open(netdev, &kingsun->qos, hwname);
	if (!kingsun->irlap) {
		err("ks-959: irlap_open failed");
		goto free_mem;
	}

	/* Start first reception */
	kingsun->rx_setuprequest->bRequestType = USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE;
	kingsun->rx_setuprequest->bRequest = KINGSUN_REQ_RECV;
	kingsun->rx_setuprequest->wValue = cpu_to_le16(0x0200);
	kingsun->rx_setuprequest->wIndex = cpu_to_le16(0x0000);
	kingsun->rx_setuprequest->wLength = cpu_to_le16(0);
	usb_fill_control_urb(kingsun->rx_urb, kingsun->usbdev,
			usb_rcvctrlpipe(kingsun->usbdev, 0),
			(unsigned char *)kingsun->rx_setuprequest,
			kingsun->rx_buf, KINGSUN_FIFO_SIZE,
			ks959_rcv_irq, kingsun);	
	kingsun->rx_urb->status = 0;
	err = usb_submit_urb(kingsun->rx_urb, GFP_KERNEL);
	if (err) {
		err("ks-959: first urb-submit failed: %d", err);
		goto close_irlap;
	}

	netif_start_queue(netdev);

	/* Situation at this point:
	   - all work buffers allocated
	   - urbs allocated and ready to fill
	   - max rx packet known (in max_rx)
	   - unwrap state machine initialized, in state outside of any frame
	   - receive request in progress
	   - IrLAP layer started, about to hand over packets to send
	 */

	return 0;

 close_irlap:
	irlap_close(kingsun->irlap);
 free_mem:
	if (kingsun->tx_urb) {
		usb_free_urb(kingsun->tx_urb);
		kingsun->tx_urb = NULL;
	}
	if (kingsun->rx_urb) {
		usb_free_urb(kingsun->rx_urb);
		kingsun->rx_urb = NULL;
	}
	if (kingsun->rx_unwrap_buff.skb) {
		kfree_skb(kingsun->rx_unwrap_buff.skb);
		kingsun->rx_unwrap_buff.skb = NULL;
		kingsun->rx_unwrap_buff.head = NULL;
	}
err_exit:
	return err;
}

/*
 * Function kingsun_net_close (kingsun)
 *
 *    Network device is taken down. Usually this is done by
 *    "ifconfig irda0 down"
 */
static int ks959_net_close(struct net_device *netdev)
{
	struct ks959_cb *kingsun = netdev_priv(netdev);

	/* Stop transmit processing */
	netif_stop_queue(netdev);

	/* Mop up receive && transmit urb's */
	usb_kill_urb(kingsun->tx_urb);
	usb_kill_urb(kingsun->rx_urb);
	
	usb_free_urb(kingsun->tx_urb);
	usb_free_urb(kingsun->rx_urb);
	
	kingsun->tx_urb = NULL;
	kingsun->rx_urb = NULL;
	
	kfree_skb(kingsun->rx_unwrap_buff.skb);
	kingsun->rx_unwrap_buff.skb = NULL;
	kingsun->rx_unwrap_buff.head = NULL;
	kingsun->rx_unwrap_buff.in_frame = FALSE;
	kingsun->rx_unwrap_buff.state = OUTSIDE_FRAME;
	kingsun->receiving = 0;

	/* Stop and remove instance of IrLAP */
	if (kingsun->irlap)
		irlap_close(kingsun->irlap);

	kingsun->irlap = NULL;

	return 0;
}

/*
 * IOCTLs : Extra out-of-band network commands...
 */
static int ks959_net_ioctl(struct net_device *netdev, struct ifreq *rq, int cmd)
{
	struct if_irda_req *irq = (struct if_irda_req *) rq;
	struct ks959_cb *kingsun = netdev_priv(netdev);
	int ret = 0;

	switch (cmd) {
	case SIOCSBANDWIDTH: /* Set bandwidth */
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;

		/* Check if the device is still there */
		if (netif_device_present(kingsun->netdev))
			return ks959_change_speed(kingsun, irq->ifr_baudrate);
		break;

	case SIOCSMEDIABUSY: /* Set media busy */
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;

		/* Check if the IrDA stack is still there */
		if (netif_running(kingsun->netdev))
			irda_device_set_media_busy(kingsun->netdev, TRUE);
		break;

	case SIOCGRECEIVING:
		/* Only approximately true */
		irq->ifr_receiving = kingsun->receiving;
		break;

	default:
		ret = -EOPNOTSUPP;
	}

	return ret;
}

/*
 * Get device stats (for /proc/net/dev and ifconfig)
 */
static struct net_device_stats *ks959_net_get_stats(struct net_device *netdev)
{
	struct ks959_cb *kingsun = netdev_priv(netdev);
	return &kingsun->stats;
}

/*
 * This routine is called by the USB subsystem for each new device
 * in the system. We need to check if the device is ours, and in
 * this case start handling it.
 */
static int ks959_probe(struct usb_interface *intf,
		      const struct usb_device_id *id)
{
	struct usb_device *dev = interface_to_usbdev(intf);
	struct ks959_cb *kingsun = NULL;
	struct net_device *net = NULL;
	int ret = -ENOMEM;


	/* Allocate network device container. */
	net = alloc_irdadev(sizeof(*kingsun));
	if(!net)
		goto err_out1;

	SET_MODULE_OWNER(net);
	SET_NETDEV_DEV(net, &intf->dev);
	kingsun = netdev_priv(net);
	kingsun->netdev = net;
	kingsun->usbdev = dev;
	kingsun->irlap = NULL;
	kingsun->tx_setuprequest = NULL;
	kingsun->tx_urb = NULL;
	kingsun->tx_buf_clear = NULL;
	kingsun->tx_buf_xored = NULL;
	
	kingsun->rx_setuprequest = NULL;
	kingsun->rx_urb = NULL;
	kingsun->rx_buf = NULL;
	kingsun->rx_variable_xormask = 0;
	kingsun->rx_unwrap_buff.in_frame = FALSE;
	kingsun->rx_unwrap_buff.state = OUTSIDE_FRAME;
	kingsun->rx_unwrap_buff.skb = NULL;
	kingsun->receiving = 0;
	spin_lock_init(&kingsun->lock);

	/* Allocate input buffer */
	kingsun->rx_buf = (__u8 *)kmalloc(KINGSUN_FIFO_SIZE, GFP_KERNEL);
	if (!kingsun->rx_buf) 
		goto free_mem;

	/* Allocate input setup packet */
	kingsun->rx_setuprequest = (struct usb_ctrlrequest *)kmalloc(sizeof(struct usb_ctrlrequest), GFP_KERNEL);
	if (!kingsun->rx_setuprequest)
		goto free_mem;
	
	/* Allocate output buffer */
	kingsun->tx_buf_clear = (__u8 *)kmalloc(KINGSUN_FIFO_SIZE, GFP_KERNEL);
	if (!kingsun->tx_buf_clear) 
		goto free_mem;
	kingsun->tx_buf_xored = (__u8 *)kmalloc(KINGSUN_FIFO_SIZE, GFP_KERNEL);
	if (!kingsun->tx_buf_xored) 
		goto free_mem;

	/* Allocate output setup packet */
	kingsun->tx_setuprequest = (struct usb_ctrlrequest *)kmalloc(sizeof(struct usb_ctrlrequest), GFP_KERNEL);
	if (!kingsun->tx_setuprequest)
		goto free_mem;
	
	printk(KERN_INFO "KingSun KS-959 IRDA/USB found at address %d, "
		"Vendor: %x, Product: %x\n",
	       dev->devnum, le16_to_cpu(dev->descriptor.idVendor),
	       le16_to_cpu(dev->descriptor.idProduct));

	/* Initialize QoS for this device */
	irda_init_max_qos_capabilies(&kingsun->qos);

	/* That's the Rx capability. */
	kingsun->qos.baud_rate.bits       = IR_2400 | IR_9600 | IR_19200 | IR_38400 |
		IR_57600 | IR_115200 | IR_576000 | IR_1152000 | (IR_4000000 << 8);
	kingsun->qos.min_turn_time.bits   &= KINGSUN_MTT;
	irda_qos_bits_to_value(&kingsun->qos);

	/* Override the network functions we need to use */
	net->hard_start_xmit = ks959_hard_xmit;
	net->open            = ks959_net_open;
	net->stop            = ks959_net_close;
	net->get_stats	     = ks959_net_get_stats;
	net->do_ioctl        = ks959_net_ioctl;

	ret = register_netdev(net);
	if (ret != 0)
		goto free_mem;

	info("IrDA: Registered KingSun KS-959 device %s", net->name);

	usb_set_intfdata(intf, kingsun);

	/* Situation at this point:
	   - all work buffers allocated
	   - urbs not allocated, set to NULL
	   - max rx packet known (is KINGSUN_FIFO_SIZE)
	   - unwrap state machine (partially) initialized, but skb == NULL
	 */

	return 0;

free_mem:
	if (kingsun->tx_setuprequest) kfree(kingsun->tx_setuprequest);
	if (kingsun->tx_buf_xored) kfree(kingsun->tx_buf_xored);
	if (kingsun->tx_buf_clear) kfree(kingsun->tx_buf_clear);
	if (kingsun->rx_setuprequest) kfree(kingsun->rx_setuprequest);
	if (kingsun->rx_buf) kfree(kingsun->rx_buf);
	free_netdev(net);
err_out1:
	return ret;
}

/*
 * The current device is removed, the USB layer tell us to shut it down...
 */
static void ks959_disconnect(struct usb_interface *intf)
{
	struct ks959_cb *kingsun = usb_get_intfdata(intf);

	if (!kingsun)
		return;

	unregister_netdev(kingsun->netdev);

	/* Mop up receive && transmit urb's */
	if (kingsun->tx_urb != NULL) {
		usb_kill_urb(kingsun->tx_urb);
		usb_free_urb(kingsun->tx_urb);
		kingsun->tx_urb = NULL;
	}
	if (kingsun->rx_urb != NULL) {
		usb_kill_urb(kingsun->rx_urb);
		usb_free_urb(kingsun->rx_urb);
		kingsun->rx_urb = NULL;
	}

	kfree(kingsun->tx_setuprequest);
	kfree(kingsun->tx_buf_xored);
	kfree(kingsun->tx_buf_clear);
	kfree(kingsun->rx_setuprequest);
	kfree(kingsun->rx_buf);
	free_netdev(kingsun->netdev);

	usb_set_intfdata(intf, NULL);
}

#ifdef CONFIG_PM
/* USB suspend, so power off the transmitter/receiver */
static int ks959_suspend(struct usb_interface *intf, pm_message_t message)
{
	struct ks959_cb *kingsun = usb_get_intfdata(intf);

	netif_device_detach(kingsun->netdev);
	if (kingsun->tx_urb != NULL) usb_kill_urb(kingsun->tx_urb);
	if (kingsun->rx_urb != NULL) usb_kill_urb(kingsun->rx_urb);
	return 0;
}

/* Coming out of suspend, so reset hardware */
static int ks959_resume(struct usb_interface *intf)
{
	struct ks959_cb *kingsun = usb_get_intfdata(intf);

	if (kingsun->rx_urb != NULL) {
		kingsun->rx_setuprequest->bRequestType = USB_DIR_OUT | USB_TYPE_STANDARD | USB_RECIP_DEVICE;
		kingsun->rx_setuprequest->bRequest = KINGSUN_REQ_RECV;
		kingsun->rx_setuprequest->wValue = cpu_to_le16(0x0200);
		kingsun->rx_setuprequest->wIndex = cpu_to_le16(0x0000);
		kingsun->rx_setuprequest->wLength = cpu_to_le16(0);
		usb_submit_urb(kingsun->rx_urb, GFP_KERNEL);
	}
	netif_device_attach(kingsun->netdev);

	return 0;
}
#endif

/*
 * USB device callbacks
 */
static struct usb_driver irda_driver = {
	.name		= "ks-959",
	.probe		= ks959_probe,
	.disconnect	= ks959_disconnect,
	.id_table	= dongles,
#ifdef CONFIG_PM
	.suspend	= ks959_suspend,
	.resume		= ks959_resume,
#endif
};

/*
 * Module insertion
 */
static int __init ks959_init(void)
{
	return usb_register(&irda_driver);
}
module_init(ks959_init);

/*
 * Module removal
 */
static void __exit ks959_cleanup(void)
{
	/* Deregister the driver and remove all pending instances */
	usb_deregister(&irda_driver);
}
module_exit(ks959_cleanup);

MODULE_AUTHOR("Alex Villacís Lasso <a_villacis@palosanto.com>");
MODULE_DESCRIPTION("IrDA-USB Dongle Driver for KingSun KS-959");
MODULE_LICENSE("GPL");

