# -*- coding: utf-8 -*-
"""
ldr.py

Display analog data from Arduino using Python (matplotlib)

Author: Mahesh Venkitachalam
Website: electronut.in
"""
import sys, argparse, time

import matplotlib.gridspec as gridspec
import matplotlib.pyplot as plt 
import matplotlib.animation as animation

from lib.AnalogPlot import AnalogPlot                                          # Biblioteca de buffer circular         


# main() function
def main():
    

    # create parser
    parser = argparse.ArgumentParser(description="LDR serial")
    # add expected arguments
    parser.add_argument('--port', dest='port', required=True)

    # parse args
    args = parser.parse_args()
  
    #strPort = '/dev/tty.usbserial-A7006Yqh'
    strPort = args.port

    print('reading from serial port %s...' % strPort)
        
    # Definimos o tempo inicial em que a aquisição começa
    t0 = time.time();   
    
    # A animação será apresentada de 50 em 50ms.
    tempo_anim = 90;
    
    # Atualiza o plot de 5 em 5 segundos
    draw_chk = 5000/tempo_anim;
    draw_conter = 0;
    
    # Dessa forma, definimos o tempo de amostragem na tela
    tempo_tela = 5;
    y_max = 1024
    
    # Variáveis a serem capturadas 
    # MPX, MAP, TPS, O2A, O2B, MAT, CLT, AD01, AD02, Rot
    n_var = 10;
    
    # Definimos o tamanho do buffer circular
    ringlen = int(tempo_tela/(tempo_anim*1e-3))
    
    # Define a porta serial
    ser_baudrate = 115200        

    # plot parameters
    # MPX, MAP, TPS, O2A, O2B, MAT, CLT, AD01, AD02, Rot
    Variaveis = AnalogPlot(strPort, ser_baudrate, ringlen, tempo_anim)

    print('plotting data...')

    # Cria a primeira figura
    # Nesse gráfico iremos plotar MAP, MPX, TPS, O2
    fig1 = plt.figure(1)
    
    # Dividimos o gráfico em dois (superior e inferior)
    gs1 = gridspec.GridSpec(2, 8)                                                  # Grid 2x1 

    # Define os espaçamntos
    gs1.update(hspace=0.17, left=0.02, bottom=0.08, right=0.98, top=0.92)    
    
    
    # -----------------------------------------------------------------------------
    # Chama a figura 1
    plt.figure(1)
    # Cria os plots
    # Gráfico superior com referência ao eixo Y da esquerda
    f1s1_ax1 = plt.subplot(gs1[0,0:4])
    # Gráfico superior com referência ao eixo Y da direita
    f1s1_ax2 = f1s1_ax1.twinx()               
    # Gráfico Inferior
    f1s2 = plt.subplot(gs1[1,0:4])

    # Define o label inferior como sendo de tempo
    f1s2.set_xlabel('Tempo [s]')
    # Desliga os marcadores do eixo superior
    f1s1_ax1.tick_params(labelbottom='off')

    # Ativa os grid para os dois gráficos
    f1s1_ax1.grid()
    f1s2.grid()

    # Mostra o gráfico
    fig1.show()    

    # -----------------------------------------------------------------------------
    # Chama a figura 2
    #plt.figure(2)
    # Cria todos os plots
    f2s1 = plt.subplot(gs1[0,4:6])         
    f2s2 = plt.subplot(gs1[1,4:6])
    f2s3 = plt.subplot(gs1[0,6:8])         
    f2s4 = plt.subplot(gs1[1,6:8])

    # Desativa os parâmetros de tempo dos gráficos superiores
    f2s1.tick_params(labelbottom='off')
    f2s3.tick_params(labelbottom='off')

    # Ativa os grid para os quatro gráficos
    f2s1.grid()
    f2s2.grid()
    f2s3.grid()
    f2s4.grid()
    
    #------------------------------------------------------------------------------
    # Como todos os gráficos estarão no intervalo de 30s, criamos os limites
    f1s1_ax1.axis((0,tempo_tela,0,y_max))
    f1s1_ax2.axis((0,tempo_tela,0,y_max))
    f1s2.axis((0,tempo_tela,0,y_max))
    f2s1.axis((0,tempo_tela,0,y_max))
    f2s2.axis((0,tempo_tela,0,y_max))
    f2s3.axis((0,tempo_tela,0,y_max))
    f2s4.axis((0,tempo_tela,0,y_max))

    # Invertemos o eixo x pois os dados correm da direita para a esquerda
    f1s1_ax1.invert_xaxis()
    f1s1_ax2.invert_xaxis()
    f1s2.invert_xaxis()
    f2s2.invert_xaxis()
    f2s4.invert_xaxis()

    #------------------------------------------------------------------------------
    # Criamos os gráficos de animação
    # Figura 1
    pMPX, = f1s1_ax1.plot([],[], '-g', lw=1.5, label="MPX")
    pMAP, = f1s1_ax1.plot([],[], '-r', lw=1.5, label="MAP")
    pTPS, = f1s1_ax1.plot([],[], '-k', lw=1.5, label="TPS")
    pRot, = f1s1_ax2.plot([],[], '-b', lw=1.5, label="Rot")

    pO2A, = f1s2.plot([],[], '-g', lw=1.5, label="O2A")
    pO2B, = f1s2.plot([],[], '-y', lw=1.5, label="O2B")    
    pAD01, = f1s2.plot([],[], '-r', lw=1.5, label="WB")

    # Figura 2
    pMAT, = f2s1.plot([],[], '-k', lw=1.5, label="MAT")
    pCLT, = f2s2.plot([],[], '-r', lw=1.5, label="CLT")
    pTempCpu, = f2s3.plot([],[], '-g', lw=1.5, label="Temp")
    pAD02, = f2s4.plot([],[], '-b', lw=1.5, label="AD02")    
    
    # Legenda
    f1s1_ax1.legend([pMPX, pMAP, pTPS, pRot], ['MPX', 'MAP', 'TPS', 'Rot'], 
                    bbox_to_anchor=(0., 1.01, 1., .102), loc=3, ncol=4,
                    mode="expand", borderaxespad=0.)
                
    f1s2.legend([pO2A, pO2B], ['O2A', 'O2B'], 
                bbox_to_anchor=(0., 1.01, 1., .102), loc=3, ncol=2,
                mode="expand", borderaxespad=0.)
 

    f2s1.legend(bbox_to_anchor=(0., 1.01, 1., .102), loc=8, ncol=1,
                mode="expand", borderaxespad=0.)
    f2s2.legend(bbox_to_anchor=(0., 1.01, 1., .102), loc=8, ncol=1,
                mode="expand", borderaxespad=0.)
    f2s3.legend(bbox_to_anchor=(0., 1.01, 1., .102), loc=8, ncol=1,
                mode="expand", borderaxespad=0.)
    f2s4.legend(bbox_to_anchor=(0., 1.01, 1., .102), loc=8, ncol=1,
                mode="expand", borderaxespad=0.)    
    
    # Chama a figura 1
    #plt.figure(1)
    #plt.draw()
    # set up animation
#    fig = plt.figure()
#    ax = plt.axes(xlim=(0, ringlen), ylim=(0, 1024))
#    a0, = ax.plot([], [])
#    a1, = ax.plot([], [])
    
    

    
    anim = animation.FuncAnimation(fig1, Variaveis.update,
                                   fargs=(pMPX, pMAP, pTPS, pO2A, pO2B,
                                          pMAT, pCLT, pAD01, pAD02, pRot,
                                          f1s1_ax1, f1s1_ax2, f1s2,
                                          f2s1, f2s2, f2s3, f2s4,
                                          fig1), frames=100, 
                                   interval=tempo_anim, blit=True)

    # show plot
    plt.show()
  
    # clean up
    Variaveis.close()

    print('exiting.')   

# call main
if __name__ == '__main__':
  main()
