/**
 * @file debug.c
 * @brief Debug dump function implementation.
 */
#include <stdio.h>
#include <errno.h>

#include "postgres.h"
#include "access/xlog.h"
#include "access/xlog_internal.h"

void get_segment_id(const char *filename);
void dump_record(XLogRecPtr *ptr, size_t off, XLogRecord *precord);
void dump_page_header(int num, XLogPageHeader pheader);
void dumpXLogRecord(XLogRecPtr *ptr, size_t off, XLogRecord *record);

/** @brief Current segment ID  */
static uint32 segment_id;

/** @brief List for the resource manager. */
static const char * const RM_names[RM_MAX_ID + 1] = {
	"XLOG ",					/* 0 */
	"XACT ",					/* 1 */
	"SMGR ",					/* 2 */
	"CLOG ",					/* 3 */
	"DBASE",					/* 4 */
	"TBSPC",					/* 5 */
	"MXACT",					/* 6 */
	"RM  7",					/* 7 */
	"RM  8",					/* 8 */
	"RM  9",					/* 9 */
	"HEAP ",					/* 10 */
	"BTREE",					/* 11 */
	"HASH ",					/* 12 */
	"RTREE",					/* 13 */
	"GIST ",					/* 14 */
	"SEQ  " 					/* 15 */
};

/**
 * @brief Obtain segment ID from WAL segment file name.
 * @param filename WAL segment file name.
 * @note If no slush (path delimiter) is included in the argument, or if the argument
 * does not follow WAL segment file name format, nothing will happen.
 */
void
get_segment_id(const char *filename)
{
	TimeLineID tli;
	uint32 xlogid;
	char *p;
	p = strrchr(filename, '/');
	if (!p)
		return;
	p++;
	if (sscanf(p, "%08X%08X%08X", &tli, &xlogid, &segment_id) != 3)
		return;
}

/**
 * @brief Dump the page header content.
 * @param num Page number to be included in the dump output.
 * @param page Target page.
 */
void
dump_page_header(int num, XLogPageHeader page)
{
	printf("=[%04d]==================================================\n", num);
	printf("PAGE: xlp_magic=%02X\n", page->xlp_magic);
	printf("PAGE: xlp_info=%02X\n", page->xlp_info);
	printf("PAGE: xlp_tli=%u\n", page->xlp_tli);
	printf("PAGE: xlogid=%u\n", page->xlp_pageaddr.xlogid);
	printf("PAGE: xrecoff=%u\n", page->xlp_pageaddr.xrecoff);
	if (page->xlp_info & XLP_FIRST_IS_CONTRECORD)
	{
		XLogContRecord *cont =
			(XLogContRecord *)((char *)page + XLogPageHeaderSize(page));
		printf("PAGE: rem_len=%u\n", cont->xl_rem_len);
	}
	printf("=========================================================\n");
}

/**
 * @brief Dump record header content in xlogdump format.
 * @param ptr Record position information (only log ID will be used).
 * @param off Record offset within the segment.
 * @param record Pointer to the record.
 * @note the source is copied and modified using xlogdump source.
 */
void
dumpXLogRecord(XLogRecPtr *ptr, size_t off, XLogRecord *record)
{
	static XLogRecPtr prevRecPtr = { 0, 0};

	printf("%u/%08X: prv %u/%08X",
		   ptr->xlogid, (uint32)off + segment_id * XLOG_SEG_SIZE,
		   record->xl_prev.xlogid, record->xl_prev.xrecoff);

	if (!XLByteEQ(record->xl_prev, prevRecPtr))
		printf("(?)");
	prevRecPtr.xlogid = ptr->xlogid;
	prevRecPtr.xrecoff = (uint32)off + segment_id * XLOG_SEG_SIZE;

	printf("; xid %u; ", record->xl_xid);

	if (record->xl_rmid <= RM_MAX_ID)
		printf("%s", RM_names[record->xl_rmid]);
	else
		printf("RM %2d", record->xl_rmid);

	printf(" info %02X len %u tot_len %u\n", record->xl_info,
		   record->xl_len, record->xl_tot_len);

	fflush(stdout);
}

