/**
 * @file file.c
 * @brief Common I/O routine implementation used in archive/restoration
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "postgres.h"
#include "access/xlog_internal.h"

#include "file.h"

/**
 * @brief Read the file data.  The length to read is specified by an argument.
 * @param fd File descriptor.
 * @param buff Buffer to read.
 * @param len Size to read.
 * @return Size actually read.
 * @note If error occurd, exit(2) will be called here and will not return to the
 * caller in this case.
 */
int
read_buff(int fd, char *buff, size_t len)
{
	int ret;
	size_t read_len = 0;

	do
	{
		ret = read(fd, buff + read_len, len - read_len);
		if (ret < 0)
		{
			if (errno == EINTR)
				continue;
			fprintf(stderr, "failed to read : %s\n", strerror(errno));
			exit(1);
		}
		else if (ret == 0)
			break;
		read_len += ret;
	} while (read_len < len);

	return read_len;
}

/**
 * @brief Write to the file.   The length to write should be specified by an argument.
 * @param fd File descriptor.
 * @param buff Buffer to write.
 * @param len Size to write.
 * @note If an error occurs, exit(2) will be called in this function and will not return
 * to the caller in this case.
 */
void
write_buff(int fd, const char *buff, size_t len)
{
	int ret;
	int written_len = 0;

	do
	{
		ret = write(fd, buff + written_len, len - written_len);
		if (ret < 0)
		{
			if (errno == EINTR)
				continue;
			fprintf(stderr, "failed to write : %s\n", strerror(errno));
			exit(1);
		}
		written_len += ret;
	} while (written_len < len);
}

/**
 * @brief Copy the contents of the file.
 * @param from_fd File descriptor of the file to copy from.
 * @param to_fd File descriptor of the file to copy to.
 * @note If error occurs in this function, exit(2) will be called here and will not return
 * to the caller in this case.
  */
void
copy_file(int from_fd, int to_fd)
{
	int read_len = 0;
	char buff[8 * 1024];	/* 8KB buffer */

	while (1)
	{
		/* Read to the buffer. */
		read_len = read(from_fd, buff, sizeof(buff));
		if (read_len < 0)
		{
			if (errno == EINTR)
				continue;
			fprintf(stderr, "failed to read : %s\n", strerror(errno));
			exit(1);
		}
		else if (read_len == 0)
			break;
		/* Write all the buffer content. */
		write_buff(to_fd, buff, read_len);
	}

	return;
}

/**
 * @brief Validate the record by comparing CRC value.
 *
 * CRC value will be calculated in the following order.
 *  -# Logical Log
 *  -# Full page write (if exists)
 *  -# Record header (exluding the CRC area)
 *
 * @param precord Pointer to the target record.
 * @retval true  Valid record.
 * @retval false Invalid record.
 */
bool
is_valid_record(XLogRecord *precord)
{
	pg_crc32 crc;
	BkpBlock *pblk;
	int i;

	/* Calculate CRC for a logical log. */
	INIT_CRC32(crc);
	COMP_CRC32(crc, XLogRecGetData(precord), precord->xl_len);

	/*
	 * If full page writes exist, calculate CRC for each full page write.
	 */
	pblk = (BkpBlock *)((char *)XLogRecGetData(precord) + precord->xl_len);
	for (i = 0; i < XLR_MAX_BKP_BLOCKS; i++)
	{
		uint32 blen;

		if (!(precord->xl_info & XLR_SET_BKP_BLOCK(i)))
			continue;

		if (pblk->hole_offset + pblk->hole_length > BLCKSZ)
		{
			fprintf(stderr, "incorrect hole size in record.\n");
			return false;
		}

		blen = sizeof(BkpBlock) + BLCKSZ - pblk->hole_length;
		COMP_CRC32(crc, (char *)pblk, blen);
		pblk = (BkpBlock *)((char *)pblk + blen);
	}

	/* Calculate record header CRC value. */
	COMP_CRC32(crc, (char *)precord + sizeof(pg_crc32),
		SizeOfXLogRecord - sizeof(pg_crc32));

	/* Examine if the final CRC is the same as the value found in the record. */
	FIN_CRC32(crc);
	if (!EQ_CRC32(precord->xl_crc, crc))
	{
		fprintf(stderr, "incorrect resource manager data checksum.\n");
		return false;
	}

	return true;
}
