
/* Docs say 'line' is not fully implemented.	When it is, it will most
 * likely act like a path and polygon.
 */
#include "postgres.h"
#include "catalog/pg_type.h" /* For xxxOID constants */
#include "postgres_fe.h"
#include "libpq-fe.h"
#include "libpq-int.h"
#include "utils/inet.h" /* for PGSQL_AF_INET and PGSQL_AF_INET6 */
#include "libpq/ip.h"

#define CHKARGS(r, fnum, oid, outp) \
	CHKARGS2(r, fnum, oid, InvalidOid, outp)

#define CHKARGS2(r, fnum, oid, oid2, outp) do{ \
	int ftype; \
	if(!(r)) \
	{ \
		pqSetResultError(r, "Invalid argument: PGresult pointer is NULL"); \
		return 0; \
	} \
	if(!(outp)) \
	{ \
		pqSetResultError(r, "Invalid argument: output pointer is NULL"); \
		return 0; \
	} \
	ftype = PQftype(r, fnum); \
	if((oid2) != InvalidOid) \
	{ \
		if(ftype != (oid) && ftype != (oid2)) \
		{ \
			pqSetResultError(r, "Field type mismatch"); \
			return 0; \
		} \
	} \
	else if(ftype != (oid)) \
	{ \
		pqSetResultError(r, "Field type mismatch"); \
		return 0; \
	} \
}while(0)

static int text_tofloat8(double *f8, char *text, char **endptr);
static int text_topoint(PGpoint *pt, char *text, char **endptr);
static int text_toptarray(char *valp, PGpoint **pts, int *npts);
static int bin_toptarray(char *valp, int len, PGpoint **pts, int *npts);

int PQgetchar(
	PGresult *res,
	int tup_num,
	int field_num,
	char *chp)
{
	char *valp;

	CHKARGS(res, field_num, CHAROID, chp);

	valp = PQgetvalue(res, tup_num, field_num);

	/*
	 * In text format, a single char is returned. Instead of a string
	 * like "123", you get "{". This means both binary and text
	 * format are converted in the same fashion.
	 */
	*chp = *(char *)valp;

	return 1;
}

int PQgetbool(
	PGresult *res,
	int tup_num,
	int field_num,
	int *boolp)
{
	char *valp;

	CHKARGS(res, field_num, BOOLOID, boolp);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
		*boolp = (*valp == 't') ? 1 : 0;
	else
		*boolp = *valp ? 1 : 0;

	return 1;
}

int PQgetmoney(
	PGresult *res,
	int tup_num,
	int field_num,
	double *moneyp)
{
	char *valp = PQgetvalue(res, tup_num, field_num);

	CHKARGS(res, field_num, CASHOID, moneyp);

	if(PQfformat(res, field_num) == 0)
	{
		char *p;
		char c;
		char buf[64];

		/* copy all digits "-." */
		for(p=buf; (c = *valp); valp++)
			if(isdigit(c) || c == '.' || c == '-')
				*p++ = c;

		buf[p-buf] = 0;
		if(!text_tofloat8(moneyp, buf, NULL))
		{
			pqSetResultError(res, "Money string conversion failed");
			return 0;
		}

		return 1;
	}

	/* Looks like 8.3 will use a 64-bit money type. Need compatibility
	 * for communicating with older servers.
	 */
	if(PQgetlength(res, tup_num, field_num) == 4)
		*moneyp = (int)ntohl(*(unsigned long *)valp);
	else
		pqSwapi8(moneyp, valp, 0);

	*moneyp = (*moneyp) / 100.00;
	return 0;
}

int PQgetint2(
	PGresult *res,
	int tup_num,
	int field_num,
	short *i2p)
{
	char *valp;

	CHKARGS(res, field_num, INT2OID, i2p);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		int n;

		errno = 0;
		n = (int)strtol(valp, NULL, 10);
		if(n == 0 && errno)
		{
			pqSetResultError(res, "Int2 string conversion failed");
			return 0;
		}

		*i2p = (short)n;
		return 1;
	}

	/* binary format conversion */
	*i2p = (short)ntohs(*(unsigned short *)valp);
	return 1;
}

int PQgetint4(
	PGresult *res,
	int tup_num,
	int field_num,
	int *i4p)
{
	char *valp;

	CHKARGS(res, field_num, INT4OID, i4p);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		int n;

		errno = 0;
		n = (int)strtol(valp, NULL, 10);
		if(n == 0 && errno)
		{
			pqSetResultError(res, "Int4 string conversion failed");
			return 0;
		}

		*i4p = n;
		return 1;
	}

	/* binary format conversion */
	*i4p = (int)ntohl(*(unsigned int *)valp);
	return 1;
}

int PQgetfloat4(
	PGresult *res,
	int tup_num,
	int field_num,
	float *f4p)
{
	char *valp;
	int *i = (int *)f4p;

	CHKARGS(res, field_num, FLOAT4OID, f4p);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		float f;

		errno = 0;
		f = (float)strtod(valp, NULL);
		if(f == 0 && errno)
		{
			pqSetResultError(res, "Float4 string conversion failed");
			return 0;
		}

		*(float *)f4p = f;
		return 1;
	}

	/* binary format conversion */
	*i = (int)ntohl(*(unsigned int *)valp);
	return 1;
}

int PQgetint8(
	PGresult *res,
	int tup_num,
	int field_num,
	void *i8p)
{
	char *valp;

	CHKARGS(res, field_num, INT8OID, i8p);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
#if defined(HAVE_LONG_LONG_INT_64) && defined(HAVE_STRTOLL)
		uint64 v;

		errno = 0;
		v = (uint64)strtoll(valp, NULL, 10);
		if(v == 0 && errno)
		{
			pqSetResultError(res, "Int8 string conversion failed");
			return 0;
		}

		*(uint64 *)i8p = v;

#else
		unsigned long v;

		errno = 0;
		v = (unsigned long)strtol(valp, NULL, 10);
		if(v == 0 && errno)
		{
			pqSetResultError(res, "Int8 string conversion failed");
			return 0;
		}

		*(unsigned long *)i8p = v;
#endif

		return 1;
	}

	pqSwapi8(i8p, valp, 0);
	return 1;
}

int PQgetfloat8(
	PGresult *res,
	int tup_num,
	int field_num,
	double *f8p)
{
	char *valp;

	CHKARGS(res, field_num, FLOAT8OID, f8p);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		if(!text_tofloat8(f8p, valp, NULL))
		{
			pqSetResultError(res, "Float8 string conversion failed");
			return 0;
		}

		return 1;
	}

	pqSwapi8(f8p, valp, 0);
	return 1;
}

int PQgetpoint(PGresult *res, int tup_num, int field_num, PGpoint *pt)
{
	char *valp;

	CHKARGS(res, field_num, POINTOID, pt);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		if(!text_topoint(pt, valp, NULL))
		{
			pqSetResultError(res, "Point string conversion failed");
			return 0;
		}

		return 1;
	}

	pqSwapi8(&pt->x, (unsigned int *)valp, 0);
	pqSwapi8(&pt->y, ((unsigned int *)(valp)) + 2, 0);
	return 1;
}

int PQgetlseg(PGresult *res, int tup_num, int field_num, PGlseg *lseg)
{
	char *valp;
	unsigned int *v;

	CHKARGS(res, field_num, LSEGOID, lseg);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		PGpoint *pts = (PGpoint *)lseg;

		if(*valp++ != '[' ||
			 !text_topoint(pts, valp, &valp) ||
			 *valp++ != ',' ||
			 !text_topoint(pts + 1, valp, &valp) ||
			 *valp != ']')
		{
			pqSetResultError(res, "lseg string conversion failed");
			return 0;
		}

		return 1;
	}

	v = (unsigned int *)valp;
	pqSwapi8(&lseg->pts[0].x, v, 0);
	pqSwapi8(&lseg->pts[0].y, v + 2, 0);
	pqSwapi8(&lseg->pts[1].x, v + 4, 0);
	pqSwapi8(&lseg->pts[1].y, v + 6, 0);
	return 1;
}

int PQgetbox(PGresult *res, int tup_num, int field_num, PGbox *box)
{
	char *valp;
	unsigned int *v;

	CHKARGS(res, field_num, BOXOID, box);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		PGpoint *pts = (PGpoint *)box;

		if(!text_topoint(pts, valp, &valp) ||
			 *valp++ != ',' ||
			 !text_topoint(pts + 1, valp, NULL))
		{
			pqSetResultError(res, "Box string conversion failed");
			return 0;
		}

		return 1;
	}

	v = (unsigned int *)valp;

	pqSwapi8(&box->high.x, v,	 0);
	pqSwapi8(&box->high.y, v + 2, 0);
	pqSwapi8(&box->low.x, v + 4, 0);
	pqSwapi8(&box->low.y, v + 6, 0);
	return 1;
}

int PQgetcircle(
	PGresult *res,
	int tup_num,
	int field_num,
	PGcircle *circle)
{
	char *valp;
	unsigned int *v;

	CHKARGS(res, field_num, CIRCLEOID, circle);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		if(*valp++ != '<' ||
			 !text_topoint((PGpoint *)circle, valp, &valp) ||
			 *valp++ != ',' ||
			 !text_tofloat8(&circle->radius, valp, &valp) ||
			 *valp != '>')
		{
			pqSetResultError(res, "Circle string conversion failed");
			return 0;
		}

		return 1;
	}

	v = (unsigned int *)valp;
	pqSwapi8(&circle->center.x, v, 0);
	pqSwapi8(&circle->center.y, v + 2, 0);
	pqSwapi8(&circle->radius, v + 4, 0);
	return 1;
}

int PQgetpath(PGresult *res, int tup_num, int field_num, PGpath *path)
{
	char *valp;

	CHKARGS(res, field_num, PATHOID, path);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		path->closed = *valp == '(' ? 1 : 0;
		return text_toptarray(valp, &path->pts, &path->npts);
	}

	path->closed = *valp ? 1 : 0;
	valp++;
	return bin_toptarray(
		valp + sizeof(int),
		(int)(ntohl(*(int *)valp) * sizeof(PGpoint)),
		&path->pts,
		&path->npts);
}

int PQgetpolygon(
	PGresult *res,
	int tup_num,
	int field_num,
	PGpolygon *polygon)
{
	char *valp;

	CHKARGS(res, field_num, POLYGONOID, polygon);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
		return text_toptarray(valp, &polygon->pts, &polygon->npts);

	return bin_toptarray(
		valp + sizeof(int),
		(int)(ntohl(*(int *)valp) * sizeof(PGpoint)),
		&polygon->pts,
		&polygon->npts);
}

int PQgetinet(PGresult *res, int tup_num, int field_num, PGinet *inet)
{
	char *valp;
	unsigned short family;

	CHKARGS2(res, field_num, INETOID, CIDROID, inet);

	valp = PQgetvalue(res, tup_num, field_num);
	memset(inet, 0, sizeof(PGinet));

	if(PQfformat(res, field_num) == 0)
	{
		char *p;
		char ipstr[80];
		struct addrinfo *ai = NULL;
		struct addrinfo hints;

		strcpy(ipstr, valp);
		if((p = strrchr(ipstr, '/')))
		{
			*p = 0;
			inet->mask = atoi(p+1);
		}
		else
		{
			inet->mask = 32;
		}

		inet->is_cidr = PQftype(res, field_num)==CIDROID ? 1 : 0;

		/* suppress hostname lookups */
		memset(&hints, 0, sizeof(hints));
		hints.ai_flags = AI_NUMERICHOST;

		if(pg_getaddrinfo_all(ipstr, NULL, &hints, &ai) || !ai)
		{
			pqSetResultError(res, "inet string conversion failed");
			return 0;
		}

		inet->sa_family = (unsigned short)ai->ai_family;
		inet->sa_len = (int)ai->ai_addrlen;
		memcpy(inet->sa_buf, ai->ai_addr, inet->sa_len);
		pg_freeaddrinfo_all(ai->ai_family, ai);
		return 1;
	}

	family = (unsigned short)*valp++;
	if(family == PGSQL_AF_INET)
	{
		struct sockaddr_in *sa = (struct sockaddr_in *)inet->sa_buf;
		inet->sa_family = sa->sin_family = AF_INET;
		inet->mask = (unsigned char)*valp++;
		inet->is_cidr = *valp++;
		memcpy(&sa->sin_addr, valp + 1, *valp);
		inet->sa_len = (int)sizeof(struct sockaddr_in);
	}
	else if(family == PGSQL_AF_INET6)
	{
		struct sockaddr_in6 *sa = (struct sockaddr_in6 *)inet->sa_buf;
		inet->sa_family = sa->sin6_family = AF_INET6;
		inet->mask = (unsigned char)*valp++;
		inet->is_cidr = *valp++;
		memcpy(&sa->sin6_addr, valp + 1, *valp);
		inet->sa_len = (int)sizeof(struct sockaddr_in6);
	}
	else
	{
		pqSetResultError(res, "Invalid binary representation of inet: unknown family");
		return 0;
	}

	return 1;
}

int PQgetmacaddr(PGresult *res, int tup_num, int field_num, PGmacaddr *mac)
{
	char *valp;
	unsigned char *p;

	CHKARGS(res, field_num, MACADDROID, mac);

	valp = PQgetvalue(res, tup_num, field_num);

	if(PQfformat(res, field_num) == 0)
	{
		int a,b,c,d,e,f;
		int count = sscanf(valp, "%x:%x:%x:%x:%x:%x", &a,&b,&c,&d,&e,&f);

		if(count != 6 || (a < 0) || (a > 255) || (b < 0) || (b > 255) ||
			(c < 0) || (c > 255) || (d < 0) || (d > 255) ||
			(e < 0) || (e > 255) || (f < 0) || (f > 255))
		{
			pqSetResultError(res, "macaddr string conversion failed");
			return 0;
		}

		mac->a = a;
		mac->b = b;
		mac->c = c;
		mac->d = d;
		mac->e = e;
		mac->f = f;
		return 1;
	}

	p = (unsigned char *)valp;
	mac->a = *p++;
	mac->b = *p++;
	mac->c = *p++;
	mac->d = *p++;
	mac->e = *p++;
	mac->f = *p;
	return 1;
}

static int text_toptarray(char *valp, PGpoint **pts, int *npts)
{
	char *s;
	int cnt = 0;
	PGpoint *p = NULL;

	*pts = NULL;
	*npts = 0;

	if(*valp != '(' && *valp != '[')
		return 0;

	/* get the number of points by counting the '(' */
	for(s=valp+1; *s; s++)
	{
		if(*s == '(')
		{
			if(!(s = strchr(s, ')'))) /* skip point contents */
				break;
			cnt++;
		}
	}

	if(cnt == 0)
		return 1; /* empty point list */

	if(!(p = (PGpoint *)malloc(cnt * sizeof(PGpoint))))
		return 0;

	for(cnt=0; *++valp; )
	{
		if(!text_topoint(&p[cnt++], valp, &valp))
		{
			free(p);
			return 0;
		}

		/* done */
		if(*valp != ',')
			break;
	}

	*pts = p;
	*npts = cnt;
	return 1;
}

static int bin_toptarray(char *valp, int len, PGpoint **pts, int *npts)
{
	int i;
	PGpoint *p;
	unsigned int *v = (unsigned int *)valp;
	int cnt = len / (int)sizeof(PGpoint);

	*pts = NULL;
	*npts = 0;

	if(cnt == 0)
		return 1;

	if(!(p = (PGpoint *)malloc(len)))
		return 0;

	for(i=0; i < cnt; i++, v+=4)
	{
		pqSwapi8(&p[i].x, v, 0);
		pqSwapi8(&p[i].y, v + 2, 0);
	}

	*pts = p;
	*npts = cnt;
	return 1;
}

static int text_tofloat8(double *f8, char *text, char **endptr)
{
	double d;

	errno = 0;
	d = strtod(text, endptr);
	if(d == 0 && errno)
		return 0;

	*f8 = d;
	return 1;
}

static int text_topoint(PGpoint *pt, char *text, char **endptr)
{
	if(*text++ != '(')
		return 0;

	if(!text_tofloat8(&pt->x, text, &text) || *text++ != ',')
		return 0;

	if(!text_tofloat8(&pt->y, text, &text) || *text++ != ')')
		return 0;

	if(endptr)
		*endptr = text;
	return 1;
}



