/**
 * @file pg_compresslog.c
 * @brief pg_compresslog implementation
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "postgres.h"
#include "access/xlog.h"
#include "access/xlog_internal.h"
#include "catalog/pg_control.h"

#include "file.h"
#include "debug.h"

/* =============================================================================
 * Global variables
 * ===========================================================================*/

/** @brief Buffer to read WAL segment file. */
static char xlog_buff[XLogSegSize];
/** @brief Buffer to hold archive log file image with physical log removed. */
static char arch_buff[XLogSegSize];

static int cont_log_size;		/**< @brief Log size considering the former segment. */
static int logical_log_size;	/**< @brief Total size of the logical log. */
static int physical_log_size;	/**< @brief Total size of the physical log. */

/* =============================================================================
 * Prototype declaration
 * ===========================================================================*/
static void print_usage(int code);
static int open_xlog_file(int argc, char *argv[]);
static int open_arch_file(int argc, char *argv[]);
int create_arch_image(const char *from, char *to);
static bool remove_bkp_block(XLogRecord *record);

/* =============================================================================
 * Macros
 * ===========================================================================*/
/** @brief Check if the physical log can be removed. */
#define IS_REMOVABLE(record) \
	(((record)->xl_info & XLR_BKP_BLOCK_MASK) && \
	((record)->xl_info & XLR_BKP_REMOVABLE))

/* =============================================================================
 * Function definitions
 * ===========================================================================*/
/**
 * @brief Entry point of pg_compresslog command.
 * @param argc Number of arguments
 * @param argv Array of the pointer to argument character string.
 * @retval  0 No error
 * @retval !0 Error
 */
int
main(int argc, char *argv[])
{
	int from_fd = -1;
	int to_fd = -1;
	size_t xlog_len;
	size_t arch_len;

	/** Error if there are more argument(s) other than from and to. */
	if (argc > 3)
		print_usage(1);

	/** Open WAL segment file to archive. */
	from_fd = open_xlog_file(argc, argv);

	/**
	 * If input file is not stdin, check the size of the file.
	 * If the size is not 16MB, then the specified file is not the WAL
	 * segment file.   We're not sure if we can scan the file to find
	 * removable physical log, copy the whole file and then exits.
	 */
	if (from_fd != fileno(stdin))
	{
		struct stat st;

		if (fstat(from_fd, &st) < 0)
		{
			fprintf(stderr, "failed to stat `%s': %s\n", argv[1],
				strerror(errno));
			exit(1);
		}
		if (st.st_size != XLogSegSize)
		{
			to_fd = open_arch_file(argc, argv);
			copy_file(from_fd, to_fd);
			exit(0);
		}
	}

	/**
	 * Read all the data from WAL segment file to archive.
	 * If the amount of the data is not sufficient (less than 16MB: XLogSegSize)
	 * or header is not valid, specified input file is not a WAL segment file.  
	 * Copy the whole input to the output and then exit.
	 */
	xlog_len = read_buff(from_fd, xlog_buff, XLogSegSize);
	if (xlog_len != XLogSegSize || !IS_WAL_FILE(xlog_buff))
	{
		/* Write the checked header part and then copy the rest of the input file. */
		to_fd = open_arch_file(argc, argv);
		write_buff(to_fd, xlog_buff, xlog_len);
		copy_file(from_fd, to_fd);
		if (close(from_fd) < 0)
		{
			fprintf(stderr, "failed to close `%s': %s\n", argv[1],
				strerror(errno));
			exit(1);
		}
		exit(0);
	}
	if (close(from_fd) < 0)
	{
		fprintf(stderr, "failed to close `%s': %s\n", argv[1], strerror(errno));
		exit(1);
	}

	/**
	 * Build the entire compressed output file image on the buffer,
	 * removing physical logs, then write the whole compressed file image.
	 */
	arch_len = create_arch_image(xlog_buff, arch_buff); 
	to_fd = open_arch_file(argc, argv);
	write_buff(to_fd, arch_buff, arch_len);
	if (close(to_fd) < 0)
	{
		fprintf(stderr, "failed to close `%s': %s\n", argv[2], strerror(errno));
		exit(1);
	}

	exit(0);
}

/**
 * @brief Show the usage of the command and then exits.
 * @param code exit code.
 */
static void
print_usage(int code)
{
	printf(
		"usage: pg_compresslog [from [to]]\n"
		"    from - Input file name (stdin if omitted or '-' is given)\n"
		"    to   - Output file name (stdiout if omitted or '-' is given)\n"
	);
	exit(code);
}

/**
 * @brief Build archive log file image of 16MB, from WAL segment buffer image of
 * 8kB pages.
 * @param from WAL segment buffer page
 * @param to Archive log file image
 * @return Archive log file size.
 * @note exit() will be called here when a error is detected.  In the case of
 * error, the control will not be given to the caller.
 */
int
create_arch_image(const char *from, char *to)
{
	const char *read_pos = from;
	char *write_pos = to;
	const char *crrpage = from;
	XLogPageHeader page = (XLogPageHeader)from;
	XLogRecord *rec = NULL;
	XLogRecord *write_rec = NULL;

	/*
	 * Copy the first page header of the segment to the buffer.
	 * If the record is the successor of the last record of the former segment,
	 * then copies XLogContRecord too.
	 * XLogContRecord.xl_rem_len means the total data length of the
	 * continuation record, not the length of the record in the given page.
	 * Therefore, this value is not influenced by the change of the page size.
	 */
	read_pos = crrpage = from;
	memcpy(write_pos, read_pos, XLogPageHeaderSize(page));
	write_pos += XLogPageHeaderSize(page);
	read_pos += XLogPageHeaderSize(page);
	if (page->xlp_info & XLP_FIRST_IS_CONTRECORD)
	{
		memcpy(write_pos, read_pos, SizeOfXLogContRecord);
		write_pos += SizeOfXLogContRecord;
	}

	/*
	 * Loop page by page.
	 */
	for (crrpage = from; crrpage < from + XLogSegSize; crrpage += XLOG_BLCKSZ)
	{
		XLogRecPtr ptr;

		/* Parse the page header. */
		page = (XLogPageHeader)crrpage;
		read_pos = crrpage + XLogPageHeaderSize(page);
		ptr = page->xlp_pageaddr;

		/* If there is a continuous data, copy them to the write buffer. */
		if (page->xlp_info & XLP_FIRST_IS_CONTRECORD)
		{
			int cont_len = ((XLogContRecord *)read_pos)->xl_rem_len;
			int copy_len = cont_len;
			int free_len = XLOG_BLCKSZ -
				(read_pos + SizeOfXLogContRecord - crrpage);
			/*
			 * Copy the continuous data within this page.
			 * xl_rem_len specifies the length of the continuous data after this page,
			 * so this may be larger than the length of the rest of this page.
			 */
			if (copy_len > free_len)
				copy_len = free_len;
			memcpy(write_pos, read_pos + SizeOfXLogContRecord, copy_len);
			read_pos += MAXALIGN(SizeOfXLogContRecord + copy_len);
			write_pos += copy_len;
			if (!rec)
				cont_log_size += copy_len;

			/*
			 * If the data continues to the next page and no record header
			 * exists in this file, then switch to the next page.
			 */
			if (cont_len != copy_len)
				continue;

			/*
			 * Set the write position to the end of the current record, 
			 * considering alignment.
			 */
			write_pos = to + MAXALIGN(write_pos - to);

			/*
			 * If the record should have a header in this segment (not a continuous
			 * record from the last segment), perform CRC check and check if
			 * physical log record can be removed.
			 */
			if (write_rec)
			{
				/* Check if the record is valid. */
				if (!is_valid_record(write_rec))
					exit(1);

				/*
				 * Determine if the physical log can be removed.
				 * If it can be removed, then rewind the position for the next log record
				 * to the position of the physical log (plus padding).
				 */
				if (remove_bkp_block(write_rec))
					write_pos = (char *)write_rec +
						MAXALIGN(SizeOfXLogRecord + rec->xl_len);
			}
		}

		/* Read the data within the page record by record. */
		while(read_pos <= crrpage + XLOG_BLCKSZ - SizeOfXLogRecord)
		{
			int freespace = XLOG_BLCKSZ - (read_pos - crrpage);

			/* Obtain the record header info. */
			rec = (XLogRecord *)read_pos;
			write_rec = (XLogRecord *)write_pos;
			logical_log_size += rec->xl_len;
			physical_log_size +=
				rec->xl_tot_len - (SizeOfXLogRecord + rec->xl_len);
			dumpXLogRecord(&ptr, read_pos - from, rec);

			/*
			 * If the record continues to the following pages, copy only the portion
			 * in this page and then switch to the next page.
			 */
			if (rec->xl_tot_len > freespace)
			{
				/* Copy the log data only in the current page. */
				memcpy(write_pos, read_pos, freespace);
				/* read_pos will be overwritten at the next loop.  We don't need to update this here. */
				write_pos += freespace;
				break;
			}

			/* Copy the record data to the archive buffer.  */
			memcpy(write_pos, read_pos, rec->xl_tot_len);
			read_pos += MAXALIGN(rec->xl_tot_len);
			write_pos += MAXALIGN(rec->xl_tot_len);

			/* Check if the record is valid using CRC in the record header.  */
			if (!is_valid_record(write_rec))
				exit(1);

			/*
			 *  Log record other than log switch must have it's logical data.
			 * See the comment around the line 3065 of src/backend/access/transam/xlog.c
			 * (8.2.0).a
			 */
			if (IS_XLOG_SWITCH(write_rec))
			{
				if (write_rec->xl_len != 0)
				{
					fprintf(stderr, "invalid xlog switch record.\n");
					exit(1);
				}
			}
			else if (write_rec->xl_len == 0)
			{
				fprintf(stderr, "invalid record length.\n");
				exit(1);
			}

			/* If the log record is the log switch record, then no more log record exists
			 * in * the input file.   Exit.
			 */
			if (IS_XLOG_SWITCH(write_rec))
				return write_pos - to;

			/*
			 * If the physical log is removable, then rewind the position of the next
			 * record to
			 * the physical log start position (and padding).
			 */
			if (remove_bkp_block(write_rec))
				write_pos = (char *)write_rec +
					MAXALIGN(SizeOfXLogRecord + write_rec->xl_len);
			else
				write_pos = to + MAXALIGN(write_pos - to);

		}
	}

	return (write_pos - to);
}

/**
 * @brief Remove the physical log.
 * @param record WAL record
 * @return true of the physical record has been removed.  False if not has been removed.
 */
static bool
remove_bkp_block(XLogRecord *record)
{
	pg_crc32 crc;

	/* If no record is specified or the physical log is not removable, just return. */
	if (!record || !IS_REMOVABLE(record))
		return false;

	/*
	 * Reset XLR_BKP_BLOCK_MASK. 
	 * We need the flag to show the physical log is removable to restore removed physical
	 * log with a dummy.  It is not reset.
	 */
	record->xl_info &= ~XLR_BKP_BLOCK_MASK;

	/*
	 * Record contents changes by physical log removal and CRC has to be recalculated.
	 * CRC will be accumulated as follows:
	 * 1. Logical log
	 * 2. Physical log (It has ben removed and we don't calculate its CRC here).
	 * 3. WAL record header excluding CRC part
	 * Please refer to the line 2817 of RecordIsValid(), src/backend/access/transam/xlog.c.
	 */
	INIT_CRC32(crc);
	COMP_CRC32(crc, XLogRecGetData(record), record->xl_len);
	COMP_CRC32(crc, (char *)record + sizeof(pg_crc32),
		SizeOfXLogRecord - sizeof(pg_crc32));
	FIN_CRC32(crc);
	record->xl_crc = crc;

	return true;
}

/**
 * @brief Open the WAL segment file to archive.
 * 
 * The first argument will be regarded as an input file.   If omitted or specified as
 * "-", stdin will be used as an input file.
 * @param argc Number of arguments (argument to main() will be passed as is.
 * @param argv Array of pointers to argument strings (argument to main() will be passed
 * as is.
 * @return File descriptof for WAL segment file.
 * @note exit() will be called here if error occurs.  Will not return to the caller
 * in this case.
 */
static int
open_xlog_file(int argc, char *argv[])
{
	int from_fd = -1;

	if (argc > 1 && strcmp(argv[1], "-") != 0)
	{
		/* Open WAL segment file to archive. */
		from_fd = open(argv[1], O_RDONLY, 0);
		if (from_fd < 0)
		{
			fprintf(stderr, "failed to open `%s': %s\n", argv[1],
				strerror(errno));
			exit(1);
		}

		/* Obtain segment ID from the file name (for record dump). */
		get_segment_id(argv[1]);
	}
	else
		from_fd = fileno(stdin);

	return from_fd;
}

/**
 * @brief Open the archive segment file to write the result.
 * 
 * 2番目の引数を出力ファイル名として扱う. 省略時, または"-"が指定された場合は
 * 標準出力を用いる.
 * @param argc pass the same argument to main() as is.
 * @return File descriptor for archive log file.
 * @note When an error occurs within this function, exit() will be called here and
 * will not return to the caller in this case.
 */
static int
open_arch_file(int argc, char *argv[])
{
	int to_fd = -1;

	if (argc > 2 && strcmp(argv[2], "-") != 0)
	{
		/* Open the archive log file. */
		to_fd = open(argv[2], O_RDWR | O_CREAT | O_EXCL | PG_BINARY,
			S_IRUSR | S_IWUSR);
		if (to_fd < 0)
		{
			fprintf(stderr, "failed to open `%s': %s\n", argv[2],
				strerror(errno));
			exit(1);
		}
	}
	else
		to_fd = fileno(stdout);

	return to_fd;
}
