/**
 * @file pg_decompresslog.c
 * @brief Implementation of the archive restore command (pg_decompresslog).
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "postgres.h"
#include "access/xlog.h"
#include "access/xlog_internal.h"
#include "catalog/pg_control.h"

#include "file.h"
#include "debug.h"

/* =============================================================================
 * Global variables
 * ===========================================================================*/

/** @brief Buffer to hold restored WAL segment file image. */
static char xlog_buff[XLogSegSize];
/** @brief Buffer to read an archive log file. */
static char arch_buff[XLogSegSize];
/** @brief Position to write a record data. */
static char *write_pos = xlog_buff;
/** @brief Position to read record data in the archive log buffer. */
static char *read_pos = arch_buff;
/** @brief This holds data in the first page header of the segment. */
static XLogPageHeaderData baseheader;

/* =============================================================================
 * Prototype declaration
 * ===========================================================================*/
static void print_usage(int code);
int create_wal_image(int arch_len);
static int write_record(char *record_buff, int rem_len, bool isFromPrevSeg);
static int get_freespace(void);
static void insert_XLogContRecord(char *write_pos, int rem_len);
static void insert_pageheader(char *write_pos, XLogPageHeader pheader,
	bool hasContRecord);
static int open_arch_file(int argc, char *argv[]);
static int open_xlog_file(int argc, char *argv[]);

/* =============================================================================
 * Function definitions
 * ===========================================================================*/

/**
 * @brief Entry point of pg_decompresslog command.
 * @param argc Number of the arguments.
 * @param argv Array of the pointers of the command arguments.
 * @retval  0 No error.
 * @retval !0 Error.
 */
int
main(int argc, char *argv[])
{
	int from_fd = -1;
	int to_fd = -1;
	size_t arch_len;

	/* Error if argument(s) other than <from>, and <to> are given.  */
	if (argc > 3)
		print_usage(1);

	/** Open the archive log file to restore. */
	from_fd = open_arch_file(argc, argv);

	/**
	 * Read all the data in the input archive log file.
	 * If the header at the first page is not valid, it is not a WAL segment file
	 * and then copy the whole input file to the output file.
	 */
	arch_len = read_buff(from_fd, arch_buff, XLogSegSize);
	if (!IS_WAL_FILE(arch_buff))
	{
		/* Write what is read for header validation check and then copy the rest of the input file. */
		to_fd = open_xlog_file(argc, argv);
		write_buff(to_fd, arch_buff, arch_len);
		copy_file(from_fd, to_fd);
		if (close(from_fd) < 0)
		{
			fprintf(stderr, "failed to close `%s': %s\n", argv[1],
				strerror(errno));
			exit(1);
		}
		exit(0);
	}
	if (close(from_fd) < 0)
	{
		fprintf(stderr, "failed to close `%s': %s\n", argv[1], strerror(errno));
		exit(1);
	}

	/**
	 * Build the restored WAL segment file image.
	 * Write all the restored WAL segment file image.
	 */
	if (create_wal_image((int)arch_len))
	{
		fprintf(stderr, "failed to create the image of `%s'\n", argv[1]);
		exit(1);
	}
	to_fd = open_xlog_file(argc, argv);
	write_buff(to_fd, xlog_buff, XLogSegSize);
	if (close(to_fd) < 0)
	{
		fprintf(stderr, "failed to close `%s': %s\n", argv[2], strerror(errno));
		exit(1);
	}
	
	exit(0);
}

/**
 * @brief Show the usage of the command and exit.
 * @param code return code.
 */
static void
print_usage(int code)
{
	printf(
		"usage: pg_decompresslog [from [to]]\n"
		"    from - Iput file name (stdin if omitted or specified as '-')\n"
		"    to   - Output file name (stdout if omitted or specified as '-')\n"
	);
	exit(code);
}

/**
 * @brief Restore 8KB page WAl segment file image from 16MB page archive log build by
 * pg_compresslog command.
 * @param arch_len Size of the archive log file.
 * @retval 0  No error
 * @retval -1 Error
 */
int
create_wal_image(int arch_len)
{
	/** @brief Buffer holding one record data. */
	static char record_buff[XLogSegSize];
	XLogPageHeader pheader;
	XLogContRecord *pcontrec = NULL;
	XLogRecord *precord = NULL;
	char *rec_write_pos = record_buff;
	int rec_len = 0;
	bool isFromPrevSeg = false;

	/*
	 * Copy the archive log file page header and hold info in the header.  They are
	 * used to restore page headers of WAL segment file.
	 */
	pheader = (XLogPageHeader)arch_buff;
	if (XLogPageHeaderSize(pheader) != SizeOfXLogLongPHD)
	{
		fprintf(stderr, "invalid pageheader size.\n");
		return -1;
	}
	memcpy(write_pos, (char *)pheader, SizeOfXLogLongPHD);
	read_pos += SizeOfXLogLongPHD;
	write_pos += SizeOfXLogLongPHD;

	baseheader.xlp_magic = pheader->xlp_magic;
	baseheader.xlp_info &= ~XLP_ALL_FLAGS;
	baseheader.xlp_tli = pheader->xlp_tli;
	baseheader.xlp_pageaddr.xlogid = pheader->xlp_pageaddr.xlogid;
	baseheader.xlp_pageaddr.xrecoff = pheader->xlp_pageaddr.xrecoff;

	/*
	 * Copy XLogContRecord and the continuous record to the record buffer, if there is
	 * a continuous record from the last segment file.  Then move them to WAL segment
	 * file image buffer.
	 */
	if (pheader->xlp_info & XLP_FIRST_IS_CONTRECORD)
	{
		pcontrec = (XLogContRecord *)read_pos;

		/* If the size of the continue record is not valid, it's an error. */
		if (pcontrec->xl_rem_len == 0)
		{
			printf("invalid continue record length : xl_rem_len = %u\n",
													pcontrec->xl_rem_len);
			return -1;
		}

		memcpy(write_pos, read_pos, SizeOfXLogContRecord);
		write_pos += SizeOfXLogContRecord;
		rec_len = pcontrec->xl_rem_len;
		memcpy(rec_write_pos, (read_pos + SizeOfXLogContRecord), rec_len);
		read_pos += MAXALIGN(SizeOfXLogContRecord + rec_len);
		isFromPrevSeg = true;

		/* Write the continuous data to WAL segment file image buffer. */
		if (write_record(record_buff, rec_len, isFromPrevSeg))
			return 0;
	}
	isFromPrevSeg = false;
	dump_page_header(((write_pos - xlog_buff) / XLOG_BLCKSZ), 
												(XLogPageHeader)xlog_buff);

	/**
	 * Loop record by record, and build each record image in the record buffer.
	 */
	while ((read_pos - arch_buff) < arch_len)
	{
		rec_write_pos = record_buff; /* レコードバッファの書き出し位置初期化 */
		precord = (XLogRecord *)read_pos;

		/* 
		 * If the record data fits in the current segment, validate the record.
		 * If WAL record cotinues to the next segment, we cannot calculate CRC for
		 * the whole record and skip the validation.
		 */
		if ((char *)precord - arch_buff + precord->xl_tot_len <= arch_len)
			if (!is_valid_record(precord))
				exit(1);

		/*
		 * Record other than the log switch must have corresponding logical data.
		 * Refer to the comment around the line 3056 in src/backend/access/transam/xlog.c (8.2.0).
		 */
		if (IS_XLOG_SWITCH(precord))
		{
			if (precord->xl_len != 0)
			{
				fprintf(stderr, "invalid xlog switch record.\n");
				exit(1);
			}
		}
		else if (precord->xl_len == 0)
		{
			fprintf(stderr, "invalid record length.\n");
			exit(1);
		}

		/*
		 * Copy the record header and the logical log to the record buffer.
		 * We don't move read_pos here to cauculate the alignment considering
		 * physical log.
		 */
		memcpy(rec_write_pos, read_pos, (SizeOfXLogRecord + precord->xl_len));
		rec_write_pos += (SizeOfXLogRecord + precord->xl_len);
		rec_len = precord->xl_tot_len;
		
		/* Copy the physical log, or restore it. */
		if (precord->xl_tot_len > SizeOfXLogRecord + precord->xl_len)
		{
			/*
			 * If physical log does exist (not removed), then simply copy it.
			 * If physical log is removed, then build a dummy.
			 */
			if (precord->xl_info & XLR_BKP_BLOCK_MASK)
			{
				memcpy(rec_write_pos, 
					XLogRecGetData((XLogRecord *)read_pos) + precord->xl_len,
					precord->xl_tot_len - (SizeOfXLogRecord + precord->xl_len));
				read_pos += MAXALIGN(precord->xl_tot_len);
			}
			else
			{
				/*
				 * Because full page write flag will be omitted during the archiving, CRC check
				 * should be performed only against the record header and the logical log.
				 * Therefore, we don't have to recalculate CRC value here.
				 */
				memset(rec_write_pos, '\0', 
					precord->xl_tot_len - (SizeOfXLogRecord + precord->xl_len));
				read_pos += MAXALIGN(SizeOfXLogRecord + precord->xl_len);
			}
		}
		else
		{
			/* If theres not physical log in the original log, simply updates the input position. */
			read_pos += MAXALIGN(precord->xl_tot_len);
		}

		/**
		 * Write a record image to the WAL segment image buffer.  Hey, the page size
		 * is again 8kB as the original WAL.
		 */
		if (write_record(record_buff, rec_len, isFromPrevSeg))
			break;

	}

	return 0;
}

/** 
 * @brief Build a recor image to fit in 8kB page to WAL segment image buffer.
 * @retval 0 One record data build complete.
 * @retval 1 Hit the tail of the segment.
 */
static int
write_record(char *record_buff, int rem_len, bool isFromPrevSeg)
{
	char *phead_pos = NULL;
	char *rec_read_pos = record_buff; /* Read position in the record buffer. */
	char *rec_head_pos = NULL;
	int freespace; /* Size of the free space in the page. */
	bool hasContRecord = isFromPrevSeg;

	/*
	 * Hold the position of the record header (or XLogContRecord).
	 * It is needed for an alignment.
	 */
	rec_head_pos = write_pos; /* Set the write position of the record data. */
	if (isFromPrevSeg)
		rec_head_pos -= SizeOfXLogContRecord;

	freespace = get_freespace();

 	/*
 	 * If free space size is the same as the page size, it means that the last record restoration
 	 * filled the last page.   So we add page header here.
 	 * Because the record is complete in the last page, we don't need XLogContRecord.
	 * Page header at the top of the segment must have had written at the first call of this function.
	 * So we always add short format header here.
 	 */
 	if (freespace == XLOG_BLCKSZ)
 	{
 		phead_pos = write_pos;
 		insert_pageheader(write_pos, &baseheader, false);
 		write_pos += SizeOfXLogShortPHD;
 		freespace = (XLOG_BLCKSZ - SizeOfXLogShortPHD);
 		rec_head_pos = write_pos;
 		dump_page_header(((write_pos - xlog_buff) / XLOG_BLCKSZ),
 											(XLogPageHeader)phead_pos);
 	}

	/*
	 * Loop page by page.
	 */
	while(1)
	{
		/*
		 * If the record header does not fit the page, then insert a page header to the next
		 * page and copy the record data.
		 */
		if (!hasContRecord && freespace < SizeOfXLogRecord)
		{
			write_pos += freespace;
		}
		else if (freespace < rem_len)
		{
			/**
			 * If the record data does not fit the page, fill this page with the former
			 * part of the record, copy the rest to the next page, insert a page header
			 * and XLogContRecord.
			 * the next page.
			 */
			memcpy(write_pos, rec_read_pos, freespace);
			if (!hasContRecord)
				dumpXLogRecord(&baseheader.xlp_pageaddr,
					 			(size_t)(rec_head_pos - xlog_buff),
												(XLogRecord *)rec_head_pos);
			write_pos += freespace;
			rec_read_pos += freespace;
			rem_len -= freespace;
			hasContRecord = true;
		}
		else
		{
			/**
			 * If th recor data fits to the page, copy the whole record data to the
			 * buffer and switch to the next record.
			 */
			int len;
			memcpy(write_pos, rec_read_pos, rem_len);
			if (!hasContRecord)
				dumpXLogRecord(&baseheader.xlp_pageaddr,
					 			(size_t)(rec_head_pos - xlog_buff),
												(XLogRecord *)rec_head_pos);

			/*
			 * Alignment handling.
			 * Alignment has to be adjusted for each record.
			 */
			if (hasContRecord)
				len = MAXALIGN(SizeOfXLogContRecord + rem_len);
			else
				len = MAXALIGN(rem_len);
			write_pos = rec_head_pos + len;
			hasContRecord = false;

			break;
		}

		/*
		 * Insert a page header.
		 * If the start of the page is a continuous data from the last page, 
		 * insert XLogContRecor too.
		 */
		if ((write_pos - xlog_buff) >= XLogSegSize)
			return 1;
		phead_pos = write_pos;
		insert_pageheader(write_pos, &baseheader, hasContRecord);
		write_pos += SizeOfXLogShortPHD;
		freespace = (XLOG_BLCKSZ - SizeOfXLogShortPHD);
		rec_head_pos = write_pos;
		if (hasContRecord)
		{
			insert_XLogContRecord(write_pos, rem_len);
			write_pos += SizeOfXLogContRecord;
			freespace -= SizeOfXLogContRecord;
		}
		dump_page_header(((write_pos - xlog_buff) / XLOG_BLCKSZ),
											(XLogPageHeader)phead_pos);
	}
	return 0;
}

/**
 * @brief Calculate free space size of the page.
 * @retval Free space size of the page.
 */
static int
get_freespace(void)
{
	return XLOG_BLCKSZ - (write_pos - xlog_buff) % XLOG_BLCKSZ;
}
/**
 * @brief Insert a XLogContRecord to the buffer.
 * @param write_pos Write position in the buffer.
 * @param rem_len   Length of the remaining record which continues from the last page.
 */
static void
insert_XLogContRecord(char *write_pos, int rem_len)
{
	XLogContRecord contrec;

	contrec.xl_rem_len = rem_len;
	memcpy(write_pos, (char *)&contrec, SizeOfXLogContRecord);
}

/**
 * @brief Insert a page header to the buffer.
 * @param write_pos Write position in the buffer.
 * @param pheader Pointer to the structure holding header info at the firt page of the segment.
 * @param hasContRecord Flag to indicate a continuous record from the last page.
 */
static void
insert_pageheader(char *write_pos, XLogPageHeader pheader, bool hasContRecord)
{
	/**
	 * Each page header is restored using the page header at the first page of
	 * the WAL segment.   Magic number (xlp_magic), timeline id (xlp_tli) and XLOGID
	 * (xlogid) should no change within a segment and they are copied from the first
	 * page header.  Continuous data (xlp_info) depends on the record of a given page.
	 * xrecoff is calculated by adding XLOG_BLKSZ to xrecoff value in the first
	 * page header.
	 */
	pheader->xlp_info &= ~XLP_ALL_FLAGS;
	if (hasContRecord)
		pheader->xlp_info |= XLP_FIRST_IS_CONTRECORD;
	pheader->xlp_pageaddr.xrecoff += XLOG_BLCKSZ;
	memcpy(write_pos, (char *)pheader, SizeOfXLogShortPHD);
}

/**
 * @brief Open thie archive log file to be restored.
 * 
 * The first argument of the command is an input file name.  If omitted or specified as "-",
 * stdin will be used as an input file.
 * @param argc This is one of the argument to the command, number of the arguments.
 * @param argv This is one of the argument to the command, a pointer arry to the argument list.
 * @return File descriptor of the archive log file.
 * @note If error occurs within this function, whole command will exit here using exit() and
 * the caller will not have any chance to take care of errors.
 */
static int
open_arch_file(int argc, char *argv[])
{
	int from_fd = -1;

	if (argc > 1 && strcmp(argv[1], "-") != 0)
	{
		/* Open archive log file to restore. */
		from_fd = open(argv[1], O_RDONLY, 0);
		if (from_fd < 0)
		{
			fprintf(stderr, "failed to open `%s': %s\n", argv[1],
				strerror(errno));
			exit(1);
		}

		/* Obtain the segment ID from the file name (for record dump). */
		get_segment_id(argv[1]);
	}
	else
		from_fd = fileno(stdin);

	return from_fd;
}

/**
 * @brief Open the WAL segment file to write the restored data.
 * 
 * The secomd argument to the command will be regarded as an output file.  If omitted or
 * specified as "-", stdout will be used as the output file.
 * @param argc This is one of the argument to the command, number of the arguments.
 * @param argv This is one of the argument to the command, a pointer arry to the argument list.
 * @return File descriptor of the archive log file.
 * @note If error occurs within this function, whole command will exit here using exit() and
 * the caller will not have any chance to take care of errors.
 */
static int
open_xlog_file(int argc, char *argv[])
{
	int to_fd = -1;

	if (argc > 2 && strcmp(argv[2], "-") != 0)
	{
		/* Open the WAL segment file */
		to_fd = open(argv[2], O_RDWR | O_CREAT | O_EXCL | PG_BINARY,
			S_IRUSR | S_IWUSR);
		if (to_fd < 0)
		{
			fprintf(stderr, "failed to open `%s': %s\n", argv[2],
				strerror(errno));
			exit(1);
		}
	}
	else
		to_fd = fileno(stdout);

	return to_fd;
}
