/*-------------------------------------------------------------------------
 *
 * rmgr_hooks.c
 *
 *	  Postgres Resource Manager hooks 
 *
 *	  $PostgreSQL$
 *
 *	  Two examples shown
 *
 *		* EXAMPLE_WAL_DEBUG shows how you might output debug information
 *		for selected WAL records at the same time as continuing to 
 *		apply WAL records.
 *
 *		* EXAMPLE_HEAP_FILTER shows how to filter specific heap WAL records
 * 
 * 	  simon@2ndquadrant.com
 *-------------------------------------------------------------------------
 */

#include "postgres.h"

#include "funcapi.h"
#include "miscadmin.h"

#include "access/htup.h"
#include "access/xlog_internal.h"

PG_MODULE_MAGIC;	/* Tell the server about our compile environment */

/* external calls */
void plugin_rmgr_hook(RmgrData *RmgrTable);
void _PG_init(void);

static void plugin_xlog_debug(XLogRecPtr lsn, XLogRecord *rptr);

#define EXAMPLE_WAL_DEBUG

#ifdef EXAMPLE_WAL_DEBUG
static void plugin_btree_redo(XLogRecPtr lsn, XLogRecord *rptr);
#endif

#ifdef EXAMPLE_HEAP_FILTER
static bool want_record(XLogRecord *rptr);
static void plugin_heap_redo(XLogRecPtr lsn, XLogRecord *rptr);
static void plugin_heap2_redo(XLogRecPtr lsn, XLogRecord *rptr);
#endif

void
_PG_init(void)
{
	rmgr_hook = &plugin_rmgr_hook;
}

/*
 * plugin_rmgr_hook()
 */
void
plugin_rmgr_hook(RmgrData *RmgrTable)
{
#ifdef EXAMPLE_WAL_DEBUG
	RmgrTable[RM_BTREE_ID].rm_redo = plugin_btree_redo;
#endif 

#ifdef EXAMPLE_HEAP_FILTER
	RmgrTable[RM_HEAP2_ID].rm_redo = plugin_heap2_redo;
	RmgrTable[RM_HEAP_ID].rm_redo = plugin_heap_redo;

	elog(LOG, "rmgr_plugin: filtering away everything apart from database 11505");
#endif
}

static void 
plugin_xlog_debug(XLogRecPtr lsn, XLogRecord *rptr)
{
	int i;
	StringInfoData buf;

	initStringInfo(&buf);
	appendStringInfo(&buf, "LSN %X/%X: ",
					 lsn.xlogid, lsn.xrecoff);

	appendStringInfo(&buf, "xid %u",
				 rptr->xl_xid);

	for (i = 0; i < XLR_MAX_BKP_BLOCKS; i++)
	{
		if (rptr->xl_info & XLR_SET_BKP_BLOCK(i))
			appendStringInfo(&buf, "; bkpb%d", i + 1);
	}

	appendStringInfo(&buf, ": %s", RmgrName(rptr->xl_rmid));
	appendStringInfo(&buf, " - ");
	RmgrDesc(&buf, rptr, XLogRecGetData(rptr));
	elog(LOG, "%s", buf.data);
	pfree(buf.data);
}

#ifdef EXAMPLE_WAL_DEBUG
static void 
plugin_btree_redo(XLogRecPtr lsn, XLogRecord *rptr)
{
	plugin_xlog_debug(lsn, rptr);

	RmgrRedo(lsn, rptr);
}
#endif

#ifdef EXAMPLE_HEAP_FILTER
static void 
plugin_heap_redo(XLogRecPtr lsn, XLogRecord *rptr)
{
	/*
	 * apply filter, if record not required, skip
	 */
	if (!want_record(rptr))
		return;

	heap_redo(lsn, rptr);
}

static void 
plugin_heap2_redo(XLogRecPtr lsn, XLogRecord *rptr)
{
	/*
	 * apply filter, if record not required, skip
	 */
	if (!want_record(rptr))
		return;

	heap2_redo(lsn, rptr);
}

/*
 * want_record()
 * 
 * Function decides whether we want this xlog record or not.
 * Use the fact that all HEAP and HEAP2 records begin with a RelFileNode
 * in each of their various structures.
 */
static bool
want_record(XLogRecord *rptr)
{
	RelFileNode *xlrfn = (RelFileNode *) XLogRecGetData(rptr);

	if (xlrfn->dbNode != 11505)
	{
		elog(LOG, "not for this database");
		return false;
	}

	return true;
}
#endif
