//
// Copyright (C) 2011-2014 Grigory Kronin, Sergey Stepanov
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//

#ifndef _3cec8675_c338_4936_b2e3_038e7520b9c1_
#define _3cec8675_c338_4936_b2e3_038e7520b9c1_

#include <iterator>
#include <sstream>
#include <iomanip>
#include <stdexcept>

#include "impl/strutf_impl.h"

namespace soci { namespace utf {

	/// \defgroup utf8 Conversion between UTF-8 and UNICODE
	/// \ingroup soci_loc
	/// @{

	/// \brief Convert UTF-8 sequence to UNICODE
	/// \param itB,itE input sequence begin, end
	/// \param itTarget output sequence iterator
	/// \returns true if converted successfully
	template<typename _TItUTF_from, typename _TItUNICODE_to>
	bool utf8_to_unicode(const _TItUTF_from itB, const _TItUTF_from itE, _TItUNICODE_to itTarget)
	{ return impl::utf8_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(itB, itE, itTarget, (_TItUNICODE_to*)0); }

	/// \brief Convert UTF-16 sequence to UNICODE
	/// \param itB,itE input sequence begin, end
	/// \param itTarget output sequence iterator
	/// \returns true if converted successfully
	template<typename _TItUTF_from, typename _TItUNICODE_to>
	bool utf16_to_unicode(const _TItUTF_from itB, const _TItUTF_from itE, _TItUNICODE_to itTarget)
	{ return impl::utf16_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(itB, itE, itTarget, (_TItUNICODE_to*)0); }

	/// \brief Convert UTF-8 sequence to UNICODE
	/// \param itB,itE input sequence begin, end
	/// \param itTarget output sequence iterator
	/// \param[out] itTargetEnd if not NULL, returns next iterator after the last written
	/// \returns true if converted successfully
	template<typename _TItUTF_from, typename _TItUNICODE_to>
	bool utf8_to_unicode(const _TItUTF_from itB, const _TItUTF_from itE, _TItUNICODE_to itTarget, _TItUNICODE_to* itTargetEnd)
	{ return impl::utf8_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(itB, itE, itTarget, itTargetEnd); }

	/// \brief Convert UTF-16 sequence to UNICODE
	/// \param itB,itE input sequence begin, end
	/// \param itTarget output sequence iterator
	/// \param[out] itTargetEnd if not NULL, returns next iterator after the last written
	/// \returns true if converted successfully
	template<typename _TItUTF_from, typename _TItUNICODE_to>
	bool utf16_to_unicode(const _TItUTF_from itB, const _TItUTF_from itE, _TItUNICODE_to itTarget, _TItUNICODE_to* itTargetEnd)
	{ return impl::utf16_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(itB, itE, itTarget, itTargetEnd); }

	/// \brief Size of UNICODE converted from UTF-8
	/// \param itB,itE input sequence begin, end
	/// \param[out] sz number of entries in UNICODE sequence
	/// \returns true converted successfully
	template<typename _TItUTF_from>
	bool utf8_to_unicode_size(const _TItUTF_from itB, const _TItUTF_from itE, size_t& sz)
	{
		unsigned long valueUnicode;
		bool isFinished = false;
		int ret = 0;
		sz = 0;
		
		_TItUTF_from it = itB;
		while (
			(ret = impl::_utf8_to_unicode_eat_one(it, itE, false, valueUnicode, isFinished)) == std::codecvt_base::ok
			&&
			!isFinished)
		{
			++sz;
		}
		return ret == std::codecvt_base::ok;
	}

	/// \brief Convert UTF-8 string to UNICODE
	/// \param str input string
	/// \param itTarget output sequence iterator
	/// \returns true if converted successfully
	/// \remark NOTE: the output sequence does NOT receive terminating zero of the string
	template<typename _TItUNICODE_to>
	bool utf8_to_unicode(const char* str, _TItUNICODE_to itTarget)
	{ return impl::utf8_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(str, itTarget, (_TItUNICODE_to*)0); }

	/// \brief Convert UTF-8 string to UNICODE
	/// \param str input string
	/// \param itTarget output sequence iterator (NOTE: terminal 0 is not written)
	/// \param[out] itTargetEnd if not NULL, returns next iterator after the last written
	/// \returns true if converted successfully
	/// \remark NOTE: the output sequence does NOT receive terminating zero of the string
	/// \par Example
	/// \code
	/// const char* strUTF8 = "\xE2\x82\xAC"; // represents U+20AC, "Euro" currency symbol
	/// wchar_t strUNICODE[10];
	/// char* pEnd;
	/// utf8_to_unicode(strUTF8, strUNICODE, &pEnd);
	/// *pEnd = '0'; // write terminal zero
	/// assert(0 == wcscmp(strUNICODE, L"\x20AC"));
	/// \endcode
	template<typename _TItUNICODE_to>
	bool utf8_to_unicode(const char* str, _TItUNICODE_to itTarget, _TItUNICODE_to* itTargetEnd)
	{ return impl::utf8_to_unicode_as
		<typename impl::_SIterator2Value<_TItUNICODE_to>::value_type>
		(str, itTarget, itTargetEnd); }

	/// \brief Size of UNICODE converted from UTF-8
	/// \param str input UTF-8 sequence
	/// \param[out] sz number of entries in UNICODE sequence (NOTE: terminal 0 is not included)
	/// \returns true if converted successfully
	inline bool utf8_to_unicode_size(const char* str, size_t& sz)
	{
		bool isFinished = false;
		unsigned long valueUnicode;
		int ret = 0;
		sz = 0;

		while (
			(ret = impl::_utf8_to_unicode_eat_one(str, (const char*)0, true, valueUnicode, isFinished)) == std::codecvt_base::ok
			&&
			!isFinished)
		{
			++sz;
		}
		return ret == std::codecvt_base::ok;
	}

	/// \brief Convert UNICODE sequence to UTF-8 sequence
	/// \param itB,itE input sequence begin, end
	/// \param itTarget target iterator
	/// \returns iterator of the next target position after the last written
	/// \remark see details in http://en.wikipedia.org/wiki/UTF-8
	template<typename _TItUNICODE_from, typename _TItUTF_to>
	_TItUTF_to utf8_from_unicode(const _TItUNICODE_from itB, const _TItUNICODE_from itE, _TItUTF_to itTarget)
	{
		for (_TItUNICODE_from it = itB; it != itE; ++it) {
			typedef typename std::iterator_traits<_TItUNICODE_from>::value_type _TValueFrom;
			unsigned long unicode = static_cast<unsigned long>(*it);
			// brutal trick to remove sign for correct cast from signed types
			for (size_t i=sizeof(_TValueFrom); i<sizeof(unicode); ++i) {
				unicode &= ~(0xFFU << (8*i));
			}
			itTarget = impl::_utf8_from_unicode_eat_one(unicode, itTarget);
		}
		return itTarget;
	}

	template<typename _TItUNICODE_from, typename _TItUTF_to>
	_TItUTF_to utf16_from_unicode(const _TItUNICODE_from itB, const _TItUNICODE_from itE, _TItUTF_to itTarget)
	{
		for (_TItUNICODE_from it = itB; it != itE; ++it) {
			typedef typename std::iterator_traits<_TItUNICODE_from>::value_type _TValueFrom;
			unsigned long unicode = static_cast<unsigned long>(*it);
			// brutal trick to remove sign for correct cast from signed types
			for (size_t i=sizeof(_TValueFrom); i<sizeof(unicode); ++i) {
				unicode &= ~(0xFFU << (8*i));
			}
			itTarget = impl::_utf16_from_unicode_eat_one(unicode, itTarget);
		}
		return itTarget;
	}

	/// \brief Convert UNICODE sequence to UTF-8 sequence
	/// \param str zero-terminated UNICODE string
	/// \param itTarget target iterator
	/// \returns iterator of the next target position after the last written (NOTE: terminal 0 is not written)
	/// \remark see details in http://en.wikipedia.org/wiki/UTF-8
	template<typename _TChar, typename _TItUTF_to>
	_TItUTF_to utf8_from_unicode(const _TChar* str, _TItUTF_to itTarget)
	{
		for (const _TChar* p = str; *p; ++p) {
			unsigned long unicode = static_cast<unsigned long>(*p);
			// brutal trick to remove sign for correct cast from signed types
			for (size_t i=sizeof(_TChar); i<sizeof(unicode); ++i) {
				unicode &= ~(0xFFU << (8*i));
			}
			itTarget = impl::_utf8_from_unicode_eat_one(unicode, itTarget);
		}
		return itTarget;
	}

	/// @}
}}

#endif
