//
// Copyright (C) 2004-2006 Maciej Sobczak, Stephen Hutton, David Courtney
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//

#define SOCI_ODBC_SOURCE
#include "soci-odbc.h"
#include "strutf.h"
#include <cctype>
#include <cstdio>
#include <cstring>
#include <ctime>
#include <sstream>
#include <algorithm>
#include <string>
#include <vector>

using namespace soci;
using namespace soci::details;

void odbc_vector_into_type_backend::prepare_indicators(std::size_t size)
{
    if (size == 0)
    {
         throw soci_error("Vectors of size 0 are not allowed.");
    }

    indHolderVec_.resize(size);
    indHolders_ = &indHolderVec_[0];
}

void odbc_vector_into_type_backend::define_by_pos(
    int &position, void *data, exchange_type type)
{
    data_ = data; // for future reference
    type_ = type; // for future reference

    SQLINTEGER size = 0;       // also dummy

    switch (type)
    {
    // simple cases
    case x_short:
        {
            odbcType_ = SQL_C_SSHORT;
            size = sizeof(short);
            std::vector<short> *vp = static_cast<std::vector<short> *>(data);
            std::vector<short> &v(*vp);
            prepare_indicators(v.size());
            data = &v[0];
        }
        break;
    case x_integer:
        {
            odbcType_ = SQL_C_SLONG;
            size = sizeof(long);
            std::vector<long> *vp = static_cast<std::vector<long> *>(data);
            std::vector<long> &v(*vp);
            prepare_indicators(v.size());
            data = &v[0];
        }
        break;
    case x_unsigned_long:
        {
            odbcType_ = SQL_C_ULONG;
            size = sizeof(unsigned long);
            std::vector<unsigned long> *vp
                = static_cast<std::vector<unsigned long> *>(data);
            std::vector<unsigned long> &v(*vp);
            prepare_indicators(v.size());
            data = &v[0];
        }
        break;
    case x_double:
        {
            odbcType_ = SQL_C_DOUBLE;
            size = sizeof(double);
            std::vector<double> *vp = static_cast<std::vector<double> *>(data);
            std::vector<double> &v(*vp);
            prepare_indicators(v.size());
            data = &v[0];
        }
        break;

    // cases that require adjustments and buffer management

    case x_char:
        {
            odbcType_ = SQL_C_WCHAR;
            std::vector<char> * v = static_cast<std::vector<char> *>(data);
            prepare_indicators(v->size());
			colSize_ = statement_.column_size(position); // in characters
			size = colSize_ * 2 * sizeof(SQLWCHAR); // max size for column in bytes
			// TODO : support for utf16 surrogates
            str_buf_.resize(v->size() * size);
            data = &str_buf_[0];
        }
        break;
    case x_stdstring:
        {
            odbcType_ = SQL_C_WCHAR;
            std::vector<std::string> *v = static_cast<std::vector<std::string> *>(data);
            prepare_indicators(v->size());            
			colSize_ = statement_.column_size(position); // in bytes
			size = colSize_ * sizeof(SQLWCHAR); // max size for column in bytes
			// TODO : support for utf16 surrogates
            str_buf_.resize(v->size() * size);
            data = &str_buf_[0];
        }
        break;
    case x_stdtm:
        {
            odbcType_ = SQL_C_TYPE_TIMESTAMP;
            std::vector<std::tm> *v
                = static_cast<std::vector<std::tm> *>(data);

            prepare_indicators(v->size());

            size = sizeof(TIMESTAMP_STRUCT);
            colSize_ = size;

            std::size_t bufSize = size * v->size();

            buf_ = new char[bufSize];
            data = buf_;
        }
        break;

    case x_statement: break; // not supported
    case x_rowid:     break; // not supported
    case x_blob:      break; // not supported
	case x_long_long: break; // TODO: verify if can be supported
	case x_unsigned_long_long: break; // TODO: verify if can be supported
    }

    SQLRETURN rc = SQLBindCol(
		  statement_.hstmt_
		, static_cast<SQLUSMALLINT>(position++)
		, odbcType_
		, data
		, size
		, indHolders_
	);

    if (is_odbc_error(rc))
    {
        throw odbc_soci_error(
			  SQL_HANDLE_STMT, statement_.hstmt_
			  , "vector into type define by pos"
		);
    }
}

void odbc_vector_into_type_backend::pre_fetch()
{
    // nothing to do for the supported types
}

void odbc_vector_into_type_backend::post_fetch(bool gotData, indicator *ind)
{
    if (gotData)
    {
        // first, deal with data

        // only std::string, std::tm and Statement need special handling
        if (type_ == x_char)
        {
			// here we converting utf16 to cp latin1
			// making assumption that this utf16 char sequence consists of set of 1 single char_16s
			// since other characters are anyway cannot be converted to latin1
            std::vector<char> * vp = static_cast<std::vector<char> *>(data_); // target Latin1 string
			std::size_t length = std::remove(str_buf_.begin(), str_buf_.end(), 0) - str_buf_.begin();
			if (!utf::utf16_to_unicode(str_buf_.begin(), str_buf_.begin()+length, vp->begin())) {
				// fails if UNICODE does not match Latin-1 range 0...255
				throw odbc_soci_error(
					  SQL_HANDLE_STMT, statement_.hstmt_
					, "Character is not convertible to coding table 'Latin1'"
				);
			}
        }
        if (type_ == x_stdstring)
        {
		    // target collection of utf8 strings
            std::vector<std::string> *collection = static_cast<std::vector<std::string> *>(data_); 
			std::vector<std::string>::iterator itr = collection->begin();
			std::vector<std::string>::iterator end = collection->end();
			for ( std::size_t index = 0; itr != end ; ++itr, ++index) {
				// performing conversion from utf16 to utf8 for each string
				// here we should put strings one by one from the buffer in equal chunks 
				// (i.e for colSize_ = 5 : a b c 0 0   a b 0 0 0   a b c d 0)
				itr->clear();
				std::vector<unsigned int> unicode_seq;
				std::vector<SQLWCHAR>::iterator pos_begin = str_buf_.begin() + index * colSize_;
				std::vector<SQLWCHAR>::iterator pos_end = pos_begin + 1; 
				while(*pos_end) pos_end++; // searching for end of string
				utf::utf16_to_unicode(pos_begin, pos_end, std::back_inserter(unicode_seq));
				utf::utf8_from_unicode(unicode_seq.begin(), unicode_seq.end(), std::back_inserter(*itr));
			}
        }
        else if (type_ == x_stdtm)
        {
            std::vector<std::tm> *vp
                = static_cast<std::vector<std::tm> *>(data_);

            std::vector<std::tm> &v(*vp);
            char *pos = buf_;
            std::size_t const vsize = v.size();
            for (std::size_t i = 0; i != vsize; ++i)
            {
                std::tm t;

                TIMESTAMP_STRUCT * ts = reinterpret_cast<TIMESTAMP_STRUCT*>(pos);
                t.tm_isdst = -1;
                t.tm_year = ts->year - 1900;
                t.tm_mon = ts->month - 1;
                t.tm_mday = ts->day;
                t.tm_hour = ts->hour;
                t.tm_min = ts->minute;
                t.tm_sec = ts->second;

                // normalize and compute the remaining fields
                std::mktime(&t);
                v[i] = t;
                pos += colSize_;
            }
        }

        // then - deal with indicators
        if (ind != NULL)
        {
            std::size_t const indSize = statement_.get_number_of_rows();
            for (std::size_t i = 0; i != indSize; ++i)
            {
                if (indHolderVec_[i] > 0)
                {
                    ind[i] = i_ok;
                }
                else if (indHolderVec_[i] == SQL_NULL_DATA)
                {
                    ind[i] = i_null;
                }
                else
                {
                    ind[i] = i_truncated;
                }
            }
        }
        else
        {
            std::size_t const indSize = statement_.get_number_of_rows();
            for (std::size_t i = 0; i != indSize; ++i)
            {
                if (indHolderVec_[i] == SQL_NULL_DATA)
                {
                    // fetched null and no indicator - programming error!
                    throw soci_error(
                        "Null value fetched and no indicator defined.");
                }
            }
        }
    }
    else // gotData == false
    {
        // nothing to do here, vectors are truncated anyway
    }
}

void odbc_vector_into_type_backend::resize(std::size_t sz)
{
    indHolderVec_.resize(sz);
    switch (type_)
    {
    // simple cases
    case x_char:
        {
            std::vector<char> *v = static_cast<std::vector<char> *>(data_);
            v->resize(sz);
        }
        break;
    case x_short:
        {
            std::vector<short> *v = static_cast<std::vector<short> *>(data_);
            v->resize(sz);
        }
        break;
    case x_integer:
        {
            std::vector<long> *v = static_cast<std::vector<long> *>(data_);
            v->resize(sz);
        }
        break;
    case x_unsigned_long:
        {
            std::vector<unsigned long> *v
                = static_cast<std::vector<unsigned long> *>(data_);
            v->resize(sz);
        }
        break;
    case x_double:
        {
            std::vector<double> *v
                = static_cast<std::vector<double> *>(data_);
            v->resize(sz);
        }
        break;
    case x_stdstring:
        {
            std::vector<std::string> *v
                = static_cast<std::vector<std::string> *>(data_);
            v->resize(sz);
        }
        break;
    case x_stdtm:
        {
            std::vector<std::tm> *v
                = static_cast<std::vector<std::tm> *>(data_);
            v->resize(sz);
        }
        break;

    case x_statement: break; // not supported
    case x_rowid:     break; // not supported
    case x_blob:      break; // not supported
	case x_long_long: break; // TODO: verify if can be supported
	case x_unsigned_long_long: break; // TODO: verify if can be supported
    }
}

std::size_t odbc_vector_into_type_backend::size()
{
    std::size_t sz = 0; // dummy initialization to please the compiler
    switch (type_)
    {
    // simple cases
    case x_char:
        {
            std::vector<char> *v = static_cast<std::vector<char> *>(data_);
            sz = v->size();
        }
        break;
    case x_short:
        {
            std::vector<short> *v = static_cast<std::vector<short> *>(data_);
            sz = v->size();
        }
        break;
    case x_integer:
        {
            std::vector<long> *v = static_cast<std::vector<long> *>(data_);
            sz = v->size();
        }
        break;
    case x_unsigned_long:
        {
            std::vector<unsigned long> *v
                = static_cast<std::vector<unsigned long> *>(data_);
            sz = v->size();
        }
        break;
    case x_double:
        {
            std::vector<double> *v
                = static_cast<std::vector<double> *>(data_);
            sz = v->size();
        }
        break;
    case x_stdstring:
        {
            std::vector<std::string> *v
                = static_cast<std::vector<std::string> *>(data_);
            sz = v->size();
        }
        break;
    case x_stdtm:
        {
            std::vector<std::tm> *v
                = static_cast<std::vector<std::tm> *>(data_);
            sz = v->size();
        }
        break;

    case x_statement: break; // not supported
    case x_rowid:     break; // not supported
    case x_blob:      break; // not supported
	case x_long_long: break; // TODO: verify if can be supported
	case x_unsigned_long_long: break; // TODO: verify if can be supported
    }

    return sz;
}

void odbc_vector_into_type_backend::clean_up()
{
    if (buf_ != NULL)
    {
        delete [] buf_;
        buf_ = NULL;
    }
}
