/*
 * usbmidi.c - ALSA USB US-428 Firmware Download driver
 *
 * Copyright (c) 2002 Karsten Wiese
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed and/or modified under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/string.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>
//#include <asm/semaphore.h>

MODULE_AUTHOR("Karsten Wiese <annabellesgarden@yahoo.de>");
MODULE_DESCRIPTION("USB TASCAM (0x1604) US-428 (0x8000) Firware Loader ");
MODULE_LICENSE("GPL");


static void* snd_us428_dl_usb_probe( struct usb_device* device,
				   unsigned int ifnum,
				   const struct usb_device_id* device_id);
static void snd_us428_dl_usb_disconnect( struct usb_device* usb_device, void* ptr);


struct us428_control{
	unsigned short TransferBuLe;
	unsigned char  TransferBuffer[ 16];
	unsigned char  Request;
	unsigned short Value;
};


struct us428_control us428_firmware[ 602] = {
#include "usbsnoop.log"
};




static struct usb_device_id snd_us428_dl_usb_id_table[] = {
	{ match_flags:	USB_DEVICE_ID_MATCH_DEVICE,
	  idVendor:	0x1604,
	  idProduct:	0x8000 },
	{ /* terminator */ }
};

MODULE_DEVICE_TABLE(usb, snd_us428_dl_usb_id_table);

static struct usb_driver snd_us428_dl_usb_driver = {
	name: "snd-usb-us428-dl",
	probe: snd_us428_dl_usb_probe,
	disconnect: snd_us428_dl_usb_disconnect,
	id_table: snd_us428_dl_usb_id_table,
	driver_list: LIST_HEAD_INIT(snd_us428_dl_usb_driver.driver_list)
};




/*
 * Probes for an us428 in cold boot state & downloads sniffed firmware
 */
static void* snd_us428_dl_usb_probe( struct usb_device* device,
				   unsigned int ifnum,
				   const struct usb_device_id* device_id)
{
	void* 	card = NULL;
	char* buf;
	int	i, err;
	struct usb_interface *interface = device->actconfig->interface + ifnum;

		/* See if the device offered us matches what we can accept */
	if ((device->descriptor.idVendor != 0x1604) ||
	    (device->descriptor.idProduct != 0x8000)) {
		return NULL;
	}

	printk( "snd_us428_dl_usb_probe( ifnum=%i, d_id=%i, interface->act_altsetting=%i)\n", ifnum, (int)device_id, (int)interface->act_altsetting);
	if( 0 != interface->act_altsetting)
		return NULL;
/*	
	usb_driver_claim_interface( &snd_us428_dl_usb_driver, interface, snd_us428_dl_usb_probe);
	
	if( err = usb_set_configuration( device, 1)){
		printk( "conf err \n");
		usb_driver_release_interface(&snd_us428_dl_usb_driver, interface);
		return NULL;
	}
	if( err = usb_set_interface( device, 0, 0)){
		printk( "iface err \n");
		usb_driver_release_interface(&snd_us428_dl_usb_driver, interface);
		return NULL;
	}
*/
//Jul 18 23:36:14 lise kernel: 
//	usb.c: usb_control_msg( usb_device=0xCAA22400, pipe=0x80000200, request=0xA0, requesttype=0x40,
//Jul 18 23:36:16 lise last message repeated 601 times
	if( NULL == ( buf = kmalloc( 0x20, GFP_KERNEL)))
		return -ENOMEM; 
	for( i = 0; i < ( sizeof(us428_firmware)/sizeof(struct us428_control)); ++i){
		memcpy( buf, us428_firmware[ i].TransferBuffer, us428_firmware[ i].TransferBuLe);
		if( 0 > ( err = usb_control_msg( 
					device, 
					usb_sndctrlpipe(device, 0), 
					us428_firmware[ i].Request, 
					USB_TYPE_VENDOR|USB_RECIP_DEVICE|USB_DIR_OUT,
				 	us428_firmware[ i].Value, 0, 
				 	buf, //us428_firmware[ i].TransferBuffer,
				 	us428_firmware[ i].TransferBuLe,
				 	HZ 
					)
/* 
usb_sndctrlpipe(device, 0):
	usb.c: usb_control_msg( usb_device=0xCAA22400, pipe=0x80000200, request=0xA0, requesttype=0x40,
         value=0x7F92, index=0x0, data=0xD08A64E2, size=0x1, timeout=0x64


usb_snddefctrl(device), :					
	usb.c: usb_control_msg( usb_device=0xCAA22400, pipe=0x80000000, request=0xA0, requesttype=0x40,
         value=0x7F92, index=0x0, data=0xD08A64E2, size=0x1, timeout=0x64
		*/
			)){
			printk( "msg %i: error %i! R %i RT=%X V=%i B=%i L=%i\n", i, err, 
					(int)us428_firmware[ i].Request,
					USB_TYPE_VENDOR|USB_RECIP_DEVICE|USB_DIR_OUT,
					(int)us428_firmware[ i].Value,
					(int)us428_firmware[ i].TransferBuffer[ 0],
					(int)us428_firmware[ i].TransferBuLe
					);
			usb_driver_release_interface(&snd_us428_dl_usb_driver, interface);
			kfree( buf);
			return NULL;	 
		}

/*			{
			signed long T = HZ / 50;
				set_current_state( TASK_UNINTERRUPTIBLE);
				while( T = schedule_timeout( T));
			return NULL;	 
			}*/
			
	}
	kfree( buf);
	printk( "snd_us428_dl_usb_probe: Firmware Download complete :-)\n");
	
	return snd_us428_dl_usb_probe;
}

/*
 * Frees the device.
 */
static void snd_us428_dl_usb_disconnect( struct usb_device* device, void* ptr)
{
	printk( "snd_us428_dl_usb_disconnect( struct usb_device* =0x%X, void* ptr)\n", (int)device);
}

static int __init snd_us428_dl_module_init(void)
{
	int err;

	err = usb_register(&snd_us428_dl_usb_driver);
	if (err < 0) {
		return err;
	}
	return 0;
}

static void __exit snd_us428_dl_module_exit(void)
{
	usb_deregister(&snd_us428_dl_usb_driver);
}

module_init(snd_us428_dl_module_init)
module_exit(snd_us428_dl_module_exit)

#ifndef MODULE

/*
 * format is snd-usb-midi=snd_enable,snd_index,snd_id,snd_pid,snd_int_transfer
 */
static int __init snd_us428_dl_module_setup(char* str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	(void)(get_option(&str, &snd_enable[nr_dev]) == 2 &&
	       get_option(&str, &snd_index[nr_dev]) == 2 &&
	       get_id(&str, &snd_id[nr_dev]) == 2 &&
	       get_option(&str, &snd_vid[nr_dev]) == 2 &&
	       get_option(&str, &snd_pid[nr_dev]) == 2 &&
	       get_option(&str, &snd_int_transfer[nr_dev]) == 2);
	++nr_dev;
	return 1;
}

__setup("snd-usb-midi=", snd_us428_dl_module_setup);

#endif /* !MODULE */

EXPORT_NO_SYMBOLS;
