/*
 * mausoload.c - firmware loader for the M-Audio Sonica
 *
 * Copyright Clemens Ladisch
 *
 * based on the DFU download utility
 *
 * Copyright Brad Hards and Bas Vermeulen
 */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "usb.h"

/* USB stuff */
#define USB_DIR_OUT 0x00
#define USB_DIR_IN  0x80

/* DFU states */

#define STATE_IDLE  			0x00
#define STATE_DETACH			0x01
#define STATE_DFU_IDLE			0x02
#define STATE_DFU_DOWNLOAD_SYNC		0x03
#define STATE_DFU_DOWNLOAD_BUSY		0x04
#define STATE_DFU_DOWNLOAD_IDLE		0x05
#define STATE_DFU_MANIFEST_SYNC		0x06
#define STATE_DFU_MANIFEST		0x07
#define STATE_DFU_MANIFEST_WAIT_RESET	0x08
#define STATE_DFU_UPLOAD_IDLE		0x09
#define STATE_DFU_ERROR			0x0a

/* DFU commands */
#define DFU_DETACH			0
#define DFU_DNLOAD			1
#define DFU_UPLOAD			2
#define DFU_GETSTATUS			3
#define DFU_CLRSTATUS			4
#define DFU_GETSTATE			5
#define DFU_ABORT			6

struct dfu_status {
	unsigned char bStatus;
	unsigned char bwPollTimeout[3];
	unsigned char bState;
	unsigned char iString;
} __attribute__ ((packed));


static const int timeout = 1000;

static const int dfu_packetsize = 0x40;

int DFUDownload(usb_dev_handle * dev, unsigned char *Buffer,
		unsigned long Bytes, unsigned short Block)
{
	int result;

	//printf("DOWNLOADING %ld Bytes (Block = %d)\n", Bytes, Block);
	result = usb_control_msg(dev, USB_TYPE_CLASS | USB_DIR_OUT | USB_RECIP_INTERFACE, DFU_DNLOAD, Block,    /* Value */
				 0,     /* Index */
				 Buffer,        /* Buffer */
				 Bytes, /* Size */
				 timeout);

	if (result < 0)
		printf("Write FW Block %d failed (%s)!\n", Block,
		       usb_strerror());

	return result;
}

int DFUGetStatus(usb_dev_handle * dev, struct dfu_status *status)
{
	int result;
	unsigned char stat_buff[6];

	result = usb_control_msg(dev, USB_TYPE_CLASS | USB_DIR_IN | USB_RECIP_INTERFACE, DFU_GETSTATUS, 0,      /* Value */
				 0,     /* Index */
				 (unsigned char *) &stat_buff,  /* Buffer */
				 6,     /* Size */
				 timeout);

	memcpy(status, &stat_buff, 6);

	return result;
}

int DFUGetState(usb_dev_handle * dev, unsigned char *state)
{
	return usb_control_msg(dev, USB_TYPE_CLASS | USB_DIR_IN | USB_RECIP_INTERFACE,        /* Request Type */
			       DFU_GETSTATE,  /* Request */
			       0,     /* Value */
			       0,     /* Index */
			       state, /* Buffer */
			       1,     /* Size */
			       timeout);
}

int DownloadFirmware(usb_dev_handle * dev, unsigned char *DfuBuffer,
		     unsigned int DfuLen)
{
	int status = 0;
	int NeedDFUState = 1;
	int IsDone = 0;
	struct dfu_status DfuStatusBuffer;
	unsigned char DfuState = 0;
	unsigned long DfuTimeout = 0;
	unsigned long DfuBlockBytes, DfuBytesLeft, DfuBufferOffset;
	unsigned long DfuBlockCnt;
	int t;

	DfuBlockCnt = 0;
	DfuBytesLeft = DfuLen;
	DfuBlockBytes = 0;
	DfuBufferOffset = 0;

	if (DfuLen == 0) {
		printf("FW Buffer length invalid!\n");
		return -1;
	}

	printf("Start FW Downloading...\n");

	do {
		if (NeedDFUState) {
			status = DFUGetState(dev, &DfuState);
			if (status < 0) {
				printf("DFU: Failed to get DFU state...\n");
				return -1;
			}
			NeedDFUState = 0;
		}

		switch (DfuState) {
			case STATE_DFU_DOWNLOAD_SYNC:
				printf("STATE_DFU_DOWNLOAD_SYNC\n");
				if (DFUGetStatus(dev, &DfuStatusBuffer) < 0) {
					DfuState = DfuStatusBuffer.bState;
					DfuTimeout = 0;
					DfuTimeout =
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[2] <<
								 16);
					DfuTimeout |=
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[1] <<
								 8);
					DfuTimeout |=
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[0]);
					NeedDFUState = 0;
				}
				break;
			case STATE_DFU_DOWNLOAD_BUSY:
				printf("STATE_DFU_DOWNLOAD_BUSY\n");
				NeedDFUState = 1;

				if (DfuTimeout > 0)
					printf("DFU: Resetting device\n");
				else
					printf("DFU: In progress\n");

				for (t = 1; t <= DfuTimeout / 500; t++)
					usleep(500 * 1000);     /* Sleep for 500 ms */
				break;
			case STATE_DFU_DOWNLOAD_IDLE:
				printf("DOWNLOAD ");
			case STATE_DFU_IDLE:
				printf("DFU IDLE\n");

				if (DfuBytesLeft <= dfu_packetsize)
					DfuBlockBytes = DfuBytesLeft;
				else
					DfuBlockBytes = dfu_packetsize;

				if (status >= 0) {
					DfuBytesLeft -= DfuBlockBytes;
					status = DFUDownload(dev,
							     DfuBuffer +
							     DfuBufferOffset,
							     DfuBlockBytes,
							     DfuBlockCnt);
					DfuBufferOffset += DfuBlockBytes;
					DfuBlockCnt++;
				}
				NeedDFUState = 1;
				break;
			case STATE_DFU_MANIFEST_SYNC:
				printf("STATE_DFU_MANIFEST_SYNC\n");

				status = DFUGetStatus(dev, &DfuStatusBuffer);

				if (status >= 0) {
					DfuState = DfuStatusBuffer.bState;
					DfuTimeout = 0;
					DfuTimeout =
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[2] <<
								 16);
					DfuTimeout |=
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[1] <<
								 8);
					DfuTimeout |=
						(unsigned long) (DfuStatusBuffer.
								 bwPollTimeout[0]);
					NeedDFUState = 0;

					if (DfuTimeout > 0)
						printf("DFU: Resetting device\n");
					else
						printf("DFU: In progress\n");

					for (t = 1; t <= DfuTimeout / 500; t++)
						usleep(500 * 1000);     /* Sleep for 500 ms */
				}
				break;
			case STATE_DFU_MANIFEST:
				printf("STATE_DFU_MANIFEST\n");
				IsDone = 1;
				break;
			case STATE_DFU_MANIFEST_WAIT_RESET:
				printf("STATE_DFU_MANIFEST_WAIT_RESET\n");
				usb_reset(dev);
				break;
			case STATE_DFU_UPLOAD_IDLE:
				printf("STATE_DFU_UPLOAD_IDLE\n");
				break;
			case STATE_DFU_ERROR:
				printf("STATE_DFU_ERROR\n");
				usb_reset(dev);
				break;
			default:
				printf("DFU UNKNOWN STATE (%d)\n", DfuState);
				status = -1;
				break;
		}
	} while (!IsDone && status >= 0);

	return status;
}

#include "ma005101.h"
#define VID 0x0763
#define PID 0x2805 /* before firmware download */

void download(struct usb_device *device)
{
	usb_dev_handle *dev;

	dev = usb_open(device);
	usb_claim_interface(dev, 0);
	DownloadFirmware(dev, (unsigned char*)ma005101, sizeof(ma005101));
	usb_release_interface(dev, 0);
	usb_reset(dev);
	usb_close(dev);
}

int main(void)
{
	struct usb_bus *bus;
	struct usb_device *dev;

	usb_init();
	usb_find_busses();
	usb_find_devices();
	for (bus = usb_busses; bus; bus = bus->next)
		for (dev = bus->devices; dev; dev = dev->next)
			if (dev->descriptor.idVendor == VID &&
			    dev->descriptor.idProduct == PID) {
				printf("downloading to %s\n", dev->filename);
				download(dev);
				return 0;
			}
	puts("no device?");
	return 1;
}
