package org.apache.tools.ant.taskdefs.optional.scm;

/**
 * ============================================================================
 *                   The Apache Software License, Version 1.1
 * ============================================================================
 * 
 *    Copyright (C) 2000-2002 The Apache Software Foundation. All
 *    rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modifica-
 * tion, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of  source code must  retain the above copyright  notice,
 *    this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * 3. The end-user documentation included with the redistribution, if any, must
 *    include  the following  acknowledgment:  "This product includes  software
 *    developed  by the  Apache Software Foundation  (http://www.apache.org/)."
 *    Alternately, this  acknowledgment may  appear in the software itself,  if
 *    and wherever such third-party acknowledgments normally appear.
 * 
 * 4. The names "Ant" and  "Apache Software Foundation"  must not be used to
 *    endorse  or promote  products derived  from this  software without  prior
 *    written permission. For written permission, please contact
 *    apache@apache.org.
 * 
 * 5. Products  derived from this software may not  be called "Apache", nor may
 *    "Apache" appear  in their name,  without prior written permission  of the
 *    Apache Software Foundation.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS  FOR A PARTICULAR  PURPOSE ARE  DISCLAIMED.  IN NO  EVENT SHALL  THE
 * APACHE SOFTWARE  FOUNDATION  OR ITS CONTRIBUTORS  BE LIABLE FOR  ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL,  EXEMPLARY, OR CONSEQUENTIAL  DAMAGES (INCLU-
 * DING, BUT NOT LIMITED TO, PROCUREMENT  OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR  PROFITS; OR BUSINESS  INTERRUPTION)  HOWEVER CAUSED AND ON
 * ANY  THEORY OF LIABILITY,  WHETHER  IN CONTRACT,  STRICT LIABILITY,  OR TORT
 * (INCLUDING  NEGLIGENCE OR  OTHERWISE) ARISING IN  ANY WAY OUT OF THE  USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * This software  consists of voluntary contributions made  by many individuals
 * on behalf of the  Apache Software Foundation.  For more  information  on the 
 * Apache Software Foundation, please see <http://www.apache.org/>.
 *
 */

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.optional.scm.writer.ChangeLogWriter;
import com.starbase.starteam.ChangeRequest;
import com.starbase.starteam.File;
import com.starbase.starteam.Folder;
import com.starbase.starteam.Item;
import com.starbase.starteam.Label;
import com.starbase.starteam.Project;
import com.starbase.starteam.Server;
import com.starbase.starteam.StarTeamURL;
import com.starbase.starteam.User;
import com.starbase.starteam.View;

/**
 * Creates a change log which will show the changed files and descriptions between two labels in Star Team.
 * <BR>
 * This program will create a changelog for everything that has changed between two view labels. The
 * change log class used by default is org.apache.tools.ant.taskdefs.optional.scm.writer.<BR>
 * <B>This program makes use of functions from the StarTeam API.
 * As a result STChangeLog is available only to licensed users of StarTeam and
 * requires the StarTeam SDK to function. You must have starteam- sdk. jar in
 * your classpath to run this program. For more information about the StarTeam
 * API and how  to license it, see the link below. </B>
 * @author Kevin Landry <A href="mailto:klandry@ceira.com">klandry@ceira.com</A>
 * @see <A href="http://www.starbase.com/">StarBase Web Site</A>
 */
public class STChangeLog extends Task {

  // ant vars
  private String buildLabel;
  private String previousLabel;
  private String starTeamPath;
  private boolean changeRequests;
  private String logWriter;
  private String destFile;
  private ChangeLogWriter myChangeLogWriter;

  /**
   * Default CLASS to load if no ChangeLogWriter class was specified
   */
   public static final String DEFAULT_CHANGELOGWRITER = "org.apache.tools.ant.taskdefs.optional.scm.writer.ChangeLogTextWriter";

  // Star team vars
  private Server starTeam;
  private Project project;
  private View view;
  private Folder folder;
  private Item [] items;
  private Label label = null;
  private Label prevLabel = null;
   private String projectName;
  private String viewName;
  
	/**
	 * Constructor for STChangeLog.
	 */
	public STChangeLog() {
		super();
		starTeam = null;
		project = null;
		view = null;
		folder = null;
		items = null;
		label = null;
		prevLabel = null;
	}

	/**
	 * @see org.apache.tools.ant.Task#execute()
	 */
	public void execute() throws BuildException {
	
	// Check the params
	if( buildLabel == null )
		throw new BuildException("Required argument buildLabel not specified.");
	if( starTeamPath == null )
		throw new BuildException("Required argument starTeamPath not specified.");
	if( destFile == null )
		throw new BuildException("Required argument outputFile not specified.");

    try {
      // print the change log
      printChangeLog();
      myChangeLogWriter.writeFooter();
    } catch( Exception e ) {
      throw new BuildException(e);
    } finally {
		clean();	// cleanup
    }

  }

  /**
   * Make sure everything is cleaned up before exiting.
   */
  protected void clean( )
  {
    try {
      if( myChangeLogWriter != null )
        myChangeLogWriter.closeFile();
    } catch( java.io.IOException e ) {
     log("Error closing changelog file: " + e );
    }
    if( starTeam != null )
    {
    	starTeam.disconnect();
    	starTeam = null;
  	} 
  }

  /**
   * Prints the change log
   */
  protected void printChangeLog( ) throws Exception
  {
    try {
      // open the output file
      createFile();
    } catch( Exception e ) {
      log("An error has occured while trying to create the changelog: " + e );
      throw e;
    }

    // grab the server
    processServer();


    // grab the right project
    processProject();

    // grab the right view
    getView();

    // create the header
    myChangeLogWriter.writeHeader(starTeam, "Build: " + buildLabel + "\r\n");

    // grab the change requests to iterate through
    //folder.populateNow(starTeam.getTypeNames().CHANGEREQUEST, null, -1);
    //getItems(folder, starTeam.getTypeNames().CHANGEREQUEST);
    //folder.discard();

    // grab the files to iterate through them
    folder.populateNow( starTeam.getTypeNames().FILE, null, -1 );
    log( "Checking files", org.apache.tools.ant.Project.MSG_INFO );
    getItems( folder, starTeam.getTypeNames().FILE );
    log("Cleaning up and closing connection", org.apache.tools.ant.Project.MSG_INFO);
    folder.discard();
  }

  /**
   * Connects and logs into the server
   */
  protected void processServer()
  {
    debug("Entering getServer()");
    StarTeamURL url = new StarTeamURL( starTeamPath );
    starTeam = new Server( url.getHostName(), Integer.parseInt(url.getPort()) );
    starTeam.connect();
    starTeam.logOn(url.getUserName(), url.getPassword() );
    projectName = url.getProjectName();
    viewName = url.getPath();
    log( "Connected to server " + url.getHostName() + " on port " + url.getPort(), org.apache.tools.ant.Project.MSG_INFO);
  }

  /**
   * Grabs the appropriate project and stores it in the project variable.
   */
  protected void processProject( ) throws Exception
  {
    debug("Entering getProject()");

    // Grab the project list and find the right one.
    // There may be a better way to do this
    Project [] pList = starTeam.getProjects();
    for( int i = 0; i < pList.length; i++ )
    {
      if( pList[i].getName().equals( projectName ) )
      {
        project = pList[i];
        return;
      }
    }

    // we are exiting . . project was not found
    clean();
    throw new Exception("Error: Could not find project " + projectName );
  }

  /**
   * Grabs the view that was specified (or the default view if one was
   * not specified). Checks to see if the current label and previous label is
   * in the view that was selected. This function also grabs the root folder to
   * be stored in the folder variable.
   */
  protected void getView( ) throws Exception
  {
  	// vars
    debug("Entering getView()");
    View [] views = project.getViews();
    Label [] labels = null;
    
    // grabs the default view right now . . .eventually it should grab the view that was passed
    for( int i = 0; i < views.length; i++ )
    {
      if( views[i].getName().equalsIgnoreCase(viewName) )
      {
        view = views[i];
        break;
      }
    }
    if( view == null ) view = project.getDefaultView();
    labels = view.getLabels();
    debug("Number of labels = " + (labels != null ? labels.length : 0));

    // look for the build label and see if it is valid
    for( int i = 0; i < labels.length; i++ )
    {
      debug("Label" + i + " = " + labels[i].getName());
      if( labels[i].getName().equals(previousLabel) ) // check to see if this is the previous build label
      {
        if( labels[i].isBuildLabel() )
        {
          debug("Previous label = " + previousLabel + " | ID = " + labels[i].getID() );
          prevLabel = labels[i];
        } else {
          debug("Label (" + previousLabel + ") is not a valid build label.\n");
          break;
        }
      } else if( labels[i].getName().equals(buildLabel) ) { // checks to see if this is the current build label
        if( label == null && labels[i].isBuildLabel() )
        {
          debug("Current Label = " + buildLabel + " | ID = " + labels[i].getID() );
          label = labels[i];
        } else if( label == null ) {
          debug("Label (" + buildLabel + ") is not a valid build label.\n");
          break;
        }
      }
    }

    if( label == null ) // check to see if we found the current build label
    {
      //log("Label (" + buildLabel + ") could not be found.\n");
      clean();
      throw new Exception("Label (" + buildLabel + ") could not be found.\n");
    } else if( prevLabel == null ) { // checks to see if we found the previous build label
      //log("Label (" + previousLabel + ") could not be found.\n");
      clean();
      throw new Exception("Label (" + previousLabel + ") could not be found.\n");
    }

    // grab the root folder right now
    folder = view.getRootFolder();
  }

  /**
   * Iterates through the given folder to find the types and print them out.
   * @param f The folder to search through
   * @param type The type to search for (Either ChangeRequest or File)
   */
  protected void getItems( Folder f, String type ) throws Exception
  {
    debug("Entering getItems(Folder)");
    Folder [] folders = null;
    
    items = f.getItems(type); // grab the items for the specified type
    debug("Inside  Folder " + f.getFolderHierarchy() + " with " + items.length + " items" );
    for( int i = 0; i < items.length; i++ )
    {
      if( changeRequests && items[i] instanceof ChangeRequest )
      {
        try {
          printCR( f, (ChangeRequest) items[i] );
        } catch( Exception e ) {
          log("Error while trying to write Change Request: " + e );
          throw e;
        }
      }
      else if( items[i] instanceof File )
      {
        try {
          printFile( f, (File) items[i] );
        } catch( Exception e ) {
          log("Error while trying to write Revisions: " + e );
          throw e;
        }
      }
    }

    // recurse through the directory structure
	folders = f.getSubFolders();
    for( int i = 0; i < folders.length; i++ )
    {
      getItems( folders[i], type );
    }
  }

  /**
   * Checks to see if the Change Request has the AddressInBuild value
   * set to the current build. If it is, then the Change Request
   * is printed out.
   * @param f The Folder the change request was found in
   * @param cr The ChangeRequest to examine
   */
  protected void printCR( Folder f, ChangeRequest cr ) throws Exception
  {
    debug("Entering printCR(ChangeRequest)");
	User user = null;
    
    // make sure this change request was indeed addressed in the current build
    if( cr.getAddressedIn() != label.getID() ) return;

    // Grabs the AddressedBy user for the CR
    int userId = cr.getAddressedBy();
    if( userId != -1 ) user = starTeam.getUser( userId );

    // print the change request
    myChangeLogWriter.writeCR(f, cr, user );
    debug("CR #" + cr.getNumber() + " - Ver " + cr.getDotNotation() + " [" + ( user == null ? "unknown" : user.getName()) + "]\n\t" +  cr.getSynopsis() + "\n\n");
  }

  /**
   * Prints out the file specified if it has been added, changed, or removed between
   * the specified current build label and the previous build label.
   * @param f The folder the file was found in
   * @param file The file to check if it was changed
   */
  protected void printFile( Folder f, File file ) throws Exception
  {
    // grab the files associated with the specified build labels
    File currentFile = (File) file.getFromHistoryByLabelID( label.getID() );
    File prevFile = (File) file.getFromHistoryByLabelID( prevLabel.getID() );
	Item [] items = file.getHistory();
    boolean found = false;
    int i = 0;
    int startIndex = -1;
    int endIndex = -1;
    int userId = -1;
    User user = null;
    Item [] revisionItems = null;
        
    // was the current file found and is there a difference between the current file and
    // the previous file
    if( currentFile != null && (prevFile == null || !currentFile.getDotNotation().equals(prevFile.getDotNotation())) )
    {
      debug("\nFolder - " + f.getFolderHierarchy() );
      debug("CurrentVersion = " + currentFile.getDotNotation() + " | PrevVersion = " +
          (prevFile == null ? "none" : prevFile.getDotNotation()) );

      if( prevFile != null ) // new file??
      {

        
        // first find the current file
        for( i = 0; i < items.length; i++ )
          if( items[i].equals(currentFile) )
          {
            startIndex = i;
            break;
          }
        debug("StartIndex = " + startIndex);

        // now find the file versions bewtween the current version and the previous version
        for( ; i < items.length; i++ )
        {
          if( prevFile == null || !items[i].getDotNotation().equals(prevFile.getDotNotation()) )
          {
            userId = ((File)items[i]).getModifiedBy();
            if( userId != -1 ) user = starTeam.getUser( userId );
            debug( currentFile.getName() + " [" + (userId != -1 ? user.getName() : "unknown") +"] - " + items[i].getDotNotation() +
                 " ( " + items[i].getComment() + ")");
          }
          else
          {
            endIndex = i - 1;
            break;
          }
        }
        if( endIndex == -1 ) endIndex = items.length - 1;
        if( startIndex != -1  )
        {
          revisionItems = (Item []) java.lang.reflect.Array.newInstance( (items[0]).getClass(), endIndex - startIndex + 1);
          System.arraycopy(items, startIndex, revisionItems, 0, endIndex - startIndex + 1);
          myChangeLogWriter.writeRevision(f, (File []) revisionItems, starTeam);
        }
      } else {
        debug("****File " + currentFile.getName() + " has been added.****");
        myChangeLogWriter.writeFileAdded(f, currentFile, starTeam);
      }
    } else if( prevFile != null && currentFile == null ) { // was the file removed from the current build label
      debug("****File " + prevFile.getName() + " was removed.****");
      myChangeLogWriter.writeFileDeleted(f, currentFile, starTeam);
    }
  }

  /**
   * Tries to create the output file and load the writer class. If no writer class is
   * specified, then the default writer class will be used.
   * @throws Exception if something has gone wrong
   */
  protected void createFile( ) throws Exception
  {
    Class myClass = this.getClass().getClassLoader().loadClass(
        logWriter == null ? DEFAULT_CHANGELOGWRITER : logWriter );
    myChangeLogWriter = (ChangeLogWriter) ( myClass.newInstance() );
    myChangeLogWriter.setLogName(destFile);
    myChangeLogWriter.openFile();
  }

  /**
   * Prints out the debug string if the DEBUG variable has been set.
   * @param s The string to print out
   */
  protected void debug( String s )
  {
    log(s, org.apache.tools.ant.Project.MSG_DEBUG );
  }

/**
 * Returns the buildLabel.
 * @return String
 */
public String getBuildLabel() {
	return buildLabel;
}

/**
 * Returns the changeRequests.
 * @return boolean
 */
public boolean isChangeRequests() {
	return changeRequests;
}

/**
 * Returns the logWriter.
 * @return String
 */
public String getLogWriter() {
	return logWriter;
}

/**
 * Returns the outputFile.
 * @return String
 */
public String getDestFile() {
	return destFile;
}

/**
 * Returns the previousLabel.
 * @return String
 */
public String getPreviousLabel() {
	return previousLabel;
}

/**
 * Returns the starTeamPath.
 * @return String
 */
public String getStarTeamPath() {
	return starTeamPath;
}

/**
 * Sets the buildLabel.
 * @param buildLabel The buildLabel to set
 */
public void setBuildLabel(String buildLabel) {
	this.buildLabel = buildLabel;
}

/**
 * Sets the changeRequests.
 * @param changeRequests The changeRequests to set
 */
public void setChangeRequests(boolean changeRequests) {
	this.changeRequests = changeRequests;
}

/**
 * Sets the logWriter.
 * @param logWriter The logWriter to set
 */
public void setLogWriter(String logWriter) {
	this.logWriter = logWriter;
}

/**
 * Sets the outputFile.
 * @param outputFile The outputFile to set
 */
public void setDestFile(String outputFile) {
	this.destFile = outputFile;
}

/**
 * Sets the previousLabel.
 * @param previousLabel The previousLabel to set
 */
public void setPreviousLabel(String previousLabel) {
	this.previousLabel = previousLabel;
}

/**
 * Sets the starTeamPath.
 * @param starTeamPath The starTeamPath to set
 */
public void setStarTeamPath(String starTeamPath) {
	this.starTeamPath = starTeamPath;
}



}
