/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;

import java.io.*;
import java.util.*;

/**
 * Task to create jni header files, compile C source files and link
 * libraries. This task can take the following arguments:
 * <ul>
 * <li>srcdir
 * <li>lib
 * <li>destdir
 * <li>desthdir
 * <li>destodir
 * <li>classpath
 * </ul>
 * Of these arguments, <b>srcdir</b> is required, <b>lib</b> is recommended.
 * By default, destdir is srcdir, desthdir and destodir are destdir.
 * classpath should point to the build directory containing the target java
 * files (generally built using javac).
 * <p>
 * When this task executes, it will recursively scan the srcdir
 * looking for C source files to compile.
 * <br>
 * If a there is a java source file with the same name in the same directory,
 * a jni header file will be created in desthdir. The header filename is the
 * fully-qualified class name, where the class separators are replaced by
 * underscore characters '_' (e.g. org.fred.foo -> org_fred_foo.h).
 * <br>
 * The C file will be compiled to an object file in destodir.
 * If lib is not given, destodir/library is used instead, where the
 * library name is extracted from the first System.loadLibrary() in the
 * java source file. Yes, if lib is omitted, each C file must have a
 * corresponding java file with a valid loadLibrary call.
 * <br>
 * The lib library (or all the libraries as defined above) will be linked in
 * destdir.
 * <br>
 * This task makes its create/compile/link decision based on timestamp.
 *
 * @author Gerald Squelart <a href="mailto:gerald_squelart@hp.com">gerald_squelart@hp.com</a>
 */

public class jnicc extends MatchingTask
{
//private void print(String msg) { System.out.println("** JNICC: " + msg); }
//public void log(String msg) { print(msg); }
//public void log(String msg, int msgLevel) { print(msg); }

  public void execute() throws BuildException
  {
    // Check that there is a srcdir.
    if (src == null || src.size() == 0)
    {
        throw new BuildException("srcdir attribute must be set!", location);
    }
    String [] srcList = src.list();

    // Check destdir.
    if (destDir != null && !destDir.isDirectory())
    {
        throw new BuildException("destination directory \"" + destDir + "\" does not exist or is not a directory", location);
    }

    // Check desthdir.
    if (destHDir != null && !destHDir.isDirectory())
    {
        throw new BuildException("header destination directory \"" + destHDir + "\" does not exist or is not a directory", location);
    }

    // Check destodir.
    if (destODir != null && !destODir.isDirectory())
    {
        throw new BuildException("object destination directory \"" + destODir + "\" does not exist or is not a directory", location);
    }

    // Initialize lists.
    headerCreationClassList.removeAllElements();
    headerCreationFilenameList.removeAllElements();
    compileCList.removeAllElements();
    compileOList.removeAllElements();
    compileLList.removeAllElements();
    linkOList.removeAllElements();
    linkLList.removeAllElements();

    // First, scan the source directories and find what has to be done.

    int nb = srcList.length;

    // For each source directory...
    for (int i=0; i < nb; i++)
    {
      // Check that this source directory is valid.
      File srcDir = (File)project.resolveFile(srcList[i]);
      if (!srcDir.exists())
      {
        throw new BuildException("srcdir \"" + srcDir + "\" does not exist!", location);
      }

      // Find all the C source files.
      DirectoryScanner ds = getDirectoryScanner(srcDir);
      String[] files = ds.getIncludedFiles();

      // By default, destdir is srcdir.
      File destDir = (this.destDir != null) ? this.destDir : srcDir;
      // By default, desthdir and destodir are destdir.
      File destHDir = (this.destHDir != null) ? this.destHDir : destDir;
      File destODir = (this.destODir != null) ? this.destODir : destDir;

      // Compare timestamps with current time.
      long now = (new Date()).getTime();

      // For each C file...
      for (int j = 0; j < files.length; j++)
      {
        String filename = files[j];
        // We're only interested in C files.
        // @todo get C file extension from a property.
        if (filename.endsWith(".c"))
        {
          File srcFile = new File(srcDir, filename);

          if (srcFile.lastModified() > now)
          {
            log(
              "Warning: file modified in the future: " + filename,
              Project.MSG_WARN
            );
          }

          // The filename without the extension.
          String filenameNoExt = filename.substring(0, filename.length() - 2);
          // The corresponding class name.
          String className = filenameNoExt.replace(File.separatorChar, '.');
          // The corresponding unique file name ('_' instead of '.');
          String filenameUnique = filenameNoExt.replace(File.separatorChar, '_');
          // The library name for this C file (lib by default).
          String libName = (lib != null) ? lib : null;

          // Need to build the jni header?
          boolean buildHeader = false;
          // jni header file.
          File headerFile = new File(destHDir, filenameUnique + ".h");

          // Check the java file.
          File javaFile = new File(srcDir, filenameNoExt + ".java");
          if (!javaFile.exists())
          {
            // No java file - was 'lib' provided?
            if (lib == null)
            {
              throw new BuildException(
                "Java file \"" + filenameNoExt
                  + ".java\" does not exist! (or lib missing)",
                location
              );
            }
          }
          else
          {
            if (javaFile.lastModified() > now)
            {
              log(
                "Warning: file modified in the future: " + javaFile,
                Project.MSG_WARN
              );
            }

            if (!headerFile.exists() || javaFile.lastModified() > headerFile.lastModified())
            {
              if (!headerFile.exists())
              {
                log(
                  "Need to create " + headerFile + " from class "
                    + className + " because it does not exist",
                  Project.MSG_DEBUG
                );
              }
              else
              {
                if (headerFile.lastModified() > now)
                {
                  log(
                    "Warning: file modified in the future: " + headerFile,
                    Project.MSG_WARN
                  );
                }
                log(
                  "Need to create " + headerFile + " from class "
                    + className + " because it is out of date with respect to "
                    + javaFile,
                  Project.MSG_DEBUG
                );
              }
              headerCreationClassList.addElement(className);
              headerCreationFilenameList.addElement(headerFile.getAbsoluteFile());
              buildHeader = true;
            }

            if (!javaFile.canRead())
            {
              throw new BuildException(
                "Java file \"" + filenameNoExt + ".java\" is not readable!",
                location
              );
            }

            StreamTokenizer javaTokenizer = null;
            try
            {
              // Pfew!
              javaTokenizer = new StreamTokenizer(
                new BufferedReader(
                  new InputStreamReader(
                    new FileInputStream(
                      javaFile
                    )
                  )
                )
              );
            }
            catch(Exception ex)
            {
              throw new BuildException(
                "Unable to read Java file \"" + filenameNoExt + ".java\"!",
                location
              );
            }

            if (lib == null)
            {
              // lib not provided -> look for it in the Java source file.
              boolean notFinished = true;
              while(notFinished)
              {
                try
                {
                  switch (javaTokenizer.nextToken())
                  {
                    case StreamTokenizer.TT_WORD:
                      if (javaTokenizer.sval.startsWith("System.loadLibrary"))
                      {
                        if (javaTokenizer.nextToken() == '(' && javaTokenizer.nextToken() == '"')
                        {
                          libName = javaTokenizer.sval;
                          notFinished = false;
                        }
                        else
                        {
                          throw new BuildException(
                            "Java file \"" + filenameNoExt + ".java\" doesn't contain a valid System.loadLibrary statement!",
                            location
                          );
                        }
                      }
                      break;

                    case StreamTokenizer.TT_EOF:
                      throw new BuildException(
                        "Java file \"" + filenameNoExt + ".java\" doesn't contain a valid System.loadLibrary statement!",
                        location
                      );
                  }
                }
                catch(IOException ex)
                {
                  throw new BuildException(
                    "Unable to read Java file \"" + filenameNoExt + ".java\"!",
                    location
                  );
                }
              }
            }
          }

          // The object destination directory.
          File objDir;
          if (lib != null)
          {
            // lib provided -> simply use destodir.
            objDir = destODir;
          }
          else
          {
            // lib not provided -> use destodir/libName.
            objDir = new File(destODir, libName);
            if (!objDir.exists())
            {
              if (!objDir.mkdirs())
              {
                throw new BuildException(
                  "Directory " + objDir.getAbsolutePath()
                    + " creation was not succesful for an unknown reason",
                  location
                );
              }
              log("Created dir: " + objDir.getAbsolutePath());
            }
          }

          // The object file.
          File objectFile = new File(objDir, filenameUnique + ".obj");

          // Need to keep record of all object files, may be needed to
          // rebuild the library.
          linkOList.addElement(objectFile.getAbsolutePath());
          linkLList.addElement(libName);

          if (
            buildHeader
              || !objectFile.exists()
              || srcFile.lastModified() > objectFile.lastModified()
          )
          {
            if (buildHeader)
            {
              log(
                "Need to compile " + srcFile + " to "
                  + objectFile + " because header file "
                  + headerFile + " will be rebuilt",
                Project.MSG_DEBUG
              );
            }
            else if (!objectFile.exists())
            {
              log(
                "Need to compile " + srcFile + " because object file "
                  + objectFile + " does not exist",
                Project.MSG_DEBUG
              );
            }
            else
            {
              if (srcFile.lastModified() > now)
              {
                log(
                  "Warning: file modified in the future: " + srcFile,
                  Project.MSG_WARN
                );
              }
              log(
                "Need to compile " + srcFile
                  + " because it is out of date with respect to "
                  + objectFile,
                Project.MSG_DEBUG
              );
            }
            compileCList.addElement(srcFile.getAbsolutePath());
            compileOList.addElement(objectFile.getAbsolutePath());
            compileLList.addElement(libName);

            if (!libsToCompileList.contains(libName))
            {
              log(
                "Need to link " + libName + " at least because "
                  + objectFile + " is rebuilt",
                Project.MSG_DEBUG
              );
              libsToCompileList.addElement(libName);
            }
          }

          // Finally, the missing libraries.
          File libFile = new File(destDir, libName + ".dll");
          if (!libFile.exists() && !libsToCompileList.contains(libName))
          {
            log(
              "Need to link " + libName + " because it does not exist",
              Project.MSG_DEBUG
            );
            libsToCompileList.addElement(libName);
          }

        }
      }
    }

    // Second, do the actual job.

    if (classPath == null)
    {
      classPath = Path.systemClasspath;
    }

    // Create jni headers...
    nb = headerCreationFilenameList.size();
    if (nb > 0)
    {
      // Yes, grammar is important  :-)
      log("Creating " + nb + " header file" + (nb>1?"s":""));

      for (int i = 0; i < nb; i++)
      {
        // @todo accomodate different javah versions.
        Commandline cmd = new Commandline();

        cmd.createArgument().setValue("-jni");
        cmd.createArgument().setValue("-force");

        cmd.createArgument().setValue("-o");
        cmd.createArgument().setValue((String) headerCreationFilenameList.get(i).toString());

        cmd.createArgument().setValue("-classpath");
        cmd.createArgument().setPath(classPath);

        cmd.createArgument().setValue((String) headerCreationClassList.get(i));

        log("Running javah " + cmd.toString(), Project.MSG_DEBUG);

        com.sun.tools.javah.Main javah = new com.sun.tools.javah.Main(cmd.getArguments());
        javah.run();
      }
    }

    // Compile C files...
    nb = compileCList.size();
    if (nb > 0)
    {
      log("Compiling " + nb + " C file" + (nb>1?"s":""));

      for (int i = 0; i < nb; i++)
      {
        Commandline cmd = new Commandline();

        // @todo accomodate other compilers and platforms.
        cmd.setExecutable("cl");
        cmd.createArgument().setValue("-c"); // Compiles without linking
        cmd.createArgument().setValue("-ID:\\VisualStudio\\VC98\\Include"); // Include Windows headers
        cmd.createArgument().setValue("-ID:\\JBuilder4\\jdk1.3\\include"); // Include jdk headers
        cmd.createArgument().setValue("-ID:\\JBuilder4\\jdk1.3\\include\\win32"); // Include win jdk header
        cmd.createArgument().setValue("-I" + destHDir); // Include project headers
        cmd.createArgument().setValue(compileCList.get(i).toString());
        cmd.createArgument().setValue("-Fo" + compileOList.get(i));

        log("Running " + cmd.toString(), Project.MSG_DEBUG);

        try
        {
          Execute.runCommand(this, cmd.getCommandline());
        }
        catch (Exception ex)
        {
          throw new BuildException(
            "Error while executing " + cmd,
            location
          );
        }
      }
    }

    // Link libraries...
    nb = libsToCompileList.size();
    if (nb > 0)
    {
      log("Compiling " + nb + " librar" + (nb>1?"ies":"y"));

      for (int i = 0; i < nb; i++)
      {
        Commandline cmd = new Commandline();

        // @todo accomodate other linkers and platforms.
        cmd.setExecutable("cl");
        cmd.createArgument().setValue("-LD");
        cmd.createArgument().setValue("-Fe" + destDir + File.separator + libsToCompileList.get(i) + ".dll");
        for (int j = 0; j < linkOList.size(); j++)
        {
          if (linkLList.get(j) == libsToCompileList.get(i))
          {
            cmd.createArgument().setValue(linkOList.get(j).toString());
          }
        }

        log("Running " + cmd.toString(), Project.MSG_DEBUG);

        try
        {
          Execute.runCommand(this, cmd.getCommandline());
        }
        catch (Exception ex)
        {
          throw new BuildException(
            "Error while executing " + cmd,
            location
          );
        }
      }
    }
  }

  /**
   * Create a nested <src ...> element for multiple source path
   * support.
   *
   * @return a nested src element.
   */
  public Path createSrc()
  {
    if (src == null)
    {
      src = new Path(project);
    }
    return src.createPath();
  }

  /**
   * Set the source directory
   */
  public void setSrcdir(Path srcDir)
  {
    if (src == null)
    {
      src = srcDir;
    }
    else
    {
      src.append(srcDir);
    }
  }

  /**
   * Set the library name.
   */
  public void setLib(String lib)
  {
    this.lib = lib;
  }

  /**
   * Set the destination directory into which the libraries
   * files should be linked.
   */
  public void setDestdir(File destDir)
  {
    this.destDir = destDir;
  }

  /**
   * Set the destination directory into which the jni header
   * files should be created.
   */
  public void setDesthdir(File destHDir)
  {
    this.destHDir = destHDir;
  }

  /**
   * Set the destination directory into which the C source
   * files should be compiled.
   */
  public void setDestodir(File destODir)
  {
    this.destODir = destODir;
  }

  /**
   * Set the CLASSPATH that should contain the compiled java
   * classes
   */
  public void setClasspath(Path src)
  {
    if (classPath == null)
    {
      classPath = src;
    }
    else
    {
      classPath.append(src);
    }
  }

  /**
   * Create a nested <classpath ...>.
   */
  public Path createClasspath()
  {
    if (classPath == null)
    {
      classPath = new Path(project);
    }
    return classPath.createPath();
  }

  /**
   * Add a reference to a CLASSPATH defined elsewhere.
   */
  public void setClasspathRef(Reference r)
  {
    createClasspath().setRefid(r);
  }

  // Lists containing jni header information.
  private Vector headerCreationClassList = new Vector();
  private Vector headerCreationFilenameList = new Vector();

  // Lists containing C compilation information.
  private Vector compileCList = new Vector();
  private Vector compileOList = new Vector();
  private Vector compileLList = new Vector();

  // Lists containing library linking information.
  private Vector linkOList = new Vector();
  private Vector linkLList = new Vector();
  private Vector libsToCompileList = new Vector();

  private Path classPath = null;

  private Path src = null;
  private String lib = null;
  private File destHDir = null;
  private File destODir = null;
  private File destDir = null;
}