/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.jsp;

import java.lang.reflect.Method;

/**
  * Class to delegate to Javac.  weblogic.jspc calls main on the
  * supplied compiler class.  The main method within the Javac classes
  * call exit so this class invokes the proper compile methods in the
  * Javac classes from main.  This class is supplied as the compiler
  * class to weblogic.jspc from the WLJspc task.
  *
  * @author Michael Gilbode <a href="mailto:gilbode@bea.com">gilbode@bea.com</a
> 
  */
public class JavacHelper {

    /** Class name of the modern compiler */
    private static final String MODERN_CLASS_NAME = "com.sun.tools.javac.Main";

    /** Class name of the classic compiler */
    private static final String CLASSIC_CLASS_NAME = "sun.tools.javac.Main";

    /** Indicates whether the modern compiler is available */
    private static boolean doModern = true;
    static {
        try {
            Class.forName("com.sun.tools.javac.Main");
        } catch (ClassNotFoundException e) {
            doModern = false;
        }
    }

    /** Real compiler class to use */
    private static Class compilerClass = null;
    static {
        try {
            compilerClass = Class.forName(
                doModern ?  MODERN_CLASS_NAME : CLASSIC_CLASS_NAME);
        } catch (Exception e) {
            throw new RuntimeException("Java compiler class not found");
        }
    }

    /** Instance of the real Java compiler */
    private static Object compiler = null;
    static {
        try {
            compiler = compilerClass.newInstance();
        } catch (Exception e) {
            throw new RuntimeException("Unable to instantiate Java compiler");
        }
    }

    /** Handle to the compile method of the real Java compiler */
    private static Method compile = null;
    static {
        try {
            compile = compilerClass.getMethod("compile",
                new Class[] { (new String[] {}).getClass() } );
        } catch (Exception e) {
            throw new RuntimeException(
                "Unable to find compile method for Java compiler");
        }
    }

    /**
      * Invokes the modern Java compiler with the given arguments.
      *
      * @param args command line arguments passed to the Java compiler
      */
    private static boolean doModern(String[] args) throws Exception {
        int result = ((Integer)compile.invoke(
             compiler, new Object[] { args })).intValue();

        return result == 0;
    }

    /**
      * Invokes the classic Java compiler with the given arguments.
      *
      * @param args command line arguments passed to the Java compiler
      */
    private static boolean doClassic(String[] args) throws Exception {
        return ((Boolean)compile.invoke(
             compiler, new Object[] { args })).booleanValue();
    }

    /**
      * Main method.  Invoked with arguments suitable to pass to Javac.
      * Invokes the modern compiler if available, otherwise invokes the
      * classic compiler.
      *
      * @param args command line arguments to pass to Javac
      */
    public static void main(String[] args) throws Exception {
        boolean success = false;
        if (doModern) {
            success = doModern(args);
        } else {
            success = doClassic(args);
        }

        if (!success) {
            throw new Exception("Compile failed");
        }
    }

}
