/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.jsp;

//apache/ant imports
import org.apache.tools.ant.taskdefs.MatchingTask;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.Java;
import org.apache.tools.ant.types.Commandline;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.Reference;
import org.apache.tools.ant.Project;


//java imports
import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * <p>
 * Class to precompile JSP's using weblogic's jsp compiler (weblogic.jspc)
 * 
 * <p>
 * Task to compile JSP files into servlet classes for WebLogic.  Tested
 * under WebLogic 4.5.1 and WebLogic 5.1.  
 * The following table shows the arguments this task accepts:
 *
 * <p>
 * The task attempts to use the <code>build.compiler</code> property to
 * deterimine which java compiler should be used by
 * <code>weblogic.jspc</code>
 *
 * <p>
 * weblogic.jspc will exit if the -noexit argument is not passed in.
 * However, even with this argument passed, main exits.  Therefore,
 * weblogic.jspc.main cannot be invoked without forking first or the
 * process will exit.
 *
 * <p>
 * weblogic.jspc accepts a -compilerclass command line argument so that
 * the compiler does not fork the javac process.  Unfortunately this
 * calls main on that class and main seems to exit in the javac classes.
 * This task uses a JavacHelper class which implements a main that
 * delegates to the compile method of the Java compiler class so that
 * servlet compilation can be done in process.
 *
 * <p>
 * <table border=1>
 * <tr><th>Attribute</th><th>Description</th><th>Required</th></tr>
 * <tr>
 *   <td>classpath</td>
 *   <td>the classpath to use when invoking <code>weblogic.jspc</code></td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>src</td>
 *   <td>the value to be used as the documentRoot containing JSP files</td>
 *   <td>Yes</td>
 * </tr>
 * <tr>
 *   <td>dest</td>
 *   <td>the directory where the generated class files should go</td>
 *   <td>Yes</td>
 * </tr>
 * <tr>
 *   <td>superclass</td>
 *   <td>the superclass that the generated servlet class should extend</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>keepgenerated</td>
 *   <td>whether to keep the generated java files (default: no)</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>optimize</td>
 *   <td>
 *     whether <code>weblogic.jspc</code> compiles with optimization
 *     on (default: no)
 *   </td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>debug</td>
 *   <td>
 *     whether <code>weblogic.jspc</code> compiles debugging
 *     information into the generated servlet classs (default: no)
 *   </td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>contextpath</td>
 *   <td>
 *     context path to prepend to the simulated request URI that is
 *     munged into the classname for the generated code.  This is ignored
 *     for WebLogic 4.5.1.
 *   </td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>failonerror</td>
 *   <td>
 *    Whether the task should fail if it encounters an error
 *   (default: no)</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>verbosejavac</td>
 *   <td>
 *    Whether the task should pass -verbose to Javac
 *   (default: no).</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>verbose</td>
 *   <td>
 *    Whether the task should compile with verbose output
 *   (default: no)</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>commentary</td>
 *   <td>
 *    Whether the compiler should emit commentary
 *   (default: no)</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>encoding</td>
 *   <td>
 *    The character encoding to use.
 *   (default: the default encoding of the JDK)</td>
 *   <td>No</td>
 * </tr>
 * <tr>
 *   <td>deprecation</td>
 *   <td>
 *    Indicates whether the -deprecation flag should be passed to Javac
 *   (default: no)</td>
 *   <td>No</td>
 * </tr>
 * </table>
 *
 * <p>
 * This task requires ant to be invoked with a classpath 
 * which contains the weblogic classes as well as all application classes 
 * referenced by the JSP. The system classpath is appended when
 * <code>weblogic.jspc</code> is called, so you may 
 * choose to put everything in the classpath while calling Ant. However, 
 * since presumably the JSP's will reference classes being built by Ant, 
 * it would be better to explicitly add the classpath in the task
 * 
 * <p>
 * The task checks timestamps on the JSP's and the generated classes, 
 * and compiles only those files that have changed. 
 * 
 * <p>
 * It follows the weblogic naming convention of putting classes in 
 * <code>package</code>/[<code>contextpath</code>]/_dirName/_fileName.class 
 * for dirname/fileName.jsp</b>
 * 
 * <p>
 * <pre>
 * Example:
 * &lt;target name="jspcompile" depends="compile"&gt;
 *   &lt;wljspc src="c:\\weblogic\\myserver\\public_html" 
 *           dest="c:\\weblogic\\myserver\\classfiles" 
 *           superclass="mypackage.MyJSPBaseClass"
 *           keepgenerated="yes"
 *           optimize="yes"
 *           debug="no"
 *           contextpath="/MyWebAppName"
 *           failonerror="yes"
 *           package="myapp.jsp"&gt;
 *
 *     &lt;classpath&gt;
 *       &lt;pathelement location="${weblogic.classpath}" /&gt;
 *       &lt;pathelement path="${compile.dest}" /&gt;
 *     &lt;/classpath&gt;
 *
 *   &lt;/wljspc&gt;
 *
 * &lt;/target&gt;
 * </pre>
 * 
 * @author Avik Sengupta <a href="mailto:avik@aviksengupta.com">avik@aviksengupta.com</a> 
 * @author Michael Gilbode <a href="mailto:gilbode@bea.com">gilbode@bea.com</a>
 * 
 */
public class WLJspc extends MatchingTask
{
    /** WebLogic 4.5.1 version string */
    private static final String WEBLOGIC_451 = "4.5.1";

    /** WebLogic 5.1.0 version string */
    private static final String WEBLOGIC_510 = "5.1.0";

    /** WebLogic 6.0 version string */
    private static final String WEBLOGIC_60 = "6.0";

    /** version of WebLogic that is being run */
    private String weblogicVersion;

    /** root of compiled files tree */
    private File destinationDirectory; 

    /** root of source file tree */
    private File sourceDirectory; 

    /** package under which resultant classes will reside */
    private String destinationPackage; 

    /** name of the superclass the generated servlet should extend */
    private String superClass;

    /** flag indicating whether or not to keep the generated java files */
    private boolean keepGenerated;

    /** flag indicating whether or not to optimize the generated classes*/
    private boolean optimize;

    /** flag indicating whether or not to generate debug information 
     * for the generated classes 
     */
    private boolean debug;

    /** flag indicating whether or not to fail on compile errors */
    private boolean failOnError;

    /** flag indicating whether or not pass -verbose to javac */
    private boolean verboseJavac;

    /** flag indicating whether or not to compile with verbose output */
    private boolean verbose;

    /** flag indicating whether or not the compiler should emit
     * commentary
     */
    private boolean commentary;

    /** The character encoding to use */
    private String encoding;

    /** flag indicating whether or not the -deprecation flag should be
     * passed to Javac
     */
    private boolean deprecation;

    /** classpath used to compile the jsp files. */
    private Path compileClasspath; 

    /** context path to prepend to the simulated request URI.  Only
     * valid for WebLogic version > 5.1
     */
    private String contextPath;

    /** Indicates whether the tool must add the complete destination package
     * hierarchy on the command line or just the base
     */
    private boolean mustAddPackages;

    /** Constructor for weblogic.jspc */
    private Constructor jspcConstructor;

    /**
     * Set the classpath to be used for this compilation.
     * 
     */
    public void setClasspath(Path classpath) {
        if (compileClasspath == null) {
            compileClasspath = classpath;
        } else {
            compileClasspath.append(classpath);
        }
    }

    /**
     * Maybe creates a nested classpath element.
     */
    public Path createClasspath() {
        if (compileClasspath == null) {
            compileClasspath = new Path(project);
        }
        return compileClasspath.createPath();
    }

    /**
     * Adds a reference to a CLASSPATH defined elsewhere.
     */
    public void setClasspathRef(Reference r) {
        createClasspath().setRefid(r);
    }

    /**
     * Set the directory containing the source jsp's
     * 
     *
     * @param dirName the directory containg the source jsp's
     */
    public void setSrc(File dirName) {

        sourceDirectory = dirName;
    }

    /**
     * Set the directory containing the source jsp's
     * 
     *
     * @param dirName the directory containg the source jsp's
     */
    public void setDest(File dirName) {

        destinationDirectory = dirName;
    }

    /**
     * Set the package under which the compiled classes go
     * 
     * @param packageName the package name for the clases
     */
    public void setPackage(String packageName) {

        destinationPackage=packageName; 
    }

    /**
     * Set the class name of the superclass the servlet should extend
     * 
     * @param className class name of the superclass the servlet should extend
     */
    public void setSuperclass(String className) {

        superClass = className;
    }

    /**
     * Indicate whether the generated java file should be kept.
     * 
     * @param shouldKeep whether to keep the file ('true' or 'false')
     */
    public void setKeepgenerated(boolean shouldKeep) {

        keepGenerated = shouldKeep;
    }

    /**
     * Indicate whether to optimize the generated classes
     * 
     * @param shouldOptimize whether to optimize the file ('true' or 'false')
     */
    public void setOptimize(boolean shouldOptimize) {

        optimize = shouldOptimize;
    }

    /**
     * Indicate whether to generate debug information for the generated
     * file.
     * 
     * @param shouldDebug whether to generate debug information 
     * ('true' or 'false')
     */
    public void setDebug(boolean shouldDebug) {

        debug = shouldDebug;
    }

    /**
     * Indicate whether to fail on compile errors
     * 
     * @param shouldFail whether to fail on compile errors
     */
    public void setFailonerror(boolean shouldFail) {

        failOnError = shouldFail;
    }

    /**
     * Context path to prepend to the simulated URI that is munged into
     * the classname for the generated code.
     * 
     * @param pathName context URI
     */
    public void setContextpath(String pathName) {

        if (weblogicVersion.equals(WEBLOGIC_451)) {
            log("Warning: contextPath not valid with this version " +
                "of WebLogic... ignoring", Project.MSG_WARN);
            return;
        }
        contextPath = pathName;
    }

    /**
      * Indicates whether to invoke java compiler with -verbose (default
      * false)
      *
      * @param javacShouldBeVerbose whether javac should be verbose
      */
    public void setVerbosejavac(boolean javacShouldBeVerbose) {
        verboseJavac = javacShouldBeVerbose;
    }

    /**
      * Indicates whether to compile with verbose output
      *
      * @param shouldBeVerbose whether javac should be verbose
      */
    public void setVerbose(boolean shouldBeVerbose) {
        verbose = shouldBeVerbose;
    }

    /**
      * Indicates whether to emit compiler commentary
      *
      * @param shouldEmitCommentary indicates whether the compiler
      * should emit commentary
      */
    public void setCommentary(boolean shouldEmitCommentary) {
        commentary = shouldEmitCommentary;
    }

    /**
      * Sets the character encoding to be used.
      *
      * @param characterEncoding the character encoding to use
      */
    public void setEncoding(String characterEncoding) {
        encoding = characterEncoding;
    }

    /**
      * Indicates whether the -deprecation flag should be passed to
      * Javac
      *
      * @param shouldShowDeprecated flag indicating whether -deprecation
      * flag should be passed to Javac
      */
    public void setDeprecation(boolean shouldShowDeprecated) {
        deprecation = shouldShowDeprecated;
    }

    /**
      * Initializes the tool.
      */
    public void init() throws BuildException {
        weblogicVersion = getWeblogicVersion();
        if (weblogicVersion.equals(WEBLOGIC_451)) {
            mustAddPackages = true;
        }
        jspcConstructor = getJspcConstructor();
    }

    /**
      * Execute the task
      */
    public void execute() throws BuildException {
                               
        validate();

        DirectoryScanner ds = getDirectoryScanner(sourceDirectory);
        String[] files = ds.getIncludedFiles();

        Vector args = new Vector();
        args.addElement("-noexit");
        addCommandLine(args);
        String[] argArray = new String[args.size() + 3];
        for (int j = 0; j < argArray.length - 3; ++j) {
            argArray[j] = (String)args.elementAt(j);
        }

        Vector filesToDo = scanDir(files);
        log("Compiling " + filesToDo.size() + " JSP files");
            
        for (int i=0;i<filesToDo.size();i++) {
            File jspFile = new File((String) filesToDo.elementAt(i));
            String packageName = destinationPackage;

            if (mustAddPackages) {
                String parent = jspFile.getParent();
                if (parent != null) {
                    if (parent.startsWith(sourceDirectory.toString())) {
                        parent = parent.substring(
                            sourceDirectory.toString().length());
                    }

                    StringBuffer translated = new StringBuffer();

                    StringTokenizer tok = 
                        new StringTokenizer(parent, File.separator);
                    while (tok.hasMoreTokens()) {
                        translated.append("._");
                        translated.append(tok.nextToken());
                    }
                    packageName = 
                        destinationPackage + translated.toString();
                }
            }

            argArray[argArray.length - 3] = "-package";
            argArray[argArray.length - 2] = packageName;
            argArray[argArray.length - 1] = 
                new File(sourceDirectory, jspFile.toString()).getAbsolutePath();

            try {
                weblogic.jspc compiler = (weblogic.jspc)
                    jspcConstructor.newInstance(
                        new Object[] { argArray });
                compiler.run();
            } catch (Exception e) {
                if (failOnError) {
                    throw new BuildException(jspFile + " failed to compile", e);
                }
                log(jspFile + " failed to compile", Project.MSG_WARN);
            }
        }
    }

    /**
     * Scans the directory looking for source files to be compiled.
     *
     * @return the list of files to be compiled
     */
    protected Vector scanDir(String files[]) {

        Vector filesToDo = new Vector();
        long now = (new Date()).getTime();
        String pack = "";

        for (int i = 0; i < files.length; i++) {
            File srcFile = new File(sourceDirectory, files[i]);

            if (srcFile.getName().indexOf(".jsp") == -1) {
                continue;
            }

            StringBuffer destination = new StringBuffer();
            destination.append(destinationPackage);

            if (contextPath != null && contextPath.length() > 0) {
                destination.append(File.separator);
                destination.append("_");

                String path = Path.translateFile(contextPath);
                while (path.length() > 0 && 
                    path.charAt(0) == File.separatorChar) {

                    path = ((path.length() == 1) ? "" 
                        : path.substring(1));
                }
                destination.append(path);
            }

            String parent = srcFile.getParent();
            if (parent != null) {
                if (parent.startsWith(sourceDirectory.toString())) {
                    parent = parent.substring(
                        sourceDirectory.toString().length());
                }
                StringBuffer translated = new StringBuffer();
                StringTokenizer tok = 
                    new StringTokenizer(parent, File.separator);
                while (tok.hasMoreTokens()) {
                    translated.append(File.separator);
                    translated.append("_");
                    translated.append(tok.nextToken());
                }
                destination.append(translated.toString());
            }

            destination.append(File.separator);
            destination.append("_");
            destination.append(srcFile.getName().substring(0,
                 srcFile.getName().indexOf(".jsp")));
            destination.append(".class");


            File classFile = new File(
                destinationDirectory, destination.toString());

            if (srcFile.lastModified() > now) {
                log("Warning: file modified in the future: " +
                    files[i], Project.MSG_WARN);
            }
            if (srcFile.lastModified() > classFile.lastModified()) {
                filesToDo.addElement(files[i]);
                log("Recompiling File "+files[i],Project.MSG_VERBOSE);
            }
        }
        return filesToDo;
    }


    /**
     * Determines the WebLogic version to run against and returns it.
     *
     * @return weblogic version (WEBLOGIC_510|WEBLOGIC451|WEBLOGIC_60)
     * @throws BuildException if WebLogic version cannot be determined
     * or is unrecognized.
     */
    private String getWeblogicVersion() {

        Class versionClass = null;
        String version = null;

        try {
            versionClass = Class.forName("utils.version");
        } catch (ClassNotFoundException e) {
            throw new BuildException("Cannot determine weblogic version.  " +
                "Ensure weblogic classes are in your classpath", e);
        }
        try {
            Method getVersions =
                versionClass.getDeclaredMethod("getVersions", new Class[0]);

            String versionString = 
                (String)getVersions.invoke(null, new Object[0]);

            if (versionString.indexOf("4.5.1") != -1) {
                version = WEBLOGIC_451;
            } else if (versionString.indexOf("5.1.0") != -1) {
                version = WEBLOGIC_510;
            } else if (versionString.indexOf("6.0") != -1) {
                version = WEBLOGIC_60;
            } else {
                throw new BuildException("Cannot determine weblogic version");
            }
        } catch (NoSuchMethodException e) {
            throw new BuildException("Cannot determine weblogic version", e);
        } catch (InvocationTargetException e) {
            throw new BuildException("Cannot determine weblogic version", e);
        } catch (IllegalAccessException e) {
            throw new BuildException("Cannot determine weblogic version", e);
        }
        return version;
    }

    /**
      * Uses reflection to get the constructor of the weblogic.jspc
      * class that takes a String array as an argument.
      *
      * @return weblogic.jspc constructor
      * @throws BuildException if the constructor cannot be obtained
      */
    private Constructor getJspcConstructor() throws BuildException {
        try {
            Class jspcClass = Class.forName("weblogic.jspc");
            Class jspcArgClass = Class.forName("[Ljava.lang.String;");
            Constructor constructor = 
                jspcClass.getConstructor(new Class[] { jspcArgClass });
            return constructor;
        } catch (Exception e) {
            throw new BuildException("Cannot find the constructor for " +
                "weblogic.jspc.  Check your classpath", e);
        }
    }

    /**
     * Performs validation to ensure the tool is properly configured
     * before running.
     */
    private void validate() throws BuildException {
        if (!destinationDirectory.isDirectory()) {
            throw new BuildException("destination directory " + 
                destinationDirectory.getPath() + " is not valid");
        }
        if (!sourceDirectory.isDirectory()) {
            throw new BuildException("src directory " + 
                sourceDirectory.getPath() + " is not valid");
        }
        if (destinationPackage == null) {
            throw new BuildException("package attribute must be present.", 
                location);
        }
    }

    /**
     * Builds the command-line to weblogic.jspc and adds it to the given
     * task.
     *
     * @param task the task to build the command line for
     *
     */
    private void addCommandLine(Vector args) {

        args.addElement("-d");
        args.addElement(destinationDirectory.getAbsolutePath());
        args.addElement("-docroot");
        args.addElement(sourceDirectory.getAbsolutePath());

        addCompilerArgs(args);

        if (superClass != null && superClass.length() > 0) {
            args.addElement("-superclass");
            args.addElement(superClass);
        }

        if (keepGenerated) {
            args.addElement("-keepgenerated");
        }

        if (debug) {
            args.addElement("-g");
        }

        if (optimize) {
            args.addElement("-O");
        }

        args.addElement("-classpath");
        args.addElement(getCompileClasspath().toString());

        if (contextPath != null && contextPath.length() > 0) {
            args.addElement("-contextPath");
            args.addElement(contextPath);
        }

        if (verboseJavac || verbose) {
            args.addElement("-verbose");
        }

        if (verboseJavac) {
            args.addElement("-verboseJavac");
        }

        if (encoding != null && encoding.length() > 0) {
            args.addElement("-encoding");
            args.addElement(encoding);
        }

        if (deprecation) {
            args.addElement("-deprecation");
        }
    }

    /**
     * Figures out the java compiler to use based on the project
     * properties and adds the command line arguments to the given task.
     *
     * @param task the task to add the compiler arguments to
     *
     */
    private void addCompilerArgs(Vector args) {
        String compiler = project.getProperty("build.compiler");

        if (compiler == null) {
            if (Project.getJavaVersion().startsWith("1.3")) {
                compiler = "modern";
            } else {
                compiler = "classic";
            }
        }

        if (compiler.equalsIgnoreCase("classic") ||
            compiler.equalsIgnoreCase("modern")) {
            args.addElement("-compilerclass");
            args.addElement("org.apache.tools.ant.taskdefs.optional.jsp.JavacHelper");
        } else if (compiler.equalsIgnoreCase("jikes")) {
            args.addElement("-compiler");
            args.addElement("jikes");
        }
    }

    /**
     * Builds the compile classpath.
     */
    private Path getCompileClasspath() {
        Path classpath = new Path(project);

        classpath.setLocation(destinationDirectory);

        if (compileClasspath != null) {
            classpath.addExisting(compileClasspath);
        }

        classpath.addExisting(Path.systemClasspath);
        classpath.setLocation(sourceDirectory);

        if (System.getProperty("java.vendor").toLowerCase().indexOf(
             "microsoft") >= 0) {

            FileSet msZipFiles = new FileSet();
            msZipFiles.setDir(new File(System.getProperty("java.home") + 
                 File.separator + "Packages"));
            msZipFiles.setIncludes("*.ZIP");
            classpath.addFileset(msZipFiles);
        } else if (Project.getJavaVersion() == Project.JAVA_1_1) {
            classpath.addExisting(new Path(null,
                 System.getProperty("java.home")
                 + File.separator + "lib"
                 + File.separator
                 + "classes.zip"));
        } else {
            // JDK > 1.1 seems to set java.home to the JRE directory.
            classpath.addExisting(new Path(null,
                 System.getProperty("java.home")
                 + File.separator + "lib"
                 + File.separator + "rt.jar"));
            // Just keep the old version as well and let addExistingToPath
            // sort it out.
            classpath.addExisting(new Path(null,
                 System.getProperty("java.home")
                 + File.separator +"jre"
                 + File.separator + "lib"
                 + File.separator + "rt.jar"));
        }

        return classpath;
    }
}
