
package org.apache.tools.ant.gui.xml;

import org.w3c.dom.Attr;
import org.w3c.dom.CharacterData;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.ProcessingInstruction;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.Writer;

class DOMNodeSerializer {

    private DOMNodeFactory _factory;
    
    public DOMNodeSerializer(DOMNodeFactory factory) {
        _factory = factory;
    }
    
    public void write(DOMNode domNode, Writer writer) throws SAXException {
        DataWriter w = new DataWriter(writer);
        w.setIndentStep(4);
        writeNode(w, domNode.getImpl());
    }
    
    protected void internalWrite(DataWriter w, Node n) {
        try {
            Node node = n;
            while(node != null) {

                if (node.getNodeType() == Node.ATTRIBUTE_NODE) {
                    continue;
                }
                
                writeNode(w, node);
                
                node = node.getNextSibling();
                if (node == null) {
                    Node parent = n.getParentNode();
                    if (parent != null) {
                        node = parent.getFirstChild();
                    }
                }
                
                if (node == n) {
                    break;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    protected void writeNode(DataWriter w, Node node)
            throws SAXException {
        
        Attributes attrs = new NamedDOMNodeMap(_factory,
            node.getAttributes() ).convertToAttributes();

        String uri = (node.getNamespaceURI() == null) ? "" :
            node.getNamespaceURI();

        String localname = (node.getLocalName() == null) ? "" :
            node.getLocalName();

        String prefix = (node.getPrefix() == null) ? "" :
            node.getPrefix();

        String value = (node.getNodeValue() == null) ? "" :
            node.getNodeValue();

        if (node.hasChildNodes()) {
            writeNodeStart (w, node, node.getNodeType(), uri,
                localname, prefix, attrs, value);
            internalWrite(w, node.getFirstChild());
            writeNodeStop (w, node, node.getNodeType(), uri,
                localname, prefix);
        } else {
            writeNode (w, node, node.getNodeType(), uri,
                localname, prefix, attrs, value);
        }
    }
    
    protected void writeNodeStart(DataWriter w, Node node, int type,
            String uri, String localname, String prefix, Attributes attrs,
            String content) throws SAXException {

        switch (type) {
            case Node.ELEMENT_NODE:
                w.startElement (uri, localname, prefix, attrs);
                break;
                
            case Node.DOCUMENT_NODE:
                w.startDocument();
                
            case Node.TEXT_NODE:
            case Node.CDATA_SECTION_NODE:
            case Node.PROCESSING_INSTRUCTION_NODE:
            case Node.COMMENT_NODE:
                writeNode(w, node, type, uri, localname, prefix, attrs,
                    content);
                break;
        }
    }
    
    protected void writeNodeStop(DataWriter w, Node node, int type, String uri,
            String localname, String prefix) throws SAXException {

        switch (type) {
            case Node.ELEMENT_NODE:
                w.endElement (uri, localname, prefix);
                break;
                
            case Node.DOCUMENT_NODE:
                w.endDocument();
                break;
        }
    }
    
    protected void writeNode(DataWriter w, Node node, int type, String uri,
            String localname, String prefix, Attributes attrs,
            String content) throws SAXException {

        switch (type) {
            case Node.ELEMENT_NODE:
                if (content.equals("")) {
                    w.emptyElement (uri, localname, prefix, attrs);
                } else {
                    w.dataElement (uri, localname, prefix, attrs, content);
                }
                break;
                
            case Node.CDATA_SECTION_NODE:
                CharacterData cd = (CharacterData) node;
                w.characters(cd.getData());
                break;
                
            case Node.COMMENT_NODE:
                CharacterData comment = (CharacterData) node;
                w.comment(comment.getData());
                break;
                
            case Node.PROCESSING_INSTRUCTION_NODE:
                ProcessingInstruction pi = (ProcessingInstruction) node;
                w.processingInstruction(pi.getTarget(), pi.getData());
                break;
        }
    }
}
