/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.pkgmk;

import java.io.*;
import java.util.*;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.types.Commandline;

public class Pkgmk extends Task {
    private File prototype;
    private String pkgName;
    private String destDir;
    private String srcDir;
    private String outputFile;
    private boolean overwrite = true;
    private String workingDir = "pkgmkWorkingDir";
    private String installBase;
    private String requestScript;
    private String checkInstallScript;
    private String preInstallScript;
    private String postInstallScript;
    private String preRemoveScript;
    private String postRemoveScript;
    private String copyrightFile;
    private String dependFile;
    private Vector protoEntries = new Vector();
    private Vector pkginfoEntries = new Vector();
    private LogStreamHandler streamHandler;
    private String pkgVersion;
    private String pkgDesc;
    private String pkgArch = "sparc";
    private String pkgCategory = "application";
    private String pkgBaseDir = "/opt";
    private String owner;
    private String group;

    private void clearWorkingDir(File dir) {
        if(dir == null) {
            dir = new File(workingDir);
        }
        String files[] = dir.list();
        for(int i=0; files != null && i < files.length; i++) {
            new File(files[i]).delete();
        }        
    }

    private void createPkginfo() throws BuildException {
        try {
            BufferedWriter pkginfo = new BufferedWriter(new FileWriter(workingDir + "/pkginfo"));
            pkginfo.write("PKG=" + pkgName);
            pkginfo.newLine();
            pkginfo.write("ARCH=" + pkgArch);
            pkginfo.newLine();
            pkginfo.write("NAME=" + pkgDesc);
            pkginfo.newLine();
            pkginfo.write("VERSION=" + pkgVersion);
            pkginfo.newLine();
            pkginfo.write("CATEGORY=" + pkgCategory);
            pkginfo.newLine();
            pkginfo.write("BASEDIR=" + pkgBaseDir);
            pkginfo.newLine();
            int nEntries = pkginfoEntries.size();
            if(nEntries > 0) {
                for(int i=0; i < nEntries; i++) {
                    PkginfoEntry entry = (PkginfoEntry)pkginfoEntries.elementAt(i);
                    pkginfo.write(entry.getName() + "=" + entry.getValue());
                    pkginfo.newLine();
                }
            }
            pkginfo.close();
        } catch(IOException e) {
            throw new BuildException("I/O Error while creating pkginfo file.", e);
        }
    }

    private void createPrototype() throws BuildException {
        prototype = new File(workingDir, "prototype");
        BufferedWriter proto;
        try {
            FileOutputStream fos = new FileOutputStream(prototype);
            proto = new BufferedWriter(new OutputStreamWriter(fos));
            proto.write("i pkginfo");
            proto.newLine();

            if(requestScript != null) {
                copyFile(requestScript, workingDir, "request");
                proto.write("i request");
                proto.newLine();
            }
            if(checkInstallScript != null) {
                copyFile(checkInstallScript, workingDir, "checkinstall");
                proto.write("i checkinstall");
                proto.newLine();
            }
            if(preInstallScript != null) {
                copyFile(preInstallScript, workingDir, "preinstall");
                proto.write("i preinstall");
                proto.newLine();
            }
            if(postInstallScript != null) {
                copyFile(postInstallScript, workingDir, "postinstall");
                proto.write("i postinstall");
                proto.newLine();
            }
            if(preRemoveScript != null) {
                copyFile(preRemoveScript, workingDir, "preremove");
                proto.write("i preremove");
                proto.newLine();
            }
            if(postRemoveScript != null) {
                copyFile(postRemoveScript, workingDir, "postremove");
                proto.write("i postremove");
                proto.newLine();
            } 
            if(dependFile != null) {
                copyFile(dependFile, workingDir, "depend");
                proto.write("i depend");
                proto.newLine();
            }
            if(copyrightFile != null) {
                copyFile(copyrightFile, workingDir, "copyright");
                proto.write("i copyright");
                proto.newLine();
            }

            for(int i=0; i < protoEntries.size(); i++) {
                proto.write(((PrototypeEntry)protoEntries.elementAt(i)).getText());
                proto.newLine();
            }
            
            proto.flush();

            if(srcDir != null) {
                // Now, run pkgproto
                Commandline cmd = new Commandline();
                cmd.setExecutable("pkgproto");
                if(installBase != null) {
                    cmd.addArguments(new String [] { srcDir + "=" + installBase });
                } else {
                    cmd.addArguments(new String [] { srcDir });
                }
                System.out.println("pkgproto cmd: " + cmd.toString());
                PumpStreamHandler psh = new PumpStreamHandler(fos);
                Execute exe = new Execute(psh);
                exe.setAntRun(project);
                exe.setCommandline(cmd.getCommandline());
                exe.execute();
                proto.close();
                int exitCode = exe.getExitValue();
                if(exitCode != 0) {
                    throw new BuildException("Execution of 'pkgproto' failed with exit code: " +
                                             exitCode);
                }
            }

            // Now, use "awk" to change the owner and group of the files
            // in the prototype file
            modifyPrototype(prototype);
        } catch(IOException e) {
            throw new BuildException("I/O Error during prototype generation: " + e.getMessage(), e);
        }
    }

    /**
    * Modify the owner and group in the prototype file 
    */
    private void modifyPrototype(File prototypeFile)  throws BuildException {
        int exitCode = 0;
        File tempPrototypeFile = new File(prototypeFile.getParent(), prototypeFile.getName() + ".temp");

        if(owner != null || group != null) {
            Commandline cmd [] = new Commandline[2];
            cmd[0] = new Commandline();
            cmd[0].setExecutable("awk");
            // If the prototype entry has more than 4 fields, then the fifth field
            // is the owner and the sixth field is the group for the file/directory.
            StringBuffer awkScript = new StringBuffer("NF>4{");
            // make the awk script replace the owner field of the prototype entries.
            if(owner != null) {
                awkScript.append("$5=\"" + owner + "\";");
            }
            // make the awk script replace the group field of the prototype entries.
            if(group != null) {
                awkScript.append("$6=\"" + group + "\";");
            }
            awkScript.append("print}");
            cmd[0].addArguments(new String [] { awkScript.toString() });
            cmd[0].addArguments(new String [] {prototypeFile.getAbsolutePath()});
            cmd[1] = new Commandline();
            cmd[1].setExecutable("awk");
            cmd[1].addArguments(new String [] { "NF<4{print}" });
            cmd[1].addArguments(new String [] {prototypeFile.getAbsolutePath()});
            
            FileOutputStream fos = null;        
            try
            {
                fos = new FileOutputStream(tempPrototypeFile);   
                PumpStreamHandler psh = new PumpStreamHandler(fos);     
                for(int x = 0 ; x < cmd.length ; x++) {
                    Execute exe = new Execute(psh);
                    exe.setAntRun(project);
    
                    exe.setCommandline(cmd[x].getCommandline());
                    exe.execute();
                    exitCode = exe.getExitValue();
                    if(exitCode != 0) {
                        throw new BuildException("Execution of 'awk' failed with exit code: " +
                                                 exitCode);
                    } 
                } 
                fos.flush();          
                fos.close(); 
                
                copyFile(tempPrototypeFile.getAbsolutePath(), 
                            prototypeFile.getParent(), 
                            prototypeFile.getName());   
                tempPrototypeFile.delete();                  
            } catch(IOException e) {
                throw new BuildException("I/O Error during prottype modification(using awk): " + e.getMessage(), e);
            } 
        }
    }

    /**
    *
    */
    private void copyFile(String srcFilename, String destDir, String destFilename) 
        throws IOException {
        
        Commandline cmd = new Commandline();
        cmd.setExecutable("cp");
        if(destFilename == null) {
            destFilename = new File(srcFilename).getName();
        }
        cmd.addArguments(new String[] { srcFilename, destDir + '/' + destFilename });
        Execute exe = new Execute();
        exe.setAntRun(project);
        exe.setCommandline(cmd.getCommandline());
        exe.execute();
        int exitCode = exe.getExitValue();
        if(exitCode != 0) {
            throw new BuildException("Execution of 'cp' failed with exit code: " +
                                     exitCode);
        }
    }

    private void buildPackage() throws BuildException {
        try {
            Commandline pkgMk = new Commandline();
            pkgMk.setExecutable("pkgmk");
            if(overwrite) {
                pkgMk.addArguments(new String[] { "-o" });
            }
            pkgMk.addArguments(new String[] { "-f", prototype.getAbsolutePath() });
            pkgMk.addArguments(new String[] { "-d", workingDir });
            /*if(installBase == null) {
                pkgMk.addArguments(new String[] { srcDir });
            } else {
                pkgMk.addArguments(new String[] { srcDir + '=' + installBase });
            }*/
            Execute exe = new Execute(streamHandler);
            exe.setAntRun(project);
            //exe.setWorkingDirectory(topDir);
            exe.setCommandline(pkgMk.getCommandline());
            exe.execute();
            int exitCode = exe.getExitValue();
            if(exitCode != 0) {
                throw new BuildException("Execution of 'pkgmk' failed with exit code: " +
                                         exitCode);
            }
            
        } catch(IOException e) {
            throw new BuildException("I/O Error during pkgmk: " + e.getMessage(), e);
        }
    }

    private void transferPackage() throws BuildException {
        try {
            Commandline pkgTrans = new Commandline();
            pkgTrans.setExecutable("pkgtrans");
            pkgTrans.addArguments(new String[] { "-s", workingDir, destDir + "/" +
                                               outputFile, pkgName });
            Execute exe = new Execute(streamHandler, null);
            exe.setAntRun(project);
            //exe.setWorkingDirectory(topDir);
            exe.setCommandline(pkgTrans.getCommandline());
            exe.execute();
            int exitCode = exe.getExitValue();
            if(exitCode != 0) {
                throw new BuildException("Execution of 'pkgtrans' failed with exit code: " +
                                         exitCode);
            }
        } catch(IOException e) {
            throw new BuildException("I/O Error during pkgtrans: " + e.getMessage(), e);
        }
    }
    
   
    public void execute() throws BuildException {
        try {
            streamHandler = new LogStreamHandler(this, Project.MSG_INFO,
                                                 Project.MSG_WARN);
            File dir = new File(workingDir);
            if(dir.exists()) {
                clearWorkingDir(dir);
            } else {
                dir.mkdir();
            }
            if(prototype == null) {
                createPrototype();
            } else {
                copyFile(prototype.getAbsolutePath(), workingDir, null);
            }
            createPkginfo();
            buildPackage();
            transferPackage();
            //clearWorkingDir(null);
        } catch(IOException e) {
            throw new BuildException("I/O Error: " + e.getMessage(), e);
        }
    }

    public void setPrototype(String proto) {
        prototype = new File(proto);
    }

    public void setPkgname(String pkgName) {
        this.pkgName = pkgName;
    }

    public void setOutput(String outputFile) {
        this.outputFile = outputFile;
    }

    public void setPkgdesc(String desc) {
        pkgDesc = desc;
    }

    public void setPkgcategory(String category) {
        pkgCategory = category;
    }

    public void setPkgarch(String arch) {
        pkgArch = arch;
    }

    public void setPkgbasedir(String base) {
        pkgBaseDir = base;
    }

    public void setSrcdir(String srcDir) {
        this.srcDir = srcDir;
    }

    public void setDestdir(String destDir) {
        this.destDir = destDir;
    }

    public void setOverwrite(boolean overwrite) {
        this.overwrite = overwrite;
    }

    public void setWorkingdir(String workingDir) {
        this.workingDir = workingDir;
    }

    public void setPkgversion(String ver) {
        this.pkgVersion = ver;
    }

    public void setRequestscript(String request) {
        this.requestScript = request;
    }

    public void setPreinstallscript(String preinst) {
        this.preInstallScript = preinst;
    }

    public void setPostinstallscript(String postinst) {
        this.postInstallScript = postinst;
    }

    public void setPreremovescript(String prerm) {
        this.preRemoveScript = prerm;
    }

    public void setPostremovescript(String postrm) {
        this.postRemoveScript = postrm;
    }

    public void setCheckinstallscript(String checkinst) {
        this.checkInstallScript = checkinst;
    }

    public void setCopyrightfile(String copyright) {
        this.copyrightFile = copyright;
    }

    public void setDependFile(String depend) {
	this.dependFile = depend;
    }

    public void addPrototypeEntry(PrototypeEntry entry) {
        protoEntries.addElement(entry);
    }

    public void addPkginfoEntry(PkginfoEntry entry) {
        pkginfoEntries.addElement(entry);
    }

    public void setInstallBase(String installBase) {
        this.installBase = installBase;
    }

    public void setOwner(String owner) {
        this.owner = owner;
    }

    public void setGroup(String group) {
        this.group = group;
    }
}

