/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.genjar;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;
import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;

/**
 * Driver class for the gen-jar task.<p>
 * This class is instantiated when Ant encounters the &lt;genjar&gt;
 * element.
 *
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */

public class GenJar extends Task
{
	private boolean      trace       = false;
	private List         jarSpecs    = new ArrayList( 32 );
	private Mft          mft         = new Mft();
	private Path         classpath   = null;
	private ClassFilter  classFilter = null;
	private File         jarFile     = null;
	private PathResolver resolvers[] = null;
	private static Set   resolved    = new HashSet();
	private Logger       logger      = null;
	
	
	public GenJar()
	{
		setTaskName( "GenJar" );
	}
	
	public void execute()
		throws BuildException
	{
		long start = System.currentTimeMillis();
		
		logger = new Logger( getProject() );
		if ( classFilter == null )
			classFilter = new ClassFilter( logger );

		logger.verbose( "@app-name@ Ver: @version@" );
		log( "Generating jar: " + jarFile );
			//
			// check params
			//
		if ( mft == null )
			throw new BuildException( "no manifest specified" );
		if ( jarFile == null )
			throw new BuildException( "no jar file specified" );
			//
			// set up the classpath & resolvers - file/jar/zip
			//
		try {
			if ( classpath == null )
				classpath = Path.systemClasspath;
			else
				classpath.addExisting( Path.systemClasspath );
			logger.debug( "Initializing Path Resolvers" );
			initPathResolvers();
		} catch ( IOException ioe ) {
			throw new BuildException( "unable to process classpath: " + ioe );
		}
			//
			// prep the manifest
			//
  		mft.execute( logger );
			//
			// run over all the resource and clsss specifications
			// given in the project file
			// resources are resolved to full path names while
			// class specifications are exploded to dependancy
			// graphs - when done, getJarEntries() returns a list
			// of all entries generated by this JarSpec
			//
		List entries = new LinkedList();

		for ( Iterator it = jarSpecs.iterator(); it.hasNext(); ) {
			JarSpec js = (JarSpec)it.next();
			try {
				js.resolve( this );
			} catch ( IOException ioe ) {
				throw new BuildException( "Can't resolve: " + ioe );
			}
			entries.addAll( js.getJarEntries() );
		}

			//
			// we have all the entries we're gonna jar - the manifest
			// must be fully built prior to jar generation, so run over
			// each entry and and add it to the manifest
			//
		for ( Iterator it = entries.iterator(); it.hasNext(); ) {
			JarEntrySpec jes = (JarEntrySpec)it.next();
			if ( jes.getSourceFile() == null ) {
				try {
					InputStream is = resolveEntry( jes );
					if ( is != null )
						is.close();
				} catch ( Exception e ) {
					e.printStackTrace();
					throw new BuildException( e.toString() );
				}
			}
			mft.addEntry( jes.getJarName(), jes.getAttributes() );
		}

		byte[] buff = new byte[ 4096 ]; // stream copy buffer 
		int len;

  		try {			
			JarOutputStream jout =
				new JarOutputStream(
					new FileOutputStream( jarFile ), mft.getManifest() );

			for ( Iterator it = entries.iterator(); it.hasNext(); ) {
				JarEntrySpec jes = (JarEntrySpec)it.next();
				JarEntry entry = new JarEntry( jes.getJarName() );
				InputStream is = resolveEntry( jes );

				if ( is == null ) {
					logger.error( "Unable to locate previously resolved resource" );
					logger.error( "       Jar Name:" + jes.getJarName() );
					logger.error( " Resoved Source:" + jes.getSourceFile() );
					throw new BuildException( "jar component not found (" +
											  jes.getJarName() + ')' );
				}
				jout.putNextEntry( entry );
				while ( ( len = is.read( buff, 0, buff.length ) ) != -1 )
					jout.write( buff, 0, len );
				jout.closeEntry();
				logger.verbose( "Added: " + jes.getJarName() );
			}
			jout.close();
			
		} catch ( FileNotFoundException fnfe ) {
			throw new BuildException( "unable to access jar file (" + jarFile +
									  ") msg:" + fnfe );
		} catch ( IOException ioe ) {
			throw new BuildException( "unable to create jar:" + ioe );
		}
		long end = System.currentTimeMillis();
		log( "Jar Generated (" + (end-start) + " ms)" );
	}
		/**
		 * creates a 'classpath' child
		 */
    public Path createClasspath()
	{
		Path t = new Path( project );
		if ( classpath != null )
			t.addExisting( classpath );
		return classpath = t;
	}

	public Object createClass()
	{
		ClassSpec cs = new ClassSpec();
		jarSpecs.add( cs );
		return cs;
	}
		/**
		 * create manifest attribute
		 */
	public Object createManifest()
	{
		return mft;
	}

	public Object createResource()
	{
		Resource rsc = new Resource( project );
		jarSpecs.add( rsc );
		return rsc;
	}

	public Object createClassfilter()
	{
		if ( classFilter == null ) 
			classFilter = new ClassFilter( new Logger( project ) );
		return classFilter;
	}

	public void setTrace( String t )
	{
		t = t.toLowerCase();
		if ( "on".equals( t ) || "yes".equals( t ) )
			trace = true;
		else
			trace = false;
	}

	public void setJarfile( String jname )
	{
		jarFile = new File( jname );
	}	
		//
		// TODO: path resolution needs to move to its own class
		//
	
	void initPathResolvers()
		throws IOException
	{
		List     l = new ArrayList( 32 );
		String[] pc = classpath.list();

		for ( int i = 0; i < pc.length; ++i ) {
			File f = new File( pc[ i ] );
			if ( ! f.exists() )
				continue;

			if ( f.isDirectory() )
				l.add( new FileResolver( f, logger ) );
			else if ( f.getName().endsWith( ".jar" ) )
				l.add( new JarResolver( f, logger ) );
			else if ( f.getName().endsWith( ".zip" ) )
				l.add( new ZipResolver( f, logger ) );
			else
				throw new BuildException( "unknown classpath component" );
		}
		resolvers = (PathResolver[])l.toArray( new PathResolver[ 0 ] );
	}


	InputStream resolveEntry( JarEntrySpec spec )
		throws IOException
	{
		InputStream is = null;
		for ( int i = 0; i < resolvers.length; ++i ) {
			is = resolvers[ i ].resolve( spec );
			if ( is != null )
				return is;
		}
		return null;
	}		
		/**
		 * Resolves a partial file name against the classpath elements
		 *
		 * @return An InputStream open on the named file or null
		 */
	InputStream resolveEntry( String cname )
		throws IOException
	{
		InputStream is = null;
		
		for ( int i = 0; i < resolvers.length; ++i ) {
			is = resolvers[ i ].resolve( cname );
			if ( is != null )
				return is;
		}
		return null;
	}


		//=====================================================================
		// TODO: class dependancy determination needs to move to either its own
		// class or to ClassSpec
		//=====================================================================

		/**
		 * Generates a list of all classes upon which the list of classes
		 * depend.
		 *
		 * @param entries List of <code>JarEntrySpec</code>s used as a
		 *                list of class names from which to start.
		 * @exception IOException If there's an error reading a class file
		 */
	void generateDependancies( List entries )
		throws IOException
	{
		List dependants = new LinkedList();
		
		for ( Iterator it = entries.iterator(); it.hasNext();  ) {
			JarEntrySpec js = (JarEntrySpec)it.next();
			generateClassDependancies( js.getJarName(), dependants );
		}

		for ( Iterator it = dependants.iterator(); it.hasNext(); ) 
			entries.add( new JarEntrySpec( it.next().toString(), null ) );
	}

		/**
		 * Generates a list of classes upon which the named class is dependant.
		 *
		 * @param classFIleName File name hold the class to examine
		 * @param classes A List into which the class names are placed
		 */
	void generateClassDependancies( String classFileName, List classes )
		throws IOException
	{
		if ( ! resolved.contains( classFileName ) ) {
			resolved.add( classFileName );
			InputStream is = resolveEntry( classFileName );
			if ( is == null )
				throw new FileNotFoundException( classFileName );
			
			List referenced = ClassUtil.getDependancies( is );
			
			for ( Iterator it = referenced.iterator(); it.hasNext(); ) {
				String cname = it.next().toString() + ".class";

				if ( ! classFilter.include( cname ) || resolved.contains( cname )  )
					continue;

				classes.add( cname );
				generateClassDependancies( cname, classes );
			}
		}
	}
}
