/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.genjar;



import java.io.*;
import java.text.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;
import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;

/**
 * <p>A PathResolver is used for each component of the classpath.</p>
 * <p>
 *   Each type of component is a specialization of this base class.
 *   For example, a jar file encountered in the classpath causes a
 *   JarResolver to be instantiated. A JarResolver knows how to search
 *   within jar files for specific files.
 * </p>
 * <p>
 *  This approach is taken for two reasons:
 *  <ol>
 *    <li>
 *      To encapsulate the specifics of fetching streams and what attributes
 *      are set on a jar entry's manifest entry.
 *    </li>
 *    <li>
 *      To provide an association between the <i>source</i> of a class (or resource)
 *      and the repository from which it came.  This info is priceless when trying
 *      to figure out why the wrong classes are being included in your jar.
 *    </li>
 *  </ol>
 * </p>
 *
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */
abstract class PathResolver
{
		/** name of attribute for content's original location (path) */
	final static String CONTENT_LOC = "Content-Location";
		/** name of attribute for original content's modification time */
	final static String LAST_MOD    = "Last-Modified";
		/** format string for RFC1123 date */
	final static String RFC1123     = "EEE, dd MMM yyyy HH:mm:ss zzz";
	
	protected Logger log;
	static private SimpleDateFormat dateFmt = null;

	PathResolver( Logger log )
	{
		this.log = log;
		if ( dateFmt == null ) {
			dateFmt = new SimpleDateFormat( RFC1123, Locale.getDefault() );
				//
				// true conformance to rfc1123 would have us using GMT
				// rather than local time - this doesn;t make sense to
				// me since the biggest reason to use the time stamps
				// is to compare against the source files - which are in
				// local time
				//
				/*
				  dateFmt.setTimeZone( TimeZone.getTimeZone( "GMT" ) );
				*/
		}
	}
	
		/**
		 * Given a JarEntrySpec, a path to the actual resource is generated
		 * and an input stream is returned on the path.  
		 *
		 * @param JarEntrySpec specification of the entry to resolve
		 * @exception IOException if any errors are encountered
		 * @return IOStream opened on the file referenced
		 */
	abstract InputStream resolve( JarEntrySpec je ) throws IOException;
		/**
		 * Given a file name (possibly relative), an InputStream is
		 * opened on that file or an exception is thrown.
		 *
		 * @param JarEntrySpec specification of the entry to resolve
		 * @exception IOException if any errors are encountered
		 * @return IOStream opened on the named file
		 */
	abstract InputStream resolve( String fname ) throws IOException;
		/**
		 * Formats a date compatable with RFC1123.  Well, close anyway.
		 * To be truly conformant the time zone would always be GMT.
		 * This formats the date in local time.
		 * @param date the date/time to format
		 * @return String representation of the date
		 */
	protected String formatDate( Date d )
	{
		return dateFmt.format( d );
	}

	protected String formatDate( long l )
	{
		return formatDate( new Date( l ) );
	}
}

/**
 * Represents a directory in the classpath.<p>
 * When a directory is located in the classpath, a FileResolver
 * is instantiated that encapsulates the path and performs searches
 * in that directory.  This class is used primarily to allow easy
 * association of the <i>source directory</i> with the jar entry's
 * attributes.<p>
 *
 * When a file is resolved from a JarEntrySpec, Attributes are added
 * for the source file's path and last modification time.<p>
 *
 *
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */
class FileResolver extends PathResolver
{
		/**
		 * the 'base' directory as specified in the classpath
		 */
	File base = null;
		/**
		 * constructs a new FileResolver using the given
		 * <i>base directory</i>
		 *
		 * @param base a directory at which file searches begin
		 * @param log  an ant logging mechanism
		 */
	FileResolver( File base, Logger log )
	{
		super( log );

		this.base = base;
		log.verbose( "Resolver: " + base );
	}
		/**
		 * Resolve the file specified in a JarEntrySpec to
		 * a stream.
		 *
		 * @param spec the JarEntrySpec to resolve
		 * @return an InputStream open on the resolved file or null
		 * @exception IOException if opening the stream fails
		 */
	public InputStream resolve( JarEntrySpec spec )
		throws IOException
	{
		InputStream is = null;
		File f;

			//
			// if the entrySpec already has a source file
			// associated, use it otherwise attempt to
			// create one
			//
		if ( ( f = spec.getSourceFile() ) == null )
			f = new File( base, spec.getJarName() );

		if ( f.exists() ) {
			spec.setSourceFile( f );
			is = new FileInputStream( f );
			Attributes atts = spec.getAttributes();
			atts.putValue( CONTENT_LOC, f.getAbsolutePath() );
			atts.putValue( LAST_MOD, formatDate( f.lastModified() ) );
			log.debug( spec.getJarName() + "->" + base );
		}
		return is;
	}
		
	public InputStream resolve( String fname )
		throws IOException
	{
		InputStream is = null;
		File f = new File( base, fname );

		if ( f.exists() ) {
			is = new FileInputStream( f );
			log.debug( fname + "->" + base );
		}
		return is;
	}
}

/**
 * Represents a zip file in the classpath.<p>
 * When a Zip file is located in the classpath, a ZipResolver
 * is instantiated that encapsulates the path and performs searches
 * in that zip.  This class is used primarily to allow easy
 * association of the <i>zip file</i> with the jar entry's
 * attributes.<p>
 *
 * When a file is resolved from a JarEntrySpec, Attributes are added
 * for the zip file and last modification time.<p>
 * 
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */
class ZipResolver extends PathResolver
{
	File    file = null;
	ZipFile zip = null;
	String  modified;
	
		
	ZipResolver( File file, Logger log )
		throws IOException
	{
		super( log );

		this.file = file;
		this.zip  = new ZipFile( file );
		modified = formatDate( file.lastModified() );
		log.verbose( "Resolver: " + file );
	}

	public InputStream resolve( JarEntrySpec spec )
		throws IOException
	{
		InputStream is = null;
		ZipEntry ze = zip.getEntry( spec.getJarName() );
		if ( ze != null ) {
			Attributes atts = spec.getAttributes();
			atts.putValue( CONTENT_LOC, file.getAbsolutePath() );
			long modTime = ze.getTime();
			if ( modTime > 0 )
				atts.putValue( LAST_MOD, formatDate( modTime ) );
			else
				atts.putValue( LAST_MOD, modified );
			is = zip.getInputStream( ze );
			log.debug( spec.getJarName() + "->" + file );
		}
		return is;
	}
		
	public InputStream resolve( String fname )
		throws IOException
	{
		InputStream is = null;
		ZipEntry ze = zip.getEntry( fname );
		if ( ze != null ) {
			is = zip.getInputStream( ze );
			log.debug( fname + "->" + file );
		}
		return is;
	}
}

/**
 * Represents a jar file in the classpath.<p>
 * When a Jar file is located in the classpath, a JarResolver
 * is instantiated that remembers the path and performs searches
 * in that jar.  This class is used primarily to allow easy
 * association of the <i>jar file</i> with the jar entry's
 * attributes.<p>
 *
 * When a file is resolved from a JarEntrySpec, Attributes are added
 * for the jar file and last modification time.<p>
 *
 * <p>TODO: copy all entry-attributes from the source jar into our manifest</p>
 * 
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */
class JarResolver extends PathResolver
{
	File file = null;
	JarFile jarFile = null;
	String  modified;
	
	JarResolver( File file, Logger log )
		throws IOException
	{
		super( log );
		
		this.file = file;
		this.jarFile = new JarFile( file );
		modified = formatDate( file.lastModified() );
		log.verbose( "Resolver: " + file );
	}
		
	public InputStream resolve( JarEntrySpec spec )
		throws IOException
	{
		InputStream is = null;
			
		JarEntry je = jarFile.getJarEntry( spec.getJarName() );
		if ( je != null ) {
			Attributes atts = spec.getAttributes();
			atts.putValue( CONTENT_LOC, file.getAbsolutePath() );
			long modTime = je.getTime();
			if ( modTime > 0 )
				atts.putValue( LAST_MOD, formatDate( modTime ) );
			else
				atts.putValue( LAST_MOD, modified );
			spec.addAttributes( je.getAttributes() );
			is = jarFile.getInputStream( je );
			log.debug( spec.getJarName() + "->" + file );
		}
		return is;
	}
		
	public InputStream resolve( String fname )
		throws IOException
	{
		InputStream is = null;

		JarEntry je = jarFile.getJarEntry( fname );
		if ( je != null ) {
			is = jarFile.getInputStream( je );
			log.debug( fname + "->" + file );
		}
		return is;
	}
}
