/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs.optional.genjar;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;
import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;

/**
 * <p>Represents a &lt;resource&gt; element within the project file.</p>
 * <p>In addition to holding the final <i>jar name</i> of the resource, it
 * performs the actual resolution of file names along with expansion
 * of <i>filesets</i>.
 *
 * @author Original Code: <a href="mailto:jake@riggshill.com">John W. Kohler</a>
 * @version @version@
 */
public class Resource extends JarSpec
{
	private Project project;
	private List jarEntries = new ArrayList();
	
	private List filesets = new ArrayList( 8 );
	private File file = null;

	private String pkg = null;
	

		/**
		 * Constructs a new Resource
		 * @param project the owning project
		 */
	Resource( Project project )
	{
		this.project = project;
	}

		/**
		 * Returns a List of all JarEntry objects
		 * collected by this Resource
		 * @return List all collected JarEntry objects
		 */
	public List getJarEntries()
	{
		return jarEntries;
	}
		/**
		 * Sets the value of the file attribute.  When the
		 * resource element ha a single file attribute, Ant calls
		 * this to 'set' the value.
		 *
		 * @param f the value of the file attribute
		 */
	public void setFile( String f )
	{
		if ( filesets.size() > 0 )
			throw new BuildException( "can't add 'file' - fileset already used" );
		file = new File( f );
	}
		/**
		 * synonym for 'file'
		 */
	public void setName( String n )
	{
		setFile( n );
	}
		/**
		 * set the package (path) that'll be applied to ALL resources in this resource set
		 * - make sure to handle the empty package properly
		 *
		 * @param pkg sets the value of the <code>package</code> attribute
		 */
	public void setPackage( String pkg )
	{
		pkg = pkg.replace( '.', '/' );
		if ( pkg.length() > 0 && ! pkg.endsWith( "/" ) )
			this.pkg = pkg + "/";
		else
			this.pkg = pkg;
	}
		/**
		 * creates a FileSet - in response to the
		 * ant parse phase
		 */
	public Object createFileset()
	{
		if ( file != null )
			throw new BuildException( "can't add Fileset - file already set" );
		FileSet set = new FileSet();
		filesets.add( set );
		return set;
	}
		/**
		 * changes the path portion of the file
		 * if the <var>pkg</var> variable is
		 * not null
		 */
	String repackage( String s )
	{
		return repackage( new File( s ) );
	}
		/**
		 * changes the path portion of the file
		 * if the <var>pkg</var> variable is
		 * not null
		 */
	String repackage( File f )
	{
		if ( pkg == null )
			return f.toString();
		else
			return pkg + f.getName();
	}
		/**
		 * resolves the file attribute or fileset children
		 * into a collection of JarEntrySpec objects
		 */
	public void resolve( GenJar gj )
		throws IOException
	{
		if ( file != null ) {
			if ( file.exists() ) 
				jarEntries.add( new JarEntrySpec( repackage( file ),
												  file.getAbsoluteFile() ) );
			else
				jarEntries.add( new JarEntrySpec( repackage( file ), null ) );
		}	
			
		for ( Iterator it = filesets.iterator(); it.hasNext(); ) {
			FileSet fs = (FileSet)it.next();
			File    dir = fs.getDir( project );
				
			DirectoryScanner ds = fs.getDirectoryScanner( project );

			String[] a = ds.getIncludedFiles();
			for ( int i = 0; i < a.length; ++i ) 
				jarEntries.add( new JarEntrySpec( repackage( a[ i ] ),
												  new File( dir, a[ i ] ) ) );
		}
	}
		/**
		 * return a string representation of this resource set
		 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer();
		for ( Iterator it = jarEntries.iterator(); it.hasNext(); ) {
			sb.append( "\n" );
			sb.append( it.next() );
		}
		return sb.toString();
	}
}
