/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.jdoyle.ant.taskdefs.optional.ejb;

import java.util.*;
import java.io.*;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.types.Commandline;
import org.apache.tools.ant.types.Environment;
import org.apache.tools.ant.taskdefs.optional.junit.*;

/**
 * Execute a Weblogic server and unit test it
 * @author <a href="mailto:jimdoyle@gis.net">Jim Doyle</a>
 */
public class WLUnit extends Task {
	 /**
	  * The weblogic server that this task will run
	  */
	 private WeblogicServer server = new WeblogicServer();

	 /**
	  * The JUnit sub-task that this task will run all tests with
	  */
	 private JUnitTask testsTask;

	 /**
	  * Gets the subtask for tests.  In a method so we can lazy
	  * initialize after project gets set
	  * @return task
	  */
	 private JUnitTask getTestsTask ()
	 {
		  if (testsTask == null) {
				testsTask = (JUnitTask)project.createTask("junit");
		  }
		  return testsTask;
	 }

    /**
     * Add the classpath for the user classes
     */
    public Path createClasspath() {
		  return server.createClasspath(project);
    }

    /**
     * Get the classpath to the weblogic classpaths
     */
    public Path createWLClasspath() {
		  return server.createWLClasspath(project);
    }

    /**
     * Starts a WebLogic server in a separate process, runs JUnit tests,
	  * then shuts down the server.
     * @exception BuildException if the server process has a problem, or if the tests fail
     */
    public void execute() throws BuildException {
		  server.validate();
		  try {
				server.start(this);
				try {
					 runTests();
				} finally {
					 server.stop(this);
				}
		  } catch (IOException e) {
				throw new BuildException("Could not start server: " + e);
		  }
	 }
    
	 /**
	  * Runs tests, throwing BuildException if the tests fail and are meant
	  * to fail the target.
	  * @exception BuildException if tests fail
	  */
	 private void runTests () throws BuildException {
		  getTestsTask().setFork(true);
		  getTestsTask().setTaskName(getTaskName());

		  // Preset JNDI environment for server unit tests
		  Iterator jndiSysPropIt = server.jndiSysProps().iterator();
		  while (jndiSysPropIt.hasNext()) {
				getTestsTask().addSysproperty((Environment.Variable)jndiSysPropIt.next());
		  }

		  getTestsTask().execute();
	 }

    /**
     * Set the security policy for this invocation of weblogic.
     *
     * @param securityPolicy the security policy to use.
     */
    public void setPolicy(String securityPolicy) {
		  server.setPolicy(securityPolicy);
    }
    
    /**
     * The location where weblogic lives.
     *
     * @param weblogicHome the home directory of weblogic.
     *
     */
    public void setHome(File weblogicHome) {
		  server.setHome(weblogicHome);
    }

    /**
     * Set the name of the server to run
     *
     * @param systemName the name of the server.
     */
    public void setName(String serverName) {
		  server.setName(serverName);
    }
    
    /**
     * Set the properties file to use.
     *
     * The location of the properties file is relative to the weblogi system home
     *
     * @param propertiesFilename the properties file name
     */
    public void setProperties(String propertiesFilename) {
		  server.setProperties(propertiesFilename);
    }

    /**
     * Set the additional arguments to pass to the weblogic JVM
     * @param args the arguments to be passed to the JVM
     */
    public void setJvmargs(String args) {
		  server.setJvmargs(args);
    }
    
    public void setArgs(String args) {
		  server.setArgs(args);
    }

    public void setUsername(String username) {
		  server.setUsername(username);
    }

    public void setPassword(String password) {
		  server.setPassword(password);
    }

    public void setUrl(String url) {
		  server.setUrl(url);
    }

	 /**
	  * Sets the startup pause (in millis) that we allow
	  * for the server process coming up.
	  * @param pause Pause in millis
	  */
	 public void setStartupPause (long pause)
	 {
		  server.setStartupPause(pause);
	 }

    /**
     * Tells this task to halt when there is an error in a test.
     * this property is applied on all BatchTest (batchtest) and JUnitTest (test)
     * however it can possibly be overridden by their own properties.
     * @param   value   <tt>true</tt> if it should halt, otherwise <tt>false<tt>
     */
    public void setHaltTestsOnError(boolean value) {
		  getTestsTask().setHaltonerror(value);
	 }

    /**
     * Tells this task to halt when there is a failure in a test.
     * this property is applied on all BatchTest (batchtest) and JUnitTest (test)
     * however it can possibly be overridden by their own properties.
     * @param   value   <tt>true</tt> if it should halt, otherwise <tt>false<tt>
     */
    public void setHaltTestsOnFailure(boolean value) {
		  getTestsTask().setHaltonfailure(value);
	 }

    /**
     * Tells whether the task should print a short summary of the task.
     * @param   value   <tt>true</tt> to print a summary, <tt>false</tt> otherwise.
     * @see SummaryJUnitResultFormatter
     */
    public void setPrintTestSummary(boolean value) {
		  getTestsTask().setPrintsummary(value);
	 }

    /**
     * Set the timeout value (in milliseconds). If the test is running for more than this
     * value, the test will be canceled.
     * @param   value   the maximum time (in milliseconds) allowed before declaring the test
     *                  as 'timed-out'
     */
    public void setTestTimeout(Integer value) {
		  getTestsTask().setTimeout(value);
	 }

    /**
     * Set the maximum memory to be used by all forked JVMs.
     * @param   max     the value as defined by <tt>-mx</tt> or <tt>-Xmx</tt>
     *                  in the java command line options.
     */
    public void setTestsMaxmemory(String max) {
		  getTestsTask().setMaxmemory(max);
	 }

    /**
     * Set a new VM to execute the testcase. Default is <tt>java</tt>.
     * @param   value   the new VM to use instead of <tt>java</tt>
     */
    public void setTestsJvm(String value) {
		  getTestsTask().setJvm(value);
	 }

    /**
     * Create a new JVM argument.
     * @return  create a new JVM argument so that any argument can be passed to the JVM.
     */
    public Commandline.Argument createTestsJvmarg() {
		  return getTestsTask().createJvmarg();
	 }

    /**
     * The directory to invoke the VM in.
     * @param   dir     the directory to invoke the JVM from.
     */
    public void setTestsDir(File dir) {
		  getTestsTask().setDir(dir);
	 }

    /**
     * Add a nested sysproperty element.
     */
    public void addTestsSysproperty(Environment.Variable sysp) {
		  getTestsTask().addSysproperty(sysp);
	 }

    public Path createTestsClasspath() {
		  return getTestsTask().createClasspath();
	 }

    /**
     * Add a new single testcase.
     * @param   test    a new single testcase
     * @see JUnitTest
     */
    public void addTest(JUnitTest test) {
		  getTestsTask().addTest(test);
	 }

    /**
     * Create a new set of testcases (also called ..batchtest) and add it to the list.
     * @return  a new instance of a batch test.
     * @see BatchTest
     */
    public BatchTest createBatchTest() {
		  return getTestsTask().createBatchTest();
	 }

    /**
     * Add a new formatter to all tests of this task.
     */
    public void addFormatter(FormatterElement fe) {
		  getTestsTask().addFormatter(fe);
	 }
}
