/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.jdoyle.ant.taskdefs.optional.ejb;

import java.util.*;
import java.io.*;
import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;
import org.apache.tools.ant.taskdefs.*;

/**
 * Wraps the interaction with the weblogic server process
 * @author <a href="mailto:jimdoyle@gis.net">Jim Doyle</a>
 */
public class WeblogicServer
{
    private static final String DEFAULT_WL51_POLICY_FILE = "weblogic.policy";
    private static final String DEFAULT_PROPERTIES_FILE = "weblogic.properties";
    private static final String ServerMainClass = "weblogic.Server";
    private static final String AdminMainClass = "weblogic.Admin";
    private static final int ShutdownDelay = 0;

    /**
     * The classpath to be used when running the Java VM. It must contain the weblogic
     * classes <b>and</b> the implementation classes of the home and remote interfaces.
     */
    private Path classpath;

    /**
     * The weblogic classpath to the be used when running weblogic.
     */
    private Path weblogicClasspath;

    /**
     * Addional arguments to pass to the JVM used to run weblogic
     */
    private String additionalArgs = "";
    
    /**
     * The security policy to use when running the weblogic server
     */
    private String securityPolicy;
    
    /**
     * The weblogic system home directory
     */
    private File weblogicSystemHome;

    /**
     * The name of the weblogic server - used to select the server's directory in the 
     * weblogic home directory.
     */
    private String weblogicSystemName = "myserver";
    
    /**
     * The file containing the weblogic properties for this server.
     */
    private String weblogicPropertiesFile = null;

    /**
     * additional args to pass to the spawned jvm
     */
    private String additionalJvmArgs = "";

    /**
     * The weblogic username to use to request the shutdown.
     */
    private String username;
    
    /**
     * The password to use to shutdown the weblogic server.
     */
    private String password;
    
    /**
     * The URL which the weblogic server is listening on.
     */
    private String serverURL;

	 /**
	  * Amount of time to give the server to start up.  This could
	  * be a bit better; we could parse the output and look for 
	  * "WebLogic Server started".
	  */
	 private long startupPause;

    /**
     * Process handle for concurrent server process
     */
    private Process process;

    /**
     * IO handler thread that executes while server process executes
     */
    private LogStreamHandler processIoHandler;

    /**
     * Validates that the proper arguments are set.
     * @exception BuildException if anything's missing
     */
    public void validate () throws BuildException
    {
        if (weblogicSystemHome == null) {
            throw new BuildException("Attribute \"home\" must be set to weblogic installation dir.");
        }
        if (!weblogicSystemHome.isDirectory()) {
            throw new BuildException("Weblogic home directory \"" + weblogicSystemHome.getPath() + 
                                     "\" from Attribute \"home\" is not valid");
        }

        if (username == null || password == null) {
            throw new BuildException("Attributes \"username\" and \"password\" must both be set for the weblogic system user.");
        }
            
        if (serverURL == null) {
            throw new BuildException("Attribute \"url\" must be set to the T3 URL for the weblogic server.");
        }
    }

	 /**
	  * Returns a JNDI environment corresponding to this server.
	  * @return List of system properties set for JNDI, as Environment.Variable
	  */
	 public List jndiSysProps () {
		  ArrayList props = new ArrayList();
		  Environment.Variable jndiFactory = new Environment.Variable();
		  jndiFactory.setKey("java.naming.factory.initial");
		  jndiFactory.setValue("weblogic.jndi.WLInitialContextFactory");
		  props.add(jndiFactory);

		  Environment.Variable jndiUrl = new Environment.Variable();
		  jndiUrl.setKey("java.naming.provider.url");
		  jndiUrl.setValue(serverURL);
		  props.add(jndiUrl);
		  return props;
	 }

    /**
     * Starts the server in a concurrent process.  The server will continue
     * executing until stop() is called.  Also starts IO handler thread 
     * pumping output to log until the process exits.
	  * Pauses (see setStartupPause()) to give the server process time to 
	  * startup.
     * @param task task requiring this server, used for logging
     * @exception BuildException if the server was not started
     * @exception IOException if the server process had an IO problem starting up
     */
    public void start (Task task) throws BuildException, IOException
    {
        CommandlineJava serverCmdLine = serverCommandLine(task.getProject());

        task.log("Forking " + serverCmdLine.toString(), Project.MSG_VERBOSE);
        
        process = Runtime.getRuntime().exec(serverCmdLine.getCommandline());

        processIoHandler = 
            new LogStreamHandler(task, Project.MSG_INFO, Project.MSG_WARN);
        processIoHandler.setProcessInputStream(process.getOutputStream());
        processIoHandler.setProcessOutputStream(process.getInputStream());
        processIoHandler.setProcessErrorStream(process.getErrorStream());
        processIoHandler.start();

        task.log("Pausing for " + startupPause + " milliseconds while server process starts.", Project.MSG_VERBOSE);
		  try {
				Thread.sleep(startupPause);
		  } catch (InterruptedException e) {
				e.printStackTrace();
		  }
        task.log("Continuing after pause.", Project.MSG_VERBOSE);
    }

    /**
     * Stops the server, blocking until the concurrent process exits.
     * Also joins the thread that's pumping output to the log.
     * @param task Originally using this server, used to create a sub-task
     * @exception BuildException if the java process that requests shutdown fails
     */
    public void stop (Task task) throws BuildException
    {
        Java weblogicAdmin = (Java)task.getProject().createTask("java");
        weblogicAdmin.setFork(true);
        weblogicAdmin.setTaskName(task.getTaskName());
        weblogicAdmin.setClassname(AdminMainClass);
        weblogicAdmin.createArg().setLine(serverURL 
                                        + " SHUTDOWN " 
                                        + username 
                                        + " " + password 
                                        + " " + ShutdownDelay);
        weblogicAdmin.setClasspath(weblogicClasspath);
        weblogicAdmin.execute();

        task.log("Waiting for server exit.", Project.MSG_VERBOSE);
        try {
            process.waitFor();
        } catch (InterruptedException e) {}
        processIoHandler.stop();
    }

    /**
     * Creates propertes file object based on arguments
     * @exception BuildException if the file could not be created
     */
    private File propertiesFile () throws BuildException
    {
        File propertiesFile = null;
        
        if (weblogicPropertiesFile == null) {
            propertiesFile = new File(weblogicSystemHome, DEFAULT_PROPERTIES_FILE);
        }
        else {
            propertiesFile = new File(weblogicSystemHome, weblogicPropertiesFile);
        }
        if (!propertiesFile.exists()) {
            throw new BuildException("Properties file " + weblogicPropertiesFile +
                                     " not found in weblogic home " + weblogicSystemHome);
        }
        return propertiesFile;
    }

    /**
     * Creates security policy file object based on arguments
     * @exception BuildException if the file could not be created
     */
    private File securityPolicyFile () throws BuildException
    {
        File securityPolicyFile = null;
        if (securityPolicy == null) {
            securityPolicyFile = new File(weblogicSystemHome, DEFAULT_WL51_POLICY_FILE);
        }
        else {
            securityPolicyFile = new File(weblogicSystemHome, securityPolicy);
        }
        if (!securityPolicyFile.exists()) {
            throw new BuildException("Security policy " + securityPolicyFile +
                                     " was not found.");
        }
        return securityPolicyFile;
    }

    /**
     * Formats JVM arguments, including all the "-D" defines.
     * @return args string
     */
    private String jvmArgs ()
    {
        String jvmArgs = additionalJvmArgs;
        
        if (weblogicClasspath != null) {
            jvmArgs += " -Dweblogic.class.path=" + weblogicClasspath;
        }
            
        jvmArgs += " -Djava.security.manager -Djava.security.policy==" + securityPolicyFile();
        jvmArgs += " -Dweblogic.system.home=" + weblogicSystemHome;
        jvmArgs += " -Dweblogic.system.name=" + weblogicSystemName;
        jvmArgs += " -Dweblogic.system.propertiesFile=" + propertiesFile();
        return jvmArgs;
    }

     
    /**
     * Creates java command line for the server
     * @param project Project in which this server's task is being run, used to create a path
     * @exception BuildException if we can't make a valid command line
     */
    private CommandlineJava serverCommandLine (Project project) throws BuildException
    {
        CommandlineJava cmdLine = new CommandlineJava();
        cmdLine.setClassname(ServerMainClass);
        cmdLine.createVmArgument().setLine(jvmArgs());
        cmdLine.createArgument().setLine(additionalArgs);
        if (classpath != null) {
            cmdLine.createClasspath(project).append(classpath);
        }
        return cmdLine;
    }

    /**
     * Add the classpath for the user classes
     * @param project Project in which this server's task is being run, used to create a path
     */
    public Path createClasspath(Project project) {
        if (classpath == null) {
            classpath = new Path(project);
        }
        return classpath.createPath();
    }

    /**
     * Get the classpath to the weblogic classpaths
     * @param project Project in which this server's task is being run, used to create a path
     */
    public Path createWLClasspath(Project project) {
        if (weblogicClasspath == null) {
            weblogicClasspath = new Path(project);
        }
        return weblogicClasspath.createPath();
    }

    /**
     * Set the security policy for this invocation of weblogic.
     *
     * @param securityPolicy the security policy to use.
     */
    public void setPolicy(String securityPolicy) {
        this.securityPolicy = securityPolicy;
    }
    
    /**
     * The location where weblogic lives.
     *
     * @param weblogicHome the home directory of weblogic.
     *
     */
    public void setHome(File weblogicHome) {
        weblogicSystemHome = weblogicHome;
    }

    /**
     * Set the name of the server to run
     *
     * @param systemName the name of the server.
     */
    public void setName(String serverName) {
        this.weblogicSystemName = serverName;
    }
    
    /**
     * Set the properties file to use.
     *
     * The location of the properties file is relative to the weblogi system home
     *
     * @param propertiesFilename the properties file name
     */
    public void setProperties(String propertiesFilename) {
        this.weblogicPropertiesFile = propertiesFilename;
    }

    /**
     * Set the additional arguments to pass to the weblogic JVM
     * @param args the arguments to be passed to the JVM
     */
    public void setJvmargs(String args) {
        this.additionalJvmArgs = args;
    }
    
    public void setArgs(String args) {
        additionalArgs = args;
    }

    /**
     * Set the username to use to request shutdown of the server.
     *
     * @param s the username.
     */
    public void setUsername(String s) {
        this.username = s;
    }
    
    /**
     * Set the password to use to request shutdown of the server.
     *
     * @param s the password.
     */
    public void setPassword(String s) {
        this.password = s;
    }
    
    /**
     * Set the URL to which the weblogic server is listening.
     *
     * @param s the url.
     */
    public void setUrl(String s) {
        this.serverURL = s;
    }

	 /**
	  * Sets the startup pause (in millis) that we allow
	  * for the server process coming up.
	  * @param pause Pause in millis
	  */
	 public void setStartupPause (long pause)
	 {
		  startupPause = pause;
	 }
}
