﻿/*
 * [The "BSD licence"]
 * Copyright (c) 2005-2008 Terence Parr
 * All rights reserved.
 *
 * Conversion to C#:
 * Copyright (c) 2008-2009 Sam Harwell, Pixel Mine, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Antlr.Runtime
{
    using System.Collections.Generic;
    using System.Linq;
    using Antlr.Runtime.Tree;

    using ArgumentException = System.ArgumentException;
    using ArgumentNullException = System.ArgumentNullException;
    using InvalidOperationException = System.InvalidOperationException;

    public class PrecedenceTreeCreator
    {
        const int PrecedenceTreeType = -2;
        OperatorGroup[] _operatorGroups;

        public PrecedenceTreeCreator( params OperatorGroup[] operatorGroups )
        {
            _operatorGroups = operatorGroups;
        }

        static ITree GetExpressionTree( ITree tree )
        {
            if ( tree.Type == PrecedenceTreeType )
                return tree.GetChild( 0 );

            return tree;
        }

        public ITree BuildExpressionTree( ITreeAdaptor adaptor, List<BaseTree> expressionElements )
        {
            if ( adaptor == null )
                throw new ArgumentNullException( "adaptor" );

            if ( expressionElements == null )
                throw new ArgumentNullException( "expressionElements" );

            if ( expressionElements.Any( tree => tree == null || tree.Type == PrecedenceTreeType ) )
                throw new ArgumentException();

            if ( expressionElements.Count == 1 )
                return expressionElements[0];

            // BuildExpressionTree expects to work with a clone, because it will alter the list when it collapses ternary nodes
            expressionElements = new List<BaseTree>( expressionElements );
            CollapseCenters( adaptor, expressionElements, 0, 0 );
            return BuildExpressionTree( adaptor, expressionElements, 0, expressionElements.Count - 1 );
        }

        ITree BuildExpressionTree( ITreeAdaptor adaptor, List<BaseTree> expressionElements, int startIndex, int stopIndex, UnaryOperator op, int position )
        {
            /*
             * All parameters are supposed to be valid by the time we reach this private helper method.
             */
            System.Diagnostics.Debug.Assert( adaptor != null );
            System.Diagnostics.Debug.Assert( expressionElements != null );
            System.Diagnostics.Debug.Assert( op != null );
            System.Diagnostics.Debug.Assert( startIndex >= 0 && startIndex < expressionElements.Count );
            System.Diagnostics.Debug.Assert( stopIndex >= 0 && stopIndex < expressionElements.Count );
            // must be 2 or more elements in the set
            System.Diagnostics.Debug.Assert( stopIndex - startIndex + 1 >= 2 );
            System.Diagnostics.Debug.Assert( position == startIndex || ( op.OperatorType == UnaryOperatorType.Postfix ) );
            System.Diagnostics.Debug.Assert( position == stopIndex || ( op.OperatorType == UnaryOperatorType.Prefix ) );

            ITree subtree = null;
            if ( position == startIndex )
                subtree = BuildExpressionTree( adaptor, expressionElements, startIndex + 1, stopIndex );
            else
                subtree = BuildExpressionTree( adaptor, expressionElements, startIndex, stopIndex - 1 );

            ITree root = expressionElements[position];
            if ( op.RewriteType != 0 )
            {
                if ( root.ChildCount > 0 )
                    throw new InvalidOperationException();
                root = (ITree)adaptor.Create( op.RewriteType, ( (CommonTree)root ).Token );
            }

            adaptor.AddChild( root, GetExpressionTree( subtree ) );
            return root;
        }
        ITree BuildExpressionTree( ITreeAdaptor adaptor, List<BaseTree> expressionElements, int startIndex, int stopIndex, BinaryOperator op, int position )
        {
            /*
             * All parameters are supposed to be valid by the time we reach this private helper method.
             */
            System.Diagnostics.Debug.Assert( adaptor != null );
            System.Diagnostics.Debug.Assert( expressionElements != null );
            System.Diagnostics.Debug.Assert( op != null );
            System.Diagnostics.Debug.Assert( startIndex >= 0 && startIndex < expressionElements.Count );
            System.Diagnostics.Debug.Assert( stopIndex >= 0 && stopIndex < expressionElements.Count );
            // must be 3 or more elements in the set, and the operator can't be at either of the ends
            System.Diagnostics.Debug.Assert( stopIndex - startIndex + 1 >= 3 );
            System.Diagnostics.Debug.Assert( position > startIndex );
            System.Diagnostics.Debug.Assert( position < stopIndex );

            ITree left = BuildExpressionTree( adaptor, expressionElements, startIndex, position - 1 );
            ITree right = BuildExpressionTree( adaptor, expressionElements, position + 1, stopIndex );

            ITree root = expressionElements[position];
            if ( op.RewriteType != 0 )
            {
                if ( root.ChildCount > 0 )
                    throw new InvalidOperationException();
                root = (ITree)adaptor.Create( op.RewriteType, ( (CommonTree)root ).Token );
            }

            adaptor.AddChild( root, GetExpressionTree( left ) );
            adaptor.AddChild( root, GetExpressionTree( right ) );
            return root;
        }
        ITree BuildExpressionTree( ITreeAdaptor adaptor, List<BaseTree> expressionElements, int startIndex, int stopIndex, TernaryOperator op, int firstPosition, int secondPosition )
        {
            /*
             * All parameters are supposed to be valid by the time we reach this private helper method.
             */
            System.Diagnostics.Debug.Assert( adaptor != null );
            System.Diagnostics.Debug.Assert( expressionElements != null );
            System.Diagnostics.Debug.Assert( op != null );
            System.Diagnostics.Debug.Assert( startIndex >= 0 && startIndex < expressionElements.Count );
            System.Diagnostics.Debug.Assert( stopIndex >= 0 && stopIndex < expressionElements.Count );
            // must be 5 or more elements in the set, and the operators can't be at either of the ends
            System.Diagnostics.Debug.Assert( stopIndex - startIndex + 1 >= 5 );
            System.Diagnostics.Debug.Assert( firstPosition > startIndex );
            System.Diagnostics.Debug.Assert( secondPosition < stopIndex );
            // at least one element in the middle set
            System.Diagnostics.Debug.Assert( secondPosition - firstPosition + 1 >= 1 );

            ITree left = BuildExpressionTree( adaptor, expressionElements, startIndex, firstPosition - 1 );
            ITree center = BuildExpressionTree( adaptor, expressionElements, firstPosition + 1, secondPosition - 1 );
            ITree right = BuildExpressionTree( adaptor, expressionElements, secondPosition + 1, stopIndex );

            ITree root = expressionElements[firstPosition];
            if ( op.RewriteType != 0 )
            {
                if ( root.ChildCount > 0 )
                    throw new InvalidOperationException();
                root = (ITree)adaptor.Create( op.RewriteType, ( (CommonTree)root ).Token );
            }

            adaptor.AddChild( root, GetExpressionTree( left ) );
            adaptor.AddChild( root, GetExpressionTree( center ) );
            adaptor.AddChild( root, GetExpressionTree( right ) );
            return root;
        }

        int LookupTernaryStopType( int startType )
        {
            int index;
            TernaryOperator op = FindOperator( startType, out index ) as TernaryOperator;
            return ( op == null ) ? 0 : op.RightToken;
        }
        void CollapseCenters( ITreeAdaptor adaptor, List<BaseTree> newElements, int startIndex, int stopTokenType )
        {
            int nextStopIndex = -1;
            int nextTernaryStart = -1;
            int nextStopType = 0;

            /*
             * Locate the "next" positions:
             *  - if stopTokeType!=0, then nextStopIndex is simply the next index of the stopTokenType, else -1
             *  - nextTernaryStart is the index of the next ternary start operator *whose center section is not a single node*
             */
            if ( stopTokenType != 0 )
            {
                for ( int i = startIndex + 1; i < newElements.Count; i++ )
                {
                    if ( newElements[i].Type == stopTokenType )
                    {
                        nextStopIndex = i;
                        break;
                    }
                }
            }

            for ( int i = startIndex + 1; i < newElements.Count; i++ )
            {
                nextStopType = LookupTernaryStopType( newElements[i].Type );
                if ( nextStopType != 0 && newElements[i + 2].Type != nextStopType )
                {
                    nextTernaryStart = i;
                    break;
                }
            }

            if ( nextStopIndex >= 0 || nextTernaryStart >= 0 )
            {
                if ( nextStopIndex >= 0 && ( nextTernaryStart == -1 || nextTernaryStart > nextStopIndex ) )
                {
                    ITree center = BuildExpressionTree( adaptor, newElements, startIndex + 1, nextStopIndex - 1 );
                    center = (ITree)adaptor.BecomeRoot( adaptor.Create( PrecedenceTreeType, "" ), center );
                    newElements.RemoveRange( startIndex + 1, nextStopIndex - startIndex - 1 );
                    newElements.Insert( startIndex + 1, (BaseTree)center );
                    CollapseCenters( adaptor, newElements, startIndex + 3, 0 );
                    return;
                }
                else
                {
                    // found a ternary operator with an uncollapsed center within this center section. collapse it and try again.
                    CollapseCenters( adaptor, newElements, nextTernaryStart, nextStopType );
                    CollapseCenters( adaptor, newElements, startIndex, stopTokenType );
                    return;
                }
            }
            else
            {
                // done collapsing. verify we weren't requiring a stop token.
                if ( stopTokenType != 0 )
                    throw new ArgumentException( "Ternary expression missing the closing operator." );
            }
        }
        ITree BuildExpressionTree( ITreeAdaptor adaptor, List<BaseTree> expressionElements, int startIndex, int stopIndex )
        {
            /*
             * Here's the order we do things for a single expression:
             *   - If expressionElements has Count==1, just return the node
             *   - Collapse the center of all ternary operators to an expression (**already done before this method is called**).
             *   - For each operator group, starting with the lowest precedence (first in the array), do:
             *     - Locate the "first" operator in the subset of expressionElements from this group, keeping the following in mind:
             *       - Search end-to-beginning if the group has RightToLeft associativity
             *       - Ignore unary operators immediately next to a binary or ternary operator
             *       - Ignore ternary stop operators
             *     - If an operator from this group was found,
             *       return BuildExpressionTree( adaptor, expressionElements, startIndex, stopIndex, {operator}, {operatorIndex} )
             *   - If we reach this point, the expression is not valid.
             */
            if ( startIndex == stopIndex )
            {
                return expressionElements[startIndex];
            }

            for ( int i = 0; i < _operatorGroups.Length; i++ )
            {
                OperatorGroup group = _operatorGroups[i];
                int loopStart = ( group.Associativity == OperatorAssociativity.LeftToRight ) ? startIndex : stopIndex;
                int loopStop = ( group.Associativity == OperatorAssociativity.LeftToRight ) ? stopIndex + 1 : startIndex - 1;
                int loopIncrement = ( group.Associativity == OperatorAssociativity.LeftToRight ) ? 1 : -1;

                for ( int j = loopStart; j != loopStop; j += loopIncrement )
                {
                    // assume that expressions with children are already proper trees
                    if ( expressionElements[j].ChildCount > 0 )
                        continue;

                    Operator op;
                    if ( group.TryGetOperator( expressionElements[j].Type, out op ) )
                    {
                        UnaryOperator uop = op as UnaryOperator;
                        if ( uop != null && ( j == startIndex || j == stopIndex ) )
                            return BuildExpressionTree( adaptor, expressionElements, startIndex, stopIndex, uop, j );

                        BinaryOperator bop = op as BinaryOperator;
                        if ( bop != null && j != startIndex && j != stopIndex )
                            return BuildExpressionTree( adaptor, expressionElements, startIndex, stopIndex, bop, j );

                        TernaryOperator top = op as TernaryOperator;
                        if ( top != null )
                            return BuildExpressionTree( adaptor, expressionElements, startIndex, stopIndex, top, j, j + 2 );
                    }
                }
            }

            // if we get to this point, we failed
            throw new ArgumentException();
        }

        Operator FindOperator( int type, out int group )
        {
            for ( int i = 0; i < _operatorGroups.Length; i++ )
            {
                Operator op;
                if ( _operatorGroups[i].TryGetOperator( type, out op ) )
                {
                    group = i;
                    return op;
                }
            }

            group = -1;
            return null;
        }
    }
}
