
package org.k2d2.framework.serverframework;

import java.util.Vector;

import org.k2d2.framework.threadpackage.util.BlockingQueue;
import org.k2d2.framework.threadpackage.util.BoundedBlockingQueue;
import org.k2d2.framework.serverframework.core.ServerCommand;
import org.k2d2.framework.serverframework.core.ServerContainer;

/**
 * This class provides a framework for implementing multi-threaded servers.
 * It can be used to develop standalone servers or to develop multi-threaded code
 * running within another application server or platform (like and EJB container).
 * The server consists of an in-queue, an out-queue and one or more ServerCommand
 * objects. The ServerCommand objects are the workers. They get an object from the
 * in-queue, process it and store it in the out-queue. The server framework takes
 * care of retrieving and storing objects in the queues. The MultiDispatchServer class
 * lets you chain multiple ServerCommand objects to form an overall processing scheme.
 * Each ServerCommand in the chain processes a given object from its in-queue and
 * stores it in the next ServerCommand's out-queue. The framework takes care
 * of storing the objects in the appropriate queues and the handoff. The ServerCommand
 * objects are chained in the order in which the addServerCommand() call is made. The
 * application developer is responsible for enqueuing tasks to the in-queue and
 * dequeuing results from the out-queue. The in-queue and out-queue are instances of
 * BlockingQueue and are thread safe.
 * @see org.k2d2.threadpackage.util.BlockingQueue
 * @see org.k2d2.serverframework.core.ServerCommand
 * @see #addServerCommand(ServerCommand)
 * @author Karthik Rangaraju
 */
public class MultiDispatchServer
{
    private int mQueueSize;
    private Vector mContainerVector;
    private BlockingQueue mInQueue;
    private BlockingQueue mOutQueue;
    private final int mInitialContainerVectorSize = 10;
    private final int mContainerVectorIncrementSize = 10;
            
    /**
     * This method initializes a multi-dispatch server with a specified in-queue
     * and out-queue size. Since the server uses a blocking queue, the size of the
     * queue affects performance when using multiple threads and hence should
     * not be set too small
     * @param pQueueSize - the size of the in-queue and out-queue
     */
    public MultiDispatchServer(int pQueueSize)
    {
        mContainerVector = new Vector(mInitialContainerVectorSize,
                                      mContainerVectorIncrementSize);
        mQueueSize = pQueueSize;
        mInQueue = new BoundedBlockingQueue(mQueueSize);
        mOutQueue = new BoundedBlockingQueue(mQueueSize);
    }
    
    /**
     * Adds a server command to the server. ServerCommand objects are added
     * in order. Hence the first command to be added using this method will
     * be the first to be executed in the command chain in the server.
     * @param pCommand - A reference to your customized server command that
     * does the application specific processing
     */
    public void addServerCommand(ServerCommand pCommand)
    {
        ServerContainer container;
        ServerContainer tempcontainer;
        BlockingQueue queue;
                
        // Create a new container and add it to the container vector.
        // Add the command to it and setup the queues so that the
        // added container's in-queue is the same as the previous container's
        // out-queue
        container = new ServerContainer();
        if (mContainerVector.size() == 0)
        {
            container.setInQueue(mInQueue);
            container.setOutQueue(mOutQueue);
        }
        else
        {
            tempcontainer = (ServerContainer)
                            mContainerVector.elementAt(mContainerVector.size() - 1);
                    
            tempcontainer.setOutQueue(new BoundedBlockingQueue(mQueueSize));
            container.setInQueue(tempcontainer.getOutQueue());
            container.setOutQueue(mOutQueue);
        }
        container.setCommand(pCommand);
        mContainerVector.addElement((Object)container);
    }
    
    /**
     * Returns the in-queue for the server. This is the queue into which you
     * add tasks to be processed
     * @return BlockingQueue - reference to the inqueue. You should never call
     * close() on this queue. Instead use the stop() command on the
     * MultiDispatchServer.
     * @see org.k2d2.threadpackage.util.BlockingQueue#close()
     */
    public BlockingQueue getInQueue()
    {
        return mInQueue;
    }
    
    /**
     * Returns the out-queue associated with the server. This is the queue
     * into which completed tasks are put (if necessary only - see the
     * ServerCommand description for more information). Threads waiting on
     * this queue are exited (and QueueClosedException is thrown) when the
     * server is stopped.
     @ return BlockingQueue - reference to the outqueue.
     */
    public BlockingQueue getOutQueue()
    {
        return mOutQueue;
    }
    
    /**
     * This starts the server by starting the threads that execute the
     * ServerCommand objects.
     */
    public void start()
    {
        int index;
        
        for (index = 0; index < mContainerVector.size(); index++)
        {
            ((ServerContainer)mContainerVector.elementAt(index)).start();
        }
    }
    
    /**
     * This method stops the server. The threads are not pre-empted. The server
     * shuts down gracefully. The method also guarantees that if there were
     * tasks in the queue before the call, those tasks will be processed to
     * completion. Because of this property, the method does not return
     * immediately (and to prevent getting into an undefined state if start() is
     * called in the meantime. The server will not stop if there is a continuous
     * in-flow of tasks into the in-queue. The caller is responsible to ensure
     * that nothing is written into the queue.
     */
    public void stop()
    {
        int index;
        
        for (index = 0; index < mContainerVector.size(); index++)
        {
            ((ServerContainer)mContainerVector.elementAt(index)).stop();
        }
        mOutQueue.close();
        mInQueue.reset();
    }
    
    /**
     * This method stops and starts the server. The propeties of the start()
     * and stop() method are preserved. When this method is called,
     * close() is called on the out-queue and then it is immediately
     * reset(). This can cause unexpected behavior on the threads waiting
     * to dequeue from the out-queue.
     * @see #start()
     * @see #stop()
     * @see org.k2d2.threadpackage.util.BlockingQueue#close()
     * @see org.k2d2.threadpackage.util.BlockingQueue#reset()
     */
    public void restart()
    {
        // TODO: See if unexpected behavior can be fixed by synchronizing the
        // reset method on the queues.
        stop();
        start();
    }
}

