
package org.k2d2.framework.serverframework.core;

import org.k2d2.framework.threadpackage.util.BlockingQueue;
import org.k2d2.framework.threadpackage.util.QueueClosedException;
import org.k2d2.framework.threadpackage.semaphore.DjikstraSemaphore;

/**
 * Internal container class to hold ServerCommand objects used by
 * MultiDispatchServer.
 *
 * @author Karthik Rangaraju
 */
public class ServerContainer implements Runnable
{
    private int mWorkerCount;
    private Thread[] mWorkerArray;
    private BlockingQueue mInQueue;
    private BlockingQueue mOutQueue;

    private ServerCommand mCommand;
    private ServerCommand.ThreadingModelType mModel;
    private DjikstraSemaphore mWorkerThrottle;
    
    // TODO: Move this to org.k2d2.framework.serverframework.core.container
    // and omit from javadoc list
    /**
     * Sets the container's inqueue (and in turn, the ServerCommand object's
     * in-queue)
     * @param pInQueue instance of a BlockingQueue class that will be
     * used as this container's in-queue
     */
    public void setInQueue(BlockingQueue pInQueue)
    {
        mInQueue = pInQueue;
    }
    
    /**
     * Returns a reference to the container's in-queue
     *@return BlockingQueue reference to container's in-queue
     */
    public BlockingQueue getInQueue()
    {
        return mInQueue;
    }
    
    /**
     * Sets the container's out-queue.
     * @param pOutQueue instance of a BlockingQueue class that will be
     * used as this container's out-queue
     */
    public void setOutQueue(BlockingQueue pOutQueue)
    {
        mOutQueue = pOutQueue;
    }
    
    /**
     * Returns a reference to the container's out-queue
     * @return BlockingQueue reference to container's out-queue
     */
    public BlockingQueue getOutQueue()
    {
        return mOutQueue;
    }
        
    /**
     * Sets the the container's ServerCommand object
     * @param pCommand a reference to a ServerCommand object that this container
     * will manage
     */
    public void setCommand(ServerCommand pCommand)
    {
        mCommand = pCommand;
        mModel = pCommand.getThreadingModelType();
        mWorkerCount = pCommand.getNumberOfThreads();
        mWorkerThrottle = null;
        mWorkerThrottle = new DjikstraSemaphore(mWorkerCount);
    }
    
    /**
     * Starts this container by creating and starting the worker threads
     */
    public void start()
    {
        mWorkerArray = new Thread[mWorkerCount];
        
        mOutQueue.reset();
        for (int index = 0; index < mWorkerCount; index++)
        {
            mWorkerArray[index] = new Thread(this);
            mWorkerArray[index].start();
        }
        try
        {
            // If stop() is called before all threads have been created, then
            // the mWorkerThrottle.acquireAll() operation in stop() will deadlock
            // with worker threads that have not yet acquired the mWorkerThrottle
            // semaphore in the run() method. Ensuring that all worker threads
            // have acquired their semaphore before exiting this method solves
            // the problem.
            mWorkerThrottle.starvationCheck();
        }
        catch (InterruptedException e)
        {
        }
    }
    
    /**
     * Stops the container by stopping all the threads. The threads are stopped
     * gracefully and this method does not return until all threads have been stopped
     */
    public void stop()
    {
        // Force the threads to exit the queue if they are blocked on
        // an empty queue
        mInQueue.close();
        try
        {
            // This ensures that we wait till all the threads die
            mWorkerThrottle.acquireAll();
        }
        catch (InterruptedException e)
        {
            // Do something with this. Log it or something
        }
        finally
        {
            mWorkerThrottle.releaseAll();
        }
    }
    
    /**
     * The workhorse of the container. Threads execute an infinite loop until
     * they are ejected from the queue (due to a stop request).
     */
    public void run()
    {
        Object inobject;
        Object outobject;
        
        ServerCommand command = null;
        
        if (mModel == ServerCommand.MULTI_THREADED)
        {
            command = mCommand;
        }
        else
        {
            command = (ServerCommand)mCommand.clone();
        }
        
        command.init();
        try
        {
            // Each thread acquires a semaphore to enable to stop() method
            // to determine if the threads have terminated or not
            mWorkerThrottle.acquire();
        }
        catch (InterruptedException e)
        {
            // Need to log this error. Should never get this exception
        }
        while (true)
        {
            try
            {
                try
                {
                    inobject = (Object)mInQueue.dequeue();
                }
                catch (QueueClosedException e)
                {
                    break;
                }
                outobject = command.execute(inobject);
                if (outobject != null)
                {
                    try
                    {
                        mOutQueue.enqueue(outobject);
                    }
                    catch (QueueClosedException e)
                    {
                        // TODO: Log as internal error. Should not happen!
                        // For app exposed outqueues, it can happen! The
                        // application should not close queues! So log as application
                        // error
                    }
                }
            }
            catch (InterruptedException e)
            {
                // what do I do with the exception?
                // k2: I think I should log the error (whenever I have
                // logging implemented - use log4J) and exit!
            }
        }
        mWorkerThrottle.release();
    }
}


