
package org.k2d2.framework.serverframework.pool;

import java.util.Stack;
import org.k2d2.framework.threadpackage.util.BoundlessBlockingQueue;
import org.k2d2.framework.threadpackage.util.QueueClosedException;

/**
 * This class provides a resource pool manager for managing scare resources
 * across threads and server clients. Application resources can be added to
 * a pool created using this class and then requested via PooledResource objects.
 * The PoolManager resolves contentions in requests for PooledResource objects
 * and blocks requests that cannot be immediately fulfilled. The PooledResource
 * object provides an interface to get the actual resource being managed and
 * provides a method to release the request. This class provides for graceful
 * exit of threads blocked on a get() request
 * <br><i>enhancement</i> Timeouts for get() method
 * <br><i>enhancement</i> Method to check if any resources are available
 *
 * @author Karthik Rangaraju
 * @see PooledResource
 *
 */
public class PoolManager
{
    private BoundlessBlockingQueue mResources;
    
    /**
     * Constructs a pool manager to handle pSize resources
     * @param pSize - Number of resources in the pool
     * <br><i>enhancement</i> Explore way of removing size specification
     * requirement and let pool size increase dynamically as resources are
     * added. The implementation requires a size parameter becauses of the
     * upper-bound requirement of blocking queues.
     */
    public PoolManager(int pSize)
    {
        mResources = new BoundlessBlockingQueue(pSize);
    }

    /**
     * This method is called by the PooledResourceContainer when release
     * is called. It has been declared with default access modifier to
     * limit access to this class (classes in this package)
     */
    void reclaim(PooledResourceContainer pContainer)
    {
        try
        {
            mResources.enqueue(pContainer);
        }
        catch (InterruptedException e)
        {
            // TODO: Need to log this as nasty
        }
        catch (QueueClosedException e)
        {
            // Ignore this. If a resource is released after the PoolManager
            // has been closed, this exception is thrown.
        }
    }
    
    /**
     * This method adds an application resource referenced by pResource
     * to the managed pool
     * @param pResource - Reference to resource that needs to be managed
     * (e.g Socket)
     */
    public void add(Object pResource)
    {
        PooledResourceContainer container = new PooledResourceContainer(this);
        
        container.set(pResource);
        try
        {
            mResources.enqueue(container);
        }
        catch (InterruptedException e)
        {
            // TODO: Need to log this as nasty
        }
        catch (QueueClosedException e)
        {
            // TODO: Throw back something indicating trouble
        }
    }
    
    /**
     * Retrieves a PooledResource object from the pool. This method blocks
     * if there are no resources in the pool.
     * @return PooledResource - reference to PooledResource obtained from the
     * pool
     * @throws QueueClosedException - if the PoolManager has been shut down
     * (using the close() call)
     * @see PooledResource
     * @see #close()
     */
    public PooledResource get() throws QueueClosedException
    {
        // TODO: Put a pooling specific exception here!
        PooledResource resource = null;
        
        try
        {
            resource = (PooledResource)mResources.dequeue();
        }
        catch (InterruptedException e)
        {
            // TODO: Log something
        }
        
        return resource;
    }
    
    /**
     * This method closes the PoolManager. Any threads blocked on a get()
     * call are forced to return and QueueClosedException is thrown.
     * @see #get()
     */
    public void close()
    {
        mResources.close();
    }
    
    /**
     * This method resets the PoolManager. All resources in the queue are lost.
     * You must call this method after a close() call to be able to add resources
     * to the pool.
     * @see #close()
     */
    public void reset()
    {
        mResources.reset();
    }
}
