
package org.k2d2.framework.threadpackage.semaphore;

/**
 * A critical section protects a section of code from concurrent access. When
 * used with instance methods, the mutual exclusion applies to threads across
 * all object instances e.g.
 * <pre>
 *      public class Test
 *      {
 *          private CriticalSection cs;
 *
 *          public Test()
 *          {
 *              cs = new CriticalSection(this);
 *          }
 *
 *          public void foo()
 *          {
 *              cs.acquire();
 *              System.out.println("I'm alone");
 *              cs.release();
 *          }
 *      }
 * </pre>
 * When cs.acquire() is executed, the code it protects is blocked from access by threads
 * running on all instances of Test, not just this one. This has the same effect as using
 * a static Mutex class
 * <b>When to use</b>
 * When porting code from a POSIX environment or when writing complex multi-threaded
 * applications where you need to work with higher level abstractions
 * @author Karthik Rangaraju
 * @see org.k2d2.framework.threadpackage.semaphore.Mutex
 */
public class CriticalSection
{
    private boolean mAcquired = false;
    private Class mClassLock;

    /**
     * Creates a critical section guard.
     * @param pObject is a reference to an object of the class in which the
     * critical section is being used
     */
    public CriticalSection(Object pObject)
    {
        mClassLock = pObject.getClass();
    }

    /**
     * Acquires the critical section. Blocks on the call if the CriticalSection is already
     * acquired. Warning: If a thread tries to acquire a CriticalSection that it already owns
     * it will result in a deadlock
     * @throws InterruptedException if the thread is interrupted when blocked
     */
    public void acquire() throws InterruptedException
    {
        synchronized (mClassLock)
        {
            while (mAcquired == true)
            {
                mClassLock.wait();
            }
            mAcquired = true;
        }
    }

    /**
     * Releases the acquired critical section. Note: No check is done to ensure that the thread
     * that acquired the CriticalSection is the one that is releasing it.
     */
    public void release()
    {
        synchronized (mClassLock)
        {
            mAcquired = false;
            mClassLock.notify();
        }
    }
}
