
package org.k2d2.framework.threadpackage.util;

import java.util.ArrayList;
import org.k2d2.framework.threadpackage.semaphore.ConditionalEvent;

/**
 * This is an implementation of a boundless blocking queue. Threads that try to
 * dequeue from an empty queue are blocked until the operation can be successful.
 * The implementation also provides for a way to force the exit of threads blocked
 * on an empty queue thus providing graceful shutdown
 *
 * @author Karthik Rangaraju
 */
public class BoundlessBlockingQueue implements BlockingQueue
{
    private ArrayList mQueue;
    private boolean mQueueClosed;
    
    /**
     * Creates a boundless blocking queue of a given size (initial size)
     * Note: Specifying a good initial size value helps the performance of
     * the queue. Under the current implementation, the size of the queue
     * increases by the initial size every time the size of exceeded.
     * @param pQueueSize The number of elements the queue can hold (initial size)
     */
    public BoundlessBlockingQueue(int pQueueSize)
    {
        // TODO: Need to find out what is the best increment value to use
        mQueue = new ArrayList(pQueueSize);
        mQueueClosed = false;
    }
    
    /**
     * This method adds an object to the queue. If the queue is closed,
     * QueueClosedException is thrown.
     * @param pObj the object to add to the queue
     * @throws InterruptedException this is required for the BlockingQueue class's
     * enqueue method that overrides this
     * @throws QueueClosedException if the method is called after the queue has been
     * closed.
     */
    public void enqueue(Object pObj) throws InterruptedException,
                                            QueueClosedException
    {
        synchronized (this)
        {
            if (mQueueClosed == true)
            {
                throw new QueueClosedException();
            }
            mQueue.add(0, pObj);
            notify();
        }
    }
    
    /**
     * This method dequeues an object from the queue. If the queue is empty
     * it blocks until something is added to the queue. The blocked thread can be
     * released by calling close() which causes QueueClosedException to be thrown.
     * @return An object from the queue of type java.lang.Object
     * @throws InterruptedException if a thread blocked on this call is
     * interrupted.
     * @throws QueueClosedException if the thread exited the method because
     * someone called close()
     * @see #close()
     */
    public Object dequeue() throws InterruptedException,
                                                QueueClosedException
    {
        synchronized (this)
        {
            if (mQueueClosed == true)
            {
                throw new QueueClosedException();
            }
            while (mQueue.size() == 0)
            {
                wait();
                if (mQueueClosed == true)
                {
                    throw new QueueClosedException();
                }
            }
            return mQueue.remove(mQueue.size() - 1);
        }
    }
    
    /**
     * This method forces threads blocked on a dequeue() call to exit throwing
     * the QueueClosedException exception. Once this method has been called,
     * to reuse the queue, one must call the reset() method. Note: This method
     * does not wait for the blocked threads to exit
     * @see #reset()
     */
    public void close()
    {
        synchronized (this)
        {
            mQueueClosed = true;
            notifyAll();
        }
    }
    
    /**
     * This method returns the state of the queue. Note: In a multi-threaded
     * environment, the return value of this method does not necessarily
     * reflect the latest state of the queue
     * @return true if the queue is closed, false otherwise
     */
    public boolean isClosed()
    {
        return mQueueClosed;
    }
    
    /**
     * This method removes all elements from the queue and resets the queue for use
     * after a call to close().
     * @see #close()
     */
    public void reset()
    {
        synchronized (this)
        {
            mQueue.clear();
            mQueueClosed = false;
        }
    }
    
    /**
     * Checks if the queue is empty or not. <b>Note</b>: In a mult-threaded
     * environment, this method's return value is not reliable as a different
     * thread can add an item to the queue.
     * @return true if the queue is empty, false otherwise
     */
    public boolean isEmpty()
    {
        synchronized (this)
        {
            return  mQueue.isEmpty();
        }
    }
    
    /**
     * This method returns the size of the queue. Since this queue is designed
     * to be accessed concurrently, you cannot rely on this return value unless
     * you guard access to the queue. This method is only provided as a
     * convenience. Most clients of a blocking queue should never need to check
     * the size of the queue
     * @return an integer representing the size of the queue
     */
    public int size()
    {
        synchronized (this)
        {
            return mQueue.size();
        }
    }
    
}
 


