.PHONY: all clean

ifndef AVR_PROGRAM
$(error AVR_PROGRAM must be set to the desired name of the executable)
endif

ifndef AVR_OBJECTS
$(error AVR_OBJECTS must be set to the list of objects to be compile)
endif

ifndef AVR_TARGET
$(error AVR_TARGET must be set to the microcontroller to compile for)
endif

# Defaults for things that are typical
AVR_INCLUDES	?=
AVR_DEFINES		?=
AVR_OPTIMIZE	?= -Os
AVR_LIBS		?=

# Defaults for specific things
AVR_CFLAGS		?= -g -Wall
AVR_LDFLAGS		?= -g

#Default paths
AVR_OBJDIR 		?= .
AVR_DEPDIR 		?= .dep
AVR_PROGDIR		?= .

# Add the full path the the objects.
AVR_FULL_OBJECTS = $(patsubst %,${AVR_OBJDIR}/%,${AVR_OBJECTS})

# Executables
CC			= avr-gcc
OBJCOPY		= avr-objcopy
OBJDUMP		= avr-objdump
RM			= rm

AVR_SPP     = ${AVR_PROGDIR}/${AVR_PROGRAM}

CFLAGS		= -MD -pipe -static -mmcu=${AVR_TARGET} ${AVR_OPTIMIZE} ${AVR_INCLUDES} ${AVR_DEFINES} ${AVR_CFLAGS}
LDFLAGS		= -mmcu=${AVR_TARGET} -Wl,--section-start,.lfuse=0xf00000 -Wl,--section-start,.hfuse=0xf00001 -Wl,--section-start,.efuse=0xf00002 -Wl,--section-start,.fuse=0xf00003 -Wl,--section-start,.lock=0xf00004 -Wl,-Map,${AVR_SPP}.map ${AVR_LIBS} ${AVR_LDFLAGS}

all: ${AVR_SPP}.elf ${AVR_SPP}.hex ${AVR_SPP}.lst Makefile

clean:
	${RM} -rf $(filter-out .,${AVR_OBJDIR} ${AVR_DEPDIR} ${AVR_PROGDIR})
	${RM} -f ${AVR_FULL_OBJECTS} ${AVR_PROGRAM}{.elf,.hex,.map,.lst} ${AVR_PROGRAM}_*.hex $(AVR_OBJECTS:.o=.lst) \
		$(patsubst %.o,src/%.lst,${AVR_OBJECTS}) $(patsubst %.o,source/%.lst,${AVR_OBJECTS}) $(patsubst %.o,sources/%.lst,${AVR_OBJECTS})
	mkdir -p ${AVR_PROGDIR}

${AVR_SPP}.elf: ${AVR_FULL_OBJECTS} Makefile
	@if [[ ! -d $(AVR_PROGDIR) ]] ; then mkdir -p $(AVR_PROGDIR) ; fi
	${CC} ${LDFLAGS} -fwhole-program -o $@ ${AVR_FULL_OBJECTS} 

${AVR_SPP}.lst: ${AVR_SPP}.elf Makefile
	@if [[ ! -d $(AVR_PROGDIR) ]] ; then mkdir -p $(AVR_PROGDIR) ; fi
	${OBJDUMP} -h -S $< > $@

${AVR_SPP}.hex: ${AVR_SPP}.elf Makefile
	@if [[ ! -d $(AVR_PROGDIR) ]] ; then mkdir -p $(AVR_PROGDIR) ; fi
	${OBJCOPY} -j .text -j .data -O srec $< $@
	for section in `${OBJDUMP} -h ${AVR_SPP}.elf | grep -B 1 ALLOC | sed -e 'n;d;' | awk '{ print $$2; }' | sed -e 's/\.\(.*\)/\1/' | grep -v -E "text|data|bss"`; do \
		${OBJCOPY} -j .$$section --change-section-lma .$$section=0 -O srec ${AVR_SPP}.elf ${AVR_SPP}_$${section}.hex; \
	done

${AVR_OBJDIR}/%.o: %.c Makefile
	@if [[ ! -d $(AVR_DEPDIR) ]] ; then mkdir -p $(AVR_DEPDIR) ; fi
	@if [[ ! -d $(AVR_OBJDIR) ]] ; then mkdir -p $(AVR_OBJDIR) ; fi
	${CC} ${CFLAGS} -o $@ -c $<
	@cp $(AVR_OBJDIR)/$*.d $(AVR_DEPDIR)/$*.P ; \
	sed -e 's/#.*//' -e 's/^[^:]*: *//' -e 's/ *\\$$//' \
		-e '/^$$/ d' -e 's/$$/ :/' < $(AVR_OBJDIR)/$*.d >> $(AVR_DEPDIR)/$*.P ; \
	$(RM) -f $(AVR_OBJDIR)/$*.d

${AVR_OBJDIR}/%.o: %.asm Makefile
	avr-as ${ASMFLAGS} -o $@  $<

%.pp: %.c Makefile
	${CC} ${CFLAGS} -E $< | less

-include $(patsubst %,$(AVR_DEPDIR)/%,$(AVR_OBJECTS:.o=.P))
