#include <avr/io.h>
#include "config.h"
#include "serial.h"
#include "error.h"

#define UBRR_FOR_BAUD(x)	(((F_CPU/16)/x)-1)
#define ERROR_FOR_BAUD(x)	(((((F_CPU/16)/(UBRR_FOR_BAUD(x)+1))/x)-1)*100)
#define ERROR_FOR_BAUD_ABS(x)	(ERROR_FOR_BAUD(x) >= 0 ? ERROR_FOR_BAUD(x) : (-1 * ERROR_FOR_BAUD(x)))
#define VALID_UBRR_FOR_BAUD(x)	(ERROR_FOR_BAUD_ABS(x) >= 1 ? 0 : UBRR_FOR_BAUD(x))

const uint16_t const gUBRR[SERIAL_BAUD_COUNT] = {
	VALID_UBRR_FOR_BAUD(2400),
	VALID_UBRR_FOR_BAUD(4800),
	VALID_UBRR_FOR_BAUD(9600),
	VALID_UBRR_FOR_BAUD(14400),
	VALID_UBRR_FOR_BAUD(19200),
	VALID_UBRR_FOR_BAUD(28800),
	VALID_UBRR_FOR_BAUD(38400),
	VALID_UBRR_FOR_BAUD(57600),
	VALID_UBRR_FOR_BAUD(76800),
	VALID_UBRR_FOR_BAUD(115200),
	VALID_UBRR_FOR_BAUD(230400),
	VALID_UBRR_FOR_BAUD(250000),
};

int serial_putc(char c, FILE * stream) {
	/* Prepend a carrage return before a newline */
	if (c == '\n') {
		serial_putc('\r', stream);
	}

	/* Wait for the hardware to be ready to send. */
	loop_until_bit_is_set(UCSRA, UDRE);

	/* Push the character out the serial port */
	UDR = c;

	return 0;
}

int serial_getc(FILE * stream) {
	/* Wait for something to be received */
	loop_until_bit_is_set(UCSRA, RXC);

	/* Return the received character */
	return UDR;
}

int serial_open(int port, serial_baud_t baud_rate, uint8_t data_bits, serial_parity_t parity, uint8_t stop_bits, FILE * fd) {
	uint16_t nubrr = 0;
	uint8_t nucsrb = 0;
	uint8_t nucsrc = 0;

	/* Make sure we were passed a valid fd */
	if (fd == NULL) {
		return EINVAL;
	}

	/* Select the port */
	if (port != 0) {
		return EINVAL;
	}

	/* Set the baud rate */
	nubrr = gUBRR[baud_rate];
	if (nubrr == 0) {
		return EINVAL;
	}

	/* Set parity */
	nucsrc &= ~(_BV(UPM1) | _BV(UPM0));
	switch(parity) {
		case SERIAL_PARITY_EVEN:
			nucsrc |= _BV(UPM1);
			break;
		case SERIAL_PARITY_ODD:
			nucsrc |= _BV(UPM1) | _BV(UPM0);
			break;
		case SERIAL_PARITY_NONE:
			/* No change */
			break;
	}

	/* Set number of data bits */
	nucsrb &= ~(_BV(UCSZ2));
	nucsrc &= ~(_BV(UCSZ1) | _BV(UCSZ0));
	switch (data_bits) {
		case 5:
			/* No change */
			break;
		case 6:
			nucsrc |= _BV(UCSZ0);
			break;
		case 7:
			nucsrc |= _BV(UCSZ1);
			break;
		case 8:
			nucsrc |= _BV(UCSZ1) | _BV(UCSZ0);
			break;
		default:
			return EINVAL;
	}

	// Set number of stop bits
	nucsrc &= ~(_BV(USBS));
	switch (stop_bits) {
		case 1:
			/* No change */
			break;
		case 2:
			nucsrc |= _BV(USBS);
			break;
		default:
			return EINVAL;
	}

	// Enable transmitter
	nucsrb |= _BV(TXEN) | _BV(RXEN);

	// Commit the new values
	UBRRH = (unsigned char) (nubrr >> 8);
	UBRRL = (unsigned char) nubrr;
	UCSRB = nucsrb;
	UCSRC = nucsrc | _BV(URSEL);

	/* Create a file handle for the port */
	fdev_setup_stream(fd, serial_putc, serial_getc, _FDEV_SETUP_RW);
	return EOK;
}
