#include <iostream>
#include <fstream>
#include <cctype>
#include <algorithm>

#include "menu.hh"

using namespace std;

static void initKeywords(KeywordMap& keywords) {
  unsigned int i = 1;
  keywords.insert(KeywordMap::value_type("begin", i++));
  keywords.insert(KeywordMap::value_type("nop", i++));
  keywords.insert(KeywordMap::value_type("submenu", i++));
  keywords.insert(KeywordMap::value_type("exec", i++));
  keywords.insert(KeywordMap::value_type("restart", i++));
  keywords.insert(KeywordMap::value_type("exit", i++));
  keywords.insert(KeywordMap::value_type("reconfig", i++));
  keywords.insert(KeywordMap::value_type("config", i++));
  keywords.insert(KeywordMap::value_type("style", i++));
  keywords.insert(KeywordMap::value_type("stylesdir", i++));
  keywords.insert(KeywordMap::value_type("stylesmenu", i++));
  keywords.insert(KeywordMap::value_type("workspaces", i++));
  keywords.insert(KeywordMap::value_type("include", i++));
  keywords.insert(KeywordMap::value_type("end", i++));
}


static unsigned int keywordLookup(const KeywordMap& keys, const string& tag) {
  const KeywordMap::const_iterator it = keys.find(tag);
  if (it != keys.end())
    return (*it).second;
  return 0;
}


static char tabToSpace(char c) {
  if (c == '\t')
    return ' ';
  return c;
}


static char toLower(char c) {
  // A -> 65, Z -> 90
  if (64 < c && c < 91) return (char)(c + 25);
  return c;
}


/* read a line into the string 'line', if it is continued then keep reading
 * until we have the whole line
 * returns the number of lines read and the line via the reference
 */
static unsigned int getFullLine(ifstream& file, string& line) {
  if (! file.good()) return 0;

  getline(file, line);
  unsigned int count = 1;
  while (file.good() && line[line.length() - 1] == '\\') {
    string tmp;
    getline(file, tmp);
    line.erase(line.length() - 1);
    line += tmp;
    ++count;
  }
  transform(line.begin(), line.end(), line.begin(), tabToSpace);
  return count;
}


bool defaultTokenError(const string& filename, const string& msg,
                       unsigned long lineno) {
  //  cerr << filename << ":" << lineno << " " << msg << '\n';
  return true;
}


string find_delimited_string(const string& piece,
			     string::size_type& pos, string::size_type len,
			     char begin, char end) {
  while(pos < len && isspace(piece[pos])) ++pos; // eat whitespace

  if (piece[pos] != begin) return "";
  ++pos; // move past begin
  while(pos < len && isspace(piece[pos])) ++pos; // eat whitespace

  string::size_type start = pos;
  while (pos < len && piece[pos] != end) {
    if (piece[pos++] == '\\') ++pos;
  }

  string::size_type stop = pos++; // move post past the end char
  while(pos < len && isspace(piece[stop])) --end; // eat whitespace

  return string(piece, start, stop - start);
}


MenuTokenizer::MenuTokenizer(const KeywordMap& keys, const char* fname,
			     TokenError errhandler): filename(fname),
						     keywords(keys),
						     error(errhandler),
						     lineno(0l) {
  file.open(filename.c_str());
}


MenuTokenizer::~MenuTokenizer(void) {
  file.close();
}


bool MenuTokenizer::next(TokenBlock& block) {
  string line;
  unsigned int count;

  bool err = false; // if err is true, we stop processing
  while (! err) {
    if ((count = getFullLine(file, line)) == 0) {
      err = true;
      continue;
    }

    block.lineno = lineno += count;

    string::size_type pos = 0, len = line.length();

    string tag = find_delimited_string(line, pos, len, '[', ']');
    if (tag.empty()) {
      if (pos != len && line[pos] != '#') {
	// this isn't a comment, complain
	const string e = "invalid input: " + string(line, pos, line.length());
	if (! error(filename, e, lineno))
	  err = true;
      }
      continue;
    }

    transform(tag.begin(), tag.end(), tag.begin(), toLower);
    unsigned int value = keywordLookup(keywords, tag);
    if (value == 0) {
      if (! error(filename, "unknown tag: " + tag, lineno))
	err = true;
      continue;
    }

    block.tag = value;

    block.name = find_delimited_string(line, pos, len, '(', ')');
    if (! block.name.empty()) {
      block.data = find_delimited_string(line, pos, len, '{', '}');
    }
    break;
  }

  return ! err;
}


int main(int argc, char *argv[]) {
  KeywordMap words;
  initKeywords(words);

  MenuTokenizer tokenizer(words, argv[1]);

  while (1) {
    TokenBlock block;
    if (tokenizer.next(block) == false)
      break;
    cout << "tag: " << block.tag << ", name: " << block.name
	 << ", data: " << block.data << '\n';
  }
  exit(0);
}
