/* ddrescueLog2badblocks compute badblocks as required by:
                                           e2fsck -vf -L badBlockList.txt */
/* Copyright  2011 Valerio Messina */

/* This file is part of ddrescue.
   ddrescue is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   ddrescue is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with ddrescue.  If not, see <http://www.gnu.org/licenses/>.      */

/* input  data: pos [Bytes hex], size [Bytes hex], blockSize [Bytes]
   output data: list of unique blocks [#] */

// for each "-" line of ddrescue log file:
//    posBlocks=rint(pos/blockSize)
//    end=pos+size-1
//    endBlocks=rint(end/blockSize)
//    output all bad blocks in range: endBlocks-posBlocks inclusive
//    the list must be unique sorted to remove duplicates

#include <stdio.h>

typedef unsigned long long int u64;

// custom code start

#define Lines            80ULL   // number of badZones "-" in ddrescue log
#define BytesPerSector  512ULL   // size in Bytes of one HDD sector
#define BytesPerBlock  4096ULL   // size in Bytes of one ext3 block
#define SectorPerCluster BytesPerBlock/BytesPerSector   // in number of sectors
                                                        // like ddrescue -c

typedef struct { // one line of ddrescue log
                   u64 pos;
                   u64 size;
               } line;

line badZones[Lines] = {/* ____pos____, ___size___ */
                         { 0x68790A800, 0x00000400 },
                         { 0x6879BA000, 0x00000400 },
                         { 0x687A0B000, 0x00000400 },
                         { 0x6BCA90A00, 0x00000400 },
                         { 0x6BE59B800, 0x00000400 },
                         { 0x6C753F200, 0x00000400 },
                         { 0x6C7599000, 0x00000200 },
                         { 0x6D081F400, 0x00000200 },
                         { 0x6D98AFC00, 0x00000200 },
                         { 0xE2EC12600, 0x00000200 },
                         { 0xE2EC21200, 0x00000200 },
                         { 0xE2EC22000, 0x00000800 },
                         { 0xE2EC22C00, 0x00000400 },
                         { 0xE58C8B600, 0x00000200 },
                         { 0xE58CC7C00, 0x00000400 },
                         { 0xE58CFB800, 0x00000200 },
                         { 0xE58D37A00, 0x00000400 },
                         { 0xE58DB1A00, 0x00000200 },
                         { 0xE58DC2A00, 0x00000200 },
                         { 0xE58DED200, 0x00000200 },
                         { 0xE58EA3400, 0x00000200 },
                         { 0xE5910FA00, 0x00000200 },
                         { 0xE5F636400, 0x00000200 },
                         { 0xE5F636A00, 0x00000200 },
                         { 0xE5F637400, 0x00000400 },
                         { 0xE5F734C00, 0x00000600 },
                         { 0xE5F92CA00, 0x00000400 },
                         { 0xE5F968E00, 0x00000400 },
                         { 0xE66030400, 0x00000400 },
                         { 0xE6606C600, 0x00000400 },
                         { 0xE660A8C00, 0x00000400 },
                         { 0xE660E4E00, 0x00000200 },
                         { 0xE6621B000, 0x00000400 },
                         { 0xE66256800, 0x00000200 },
                         { 0xE66257600, 0x00000200 },
                         { 0xE66385200, 0x00000400 },
                         { 0xE66448000, 0x00000200 },
                         { 0xE6796C000, 0x00000200 },
                         { 0xE67976800, 0x00000200 },
                         { 0xE6797AE00, 0x00000200 },
                         { 0xE6797D600, 0x00000400 },
                         { 0xE6797FE00, 0x00000200 },
                         { 0xE6799DE00, 0x00000400 },
                         { 0xE6799FA00, 0x00000400 },
                         { 0xE6C9B3200, 0x00000C00 },
                         { 0xE6C9B4E00, 0x00000400 },
                         { 0xE6C9B5600, 0x00000600 },
                         { 0xE6C9BCC00, 0x00000200 },
                         { 0xE6C9BD000, 0x00000A00 },
                         { 0xE6C9BE600, 0x00000200 },
                         { 0xE6C9BF400, 0x00000200 },
                         { 0xE6C9BF800, 0x00000400 },
                         { 0xE6C9C7C00, 0x00000600 },
                         { 0xE6C9CC200, 0x00000800 },
                         { 0xE6C9CEC00, 0x00000600 },
                         { 0xE6C9D1400, 0x00000600 },
                         { 0xE6C9D5C00, 0x00000200 },
                         { 0xE6C9DFC00, 0x00000200 },
                         { 0xE6C9E8E00, 0x00000800 },
                         { 0xE6C9EDE00, 0x00000200 },
                         { 0xE6C9EEC00, 0x00000200 },
                         { 0xE6C9EF000, 0x00000A00 },
                         { 0xE6C9F0A00, 0x00000400 },
                         { 0xE6C9F1000, 0x00000600 },
                         { 0xE6C9F2600, 0x00000200 },
                         { 0xE6CA25E00, 0x00000200 },
                         { 0xE6CA2E600, 0x00000200 },
                         { 0xEB68FDE00, 0x00000200 },
                         { 0xEB6976A00, 0x00000200 },
                         { 0xEB6977600, 0x00000400 },
                         { 0xEB6A71000, 0x00000200 },
                         { 0xEB6A71E00, 0x00000400 },
                         { 0xEB6AAD600, 0x00000400 },
                         { 0xEB6C97600, 0x00000400 },
                         { 0xEB6C97C00, 0x00000200 },
                         { 0xEB6C98A00, 0x00000200 },
                         { 0xEB6CD3C00, 0x00000400 },
                         { 0xEB6D10000, 0x00000600 },
                         { 0xEB6D4C400, 0x00000400 },
                         { 0xEB6D4CA00, 0x00000200 },
                       };

u64 blocks[Lines*SectorPerCluster]; // hold the list of badblocks with duplications
u64 unique[Lines*SectorPerCluster]; // hold the list of unique badblocks

char debug=0; // set to 1 for verbose output, set to 2 for debug

/* lines are the total lines with "-" in ddrescue logfile.
   blockSize is the ext3 cluster size in Bytes
   the output is written in unique[] */
int calcBlocks (u64 lines, u64 blockSize) {
   u64 pos, size, end;
   u64 posBlocks, endBlocks;
   u64 p=0, u=0;
   u64 b;
   
   if (blockSize==0) return -1;
   if (debug>=1) printf ("ddrescue bad zones list:\n");
   if (debug>=1) printf ("------------------------\n");
   for (u64 l=0; l<lines; l++) {
      
      // read one ddrescue "-" log line. Fill variables: pos & size, then ...
      pos=badZones[l].pos;
      size=badZones[l].size;
      if (debug>=1) printf ("l:%02llu pos:0x%llX size:0x%llX ", l, pos, size);
      if (debug>=1) printf ("\n");
      // end read one ddrescue log line.
      
      posBlocks=pos/blockSize; // integer division will trunc the result
      end=pos+size-1;
      endBlocks=end/blockSize; // integer division will trunc the result
      if (debug>=2) printf ("posBlocks:%llu endBlocks:%llu\n", posBlocks, endBlocks);
      for (b=posBlocks; b<=endBlocks; b++) {
         
         // output all bad blocks in range: endBlocks-posBlocks inclusive
         // the list must be unique sorted to remove duplicates
         blocks[p]=b;
         if (debug>=2) printf ("p:%02llu %llu ", p, blocks[p]);
         if (debug>=2) printf ("\n");
         p++;
         // end output all bad blocks in range
         
      }
   }
   if (debug>=1) printf ("\n");
   if (debug>=1) printf ("e2fsck bad blocks list with duplicates:\n");
   if (debug>=1) printf ("---------------------------------------\n");
   for (b=0; b<p; b++) {
      if (debug>=1) printf ("b:%02llu %llu ", b, blocks[b]);
      if (debug>=1) printf ("\n");
   }
   if (debug>=1) printf ("\n");
   if (debug>=2) printf ("e2fsck bad blocks list:\n");
   if (debug>=2) printf ("-----------------------\n");
   blocks[p]=0;
   for (b=0; b<p; b++) {
      if (blocks[b]!=blocks[b+1]) {
         unique[u]=blocks[b];
         if (debug>=2) printf ("b:%02llu u:%02llu %llu ", b, u, unique[u]);
         if (debug>=2) printf ("\n");
         u++;
      }
   }
   if (debug>=2) printf ("\n");
   if (debug>=1) printf ("e2fsck bad blocks list:\n");
   if (debug>=1) printf ("-----------------------\n");
   for (b=0; b<u; b++) {
      printf ("%llu ", unique[b]);
      printf ("\n");
   }
   return 0;
}

int main (int argc, char** argv) {
   int ret;
   ret = calcBlocks (Lines, BytesPerBlock);
   if (ret==-1) return -1;
   return 0;
}
