#!/bin/bash
# This is a test for GNU ddrescue bad block pass-through algoritm.
#
# See README file for details how to use it.
set -e
set -u

# Script will create following 'Device Mapper' device:
BAD_DISK_DEV="/dev/mapper/disk_with_bad_blocks"

# Where is the ddrescue binary:
DD="$HOME/ddrescue-1.17-rc3/ddrescue"

# This is generated files:
INPUT_IMAGE="/tmp/bbbtest/input_image.bin"
INPUT_SEC_SIZE=128
OUT="/tmp/bbbtest/rescued_image.bin"
LOG="/tmp/bbbtest/rescue.log"
STRACE_LOG="/tmp/bbbtest/ddrescue.strace"
STRACE_LSEEK_OFFSETS="/tmp/bbbtest/strace_lseek_offsets_in_Ki.txt"


mkdir -p "/tmp/bbbtest/"

if [ ! -x $DD ] ; then
  echo "ddrescue binary is not found. See README for instructions." >&2
  exit 1
fi

# This script responsible for content and size of our test input file
if [ ! -s $INPUT_IMAGE ] ; then
  python ./generate_image.py --image-file $INPUT_IMAGE --size-in-sectors $INPUT_SEC_SIZE
fi

# Log all 'sudo' commands on console
sudo()
{
  echo "Execute sudo $@" >&2
  /usr/bin/sudo $@
}

# Setup Linux 'Device Mapper' device with specified list of bad blocks.
# Args:
#  $1 - bad block list as list of blocks or/and ranges, for ex: "5 15 19-21 25"
# Side effect:
#  /dev/mapper/... created.
imitate_device_with_bad_blocks()
{
  local BAD_BLOCKS="$1"

  # mount $INPUT_IMAGE to free loop device if it isn't already mounted
  local LO=`sudo losetup --associated $INPUT_IMAGE | head | cut -d: -f1`

  if [ -z "$LO" ] ; then
    echo "Debug: No loop device associated with $INPUT_IMAGE"
    LO=`sudo losetup --find`
    sudo losetup $LO $INPUT_IMAGE
  else
    echo "Debug: File '$INPUT_IMAGE' has already binded to $LO"
  fi

  local sz=`sudo blockdev --getsize $LO`
  echo "Debug: $LO size is $sz sectors"
  if [ "$sz" -lt "$INPUT_SEC_SIZE" ] ; then
    echo "Error: loop device was found but it smaller then $INPUT_IMAGE" >&2
    echo "Hit: try '$ sudo losetup -d $LO' and run again."
    exit 1
  fi

  # Use Linux Device Mapper to immitate device with bad blocks
  # http://stackoverflow.com/questions/1870696/simulate-a-faulty-block-device-with-read-errors

  if sudo dmsetup status $BAD_DISK_DEV > /dev/null 2>&1 ; then
    sudo dmsetup remove $BAD_DISK_DEV
  fi

  local TBL="/tmp/bbbtest/mdtable.txt.tmp"

  : > $TBL
  local cur=0

  for b in $BAD_BLOCKS ; do

    # parse bad block range  lo-hi
    local low=`echo $b | cut -d- -f1`
    local hi=`echo $b | cut -d- -f2`
    # for single block hi=lo
    test -z "$hi" && hi=$low

    if [ $low -gt $hi ] ; then
      echo "Error: invalid bad block description '$b'. Must be 'num' or range 'low-hi'."
      exit 1
    fi

    if [ $hi -ge $sz ] ; then
      echo "Error: bad block number $b is out of $INPUT_IMAGE size" >&2
      break
    fi

    if [ $cur != $low ] ; then
      echo "$cur `expr $low - $cur` linear $LO $cur" >> $TBL
    fi

    local badsize=`expr $hi - $low + 1`
    echo "$low $badsize error" >> $TBL
    cur=`expr $hi + 1`
  done

  if [ $cur -lt $sz ] ; then
    echo "$cur `expr $sz - $cur` linear $LO $cur" >> $TBL
  fi

  sudo dmsetup create `basename $BAD_DISK_DEV` $TBL
  echo "Debug: device $BAD_DISK_DEV created"
  echo "Badblocks: $BAD_BLOCKS"
}


# Run ddrescue from current build directory under strace
#  Args:
#   $1 $2 ...  - ddrescue arguments
run_ddrescue()
{
  # _llseek on 32-bit, lseek on 64-bit
  if [ `uname -m` = "x86_64" ] ; then
    LSEEK=lseek
  else
    LSEEK=_llseek
  fi

  strace -o $STRACE_LOG -e trace=$LSEEK,read -t \
    $DD $@

  # print read offsets
  cat $STRACE_LOG | gawk 'match($0, ".+seek\\(., ([[:digit:]]+), .+", a) { print a[1] }' \
     | uniq | sed '1d' | awk '{ print $1/1024 }' > $STRACE_LSEEK_OFFSETS
}


# clean all
rm -f $OUT $LOG $STRACE_LOG $STRACE_LSEEK_OFFSETS


# Run the test
#
# Imitate 128-sector device with bad block range 27-43
imitate_device_with_bad_blocks "27-43"
# and run GNU ddrescue with cluster size 4K:
run_ddrescue -vv --no-split --cluster-size=8 --skip-size=4Ki $BAD_DISK_DEV $OUT $LOG
