; Guix package definition.
(define-module (config packages prusa-slicer)
  #:use-module (guix packages)
  #:use-module (guix gexp) 
  #:use-module (guix build-system glib-or-gtk) 
  #:use-module (guix build-system gnu) 
  #:use-module ( guix licenses)
  #:export prusa-slicer)

(define-public prusa-slicer
  (package
   (name "prusa-slicer")
   (version "2.9.0")
   (source
    (origin
     (method git-fetch)
     (uri
      (git-reference
       (url "https://github.com/prusa3d/PrusaSlicer")
       (commit (string-append "version_" version))))
     (file-name (git-file-name name version))
     (sha256 (base32 "0s1cfvhfilyv0y98asr61c6rwlgyr1hf5v5hg8q9zwmzm2bkcql3"))
     (patches (search-patches "prusa-slicer-fix-tests.patch"))
     (modules '((guix build utils)))
     (snippet
      `(begin
         ;; Prusa slicer bundles a lot of dependencies in src/ directory.
         ;; Most of them contain prusa-specific modifications (e.g. avrdude),
         ;; but others do not. Here we replace the latter with Guix packages.
         ;; Remove bundled libraries that were not modified by Prusa Slicer developers.
         (delete-file-recursively "src/hidapi")
         (delete-file-recursively "src/eigen")
         (delete-file-recursively "src/libigl/igl")
         (substitute* "CMakeLists.txt"
                      (("add_library\\(libexpat INTERFACE\\)")
                       ""))
         (substitute* "src/libigl/CMakeLists.txt"
                      (("target_link_libraries\\(libigl INTERFACE igl::core\\)") ""))
         (substitute* "src/CMakeLists.txt"
                      (("add_subdirectory\\(hidapi\\)")
                       "pkg_check_modules(HIDAPI REQUIRED hidapi-hidraw)")
                      (("include_directories\\(hidapi/include\\)")
                       "include_directories()"))
         (substitute* "src/slic3r/CMakeLists.txt"
                      (("add_library\\(libslic3r_gui.*" all)
                       (string-append
                        all
                        "\ntarget_include_directories(libslic3r_gui PUBLIC ${HIDAPI_INCLUDE_DIRS})\n"))
                      (("\\bhidapi\\b") "${HIDAPI_LIBRARIES}"))))))
   (build-system cmake-build-system)
   (arguments
    (list #:configure-flags
          #~(list "-DSLIC3R_FHS=1" ;; Use The Filesystem Hierarchy Standard.
                  "-DSLIC3R_GTK=3" ;; Use GTK+
                  ;; Use wxWidgets 3.0.x.x to prevent GUI crashes when adding support enforcers.
                  "-DSLIC3R_WX_STABLE=1"
                  (format #f "-Dlibigl_DIR=~a"
                          (search-input-directory %build-inputs
                                                  "lib/cmake/igl/"))
                  (format #f "-DCatch2_DIR=~a"
                          (search-input-directory %build-inputs
                                                  "lib/cmake/Catch2/")))
          #:phases
          #~(modify-phases %standard-phases
                           (add-after 'unpack 'fix-include-paths
                                      (lambda _
                                        (substitute* "tests/libslic3r/test_quadric_edge_collapse.cpp"
                                                     (("#include <libigl/igl/qslim.h>")
                                                      "#include <igl/qslim.h>")))))))
   (native-inputs
    (list pkg-config catch2))
   (inputs
    (list boost
          cereal
          cgal
          curl
          dbus
          eigen
          eudev
          expat
          glew
          glib
          gmp
          gtk+
          heatshrink
          hidapi
          ilmbase
          libigl
          libjpeg-turbo
          libpng
          mesa
          mpfr
          nanosvg
          nlopt
          opencascade-occt
          openvdb
          pango
          prusa-libbgcode
          ;; XXX: Using Prusa wxWidgets fork as PrusaSlicer segfaults when compiled
          ;; with regular wxwidgets.
          prusa-wxwidgets
          qhull
          tbb
          zlib))
   (home-page "https://www.prusa3d.com/prusaslicer/")
   (synopsis "G-code generator for 3D printers (RepRap, Makerbot, Ultimaker etc.)")
   (description "PrusaSlicer takes 3D models (STL, OBJ, AMF) and converts them into
G-code instructions for FFF printers or PNG layers for mSLA 3D printers.")
   (license license:agpl3)

   ;; Mark as tunable to take advantage of SIMD code in Eigen and in libigl.
   (properties '((tunable? . #t)))))
