/*
 * sched.c - initializes struct for task 0
 */
#include <entry.h>
#include <sched.h>
#include <list.h>
#include <fd.h>
#include <constants.h>
#include <mm.h>
#include <mm_address.h>
#include <segment.h>
#include <types.h>
#include <io.h>
#include <utils.h>

struct list runqueue;
struct list keyboardqueue;
int total_process;
int current_quantum;

union task_union task[NR_TASKS]
  __attribute__((__section__(".data.task")));

char task_bitmap[NR_TASKS];

struct task_struct* current() {
	struct task_struct* r;
	__asm__ __volatile__ ( "\
		movl %%esp,%0; \
		andl $0xFFFFF000,%0;"
		:"=g"(r)
	);
	return r;
}

struct task_struct* list_head_to_task_struct(struct list_head* l){
	return (struct task_struct*) l;
}

void task_schedule() {
	current_quantum--;
	current()->cpu_time++;
	if (current_quantum==0) {
		struct list_head *siguiente=current()->lh.next;
		// list_head = puntero al list_head del task_struct siguiente del actual
		task_switch((union task_union*)(list_head_to_task_struct(siguiente)));
	}
}

void task_switch (union task_union *t) {
	// actualizamos la tss para el nuevo proceso 
	tss.esp0 = (DWord) &(t->stack[KERNEL_STACK_SIZE]);
	
	// Inicialización de las páginas físicas de datos + pila
	int pag;
	int i=0;
 	for (pag=PAG_LOG_INIT_DATA_P0;pag<PAG_LOG_INIT_DATA_P0+NUM_PAG_DATA;pag++){
  		set_ss_pag(PAG_LOG_INIT_DATA_P0 + i, t->task.tp[i]);
		i++;
 	}
	set_cr3();	// FLUSH

	// Cargamos el quantum del nuevo proceso.
	current_quantum = t->task.quantum; 

	// Cambiamos el estado de los procesos, el que se va a ready, el que entra a run
	current()->status = STATUS_READY;
	t->task.status = STATUS_RUN;

	// En este momento se cambia de contexto, por lo que el volvera con el contexto del
	// nuevo proceso.
	switch_process(&t->task, current());
}

void switch_process(struct task_struct *nuevo, struct task_struct *act){
	/*
	*	Movemos el esp de act a su task struct y cargamos el esp
	*	del task struct nuevo.
	*
	*	Bloque de activacion
	*
	*	|_______________|
	*	|     %ebx      |-12
	*	|_______________|
	*	|     %edi      |-8
	*	|_______________|
	*	|     %esi      |-4
	*	|_______________|
	*	|     %ebp      |<- ebp
	*	|_______________|
	*	|     @ret      |+4
	*	|_______________|
	*	|     param2    |+8
	*	|_______________|
	*	|     param1    |+12 
	*       |_______________|
	*/
	__asm__ __volatile__ ("\
		pushl %%esi;	\
		pushl %%edi; \
		pushl %%ebx; \
		movl %%esp, %0; \
		movl %1,%%esi; \
		movl (%%esi), %%esp; \
		popl %%ebx; \
		popl %%edi; \
		popl %%esi; "
		: "=g" (act->esp) : "g" (&nuevo->esp)
	);
	
}

void init_task0()
{	
	// INICIALIZAMOS EL BITMAP DE PROCESOS
	int i;
	for (i = 0; i<NR_TASKS; i++){
		task_bitmap[i] = FALSE;
	}
	task_bitmap[0] = TRUE;

	task[0].task.pid=0;
	task[0].task.ppid=0;
	task[0].task.status= STATUS_RUN;
	task[0].task.quantum= DEFAULT_QUANTUM;
	// actualizamos el tiempo de ejecución para init0
	current_quantum=task[0].task.quantum; 

	// INICIALIZACION DE LA TABLA DE CANALES
	for (i = 0; i<NR_CHANELS; i++){
		task[0].task.tc[i].status = CERRADO;
	}
	// INICIALIZACION SALIDA ESTANDAR
	task[0].task.tc[STD_OUTPUT].dd = &t_devices[STD_OUTPUT];
	task[0].task.tc[STD_OUTPUT].mode = ESCRITURA;
	task[0].task.tc[STD_OUTPUT].status = ABIERTO;
	// INICIALIZACION SALIDA ERROR
	task[0].task.tc[STD_ERROR].dd = &t_devices[STD_ERROR];
	task[0].task.tc[STD_ERROR].mode = ESCRITURA;
	task[0].task.tc[STD_ERROR].status = ABIERTO;
	// INICIALIZACION ENTRADA ESTANDARD
	task[0].task.tc[STD_INPUT].dd = &t_devices[STD_INPUT];
	task[0].task.tc[STD_INPUT].mode = LECTURA;
	task[0].task.tc[STD_INPUT].status = ABIERTO;

	/**** INICIALIZAMOS LAS PAGINAS DEL PROCESO 0 *****/
	for (i = 0; i<NUM_PAG_DATA; i++){
		task[0].task.tp[i]=FRAME_INIT_CODE_P0 + NUM_PAG_CODE + i;
	}
	
	for (i = 0; i < USER_PAGES; i++){
		user_pages_bitmap[i] = FALSE;
	}

	for (i = 0; i< NUM_PAG_DATA; i++){
		user_pages_bitmap[i] = TRUE;
	}
	
	list_init(&runqueue); //Inicializamos la lista de procesos ready
	list_init(&keyboardqueue); //Inicializamos la lista de procesos bloqueados
	list_add(&task[0].task.lh, &runqueue); // Anadimos el proceso 0 a la lista
	total_process = 1;
}

// RETORNA CIERTO SI HAY UNO O MAS PROCESOS BLOQUEADOS POR TECLADO, FALSO EN CASO CONTRARIO
char hay_blocked (){
	return keyboardqueue.first != NULL;
}

void bloquear(struct task_struct *p){
	// el proceso p == proceso actual
	struct task_struct *next = list_head_to_task_struct(p->lh.next);
	list_del(&p->lh, &runqueue);		// ELIMINAMOS EL PROCESO DE LA RUNQUEUE
	list_add(&p->lh, &keyboardqueue);	// LO AÑADIMOS A LA KEYBOARDQUEUE
	p->status = STATUS_BLOCKED;
	task_switch(next);
}


