// OPTIMIZE
/*
 * class TCPConnection
 * class TCPListener : public TCPConnection
 * 
 * A Simplified Wrapper class for Linux TCP interface
 *
 * Martin Lueker 
 * Sept, 2004
 *
 * This class is largely a collection of 
 * TCP functions copied from other projects 
 * Though this time they have been cast into an object 
 * oriented scheme with more error handling 
 *
 * See TCPConnection.h for a full dscription.
 */

#include "TCPConnection.h"

#include<iostream>
#include<errno.h>
#include<string.h>
// for fcntl (to make non-blocking sockets)

#include <unistd.h>
#include <fcntl.h>

using namespace std;

ostream &operator<<(ostream &s, const ExceptionTCP &ex){
  s << "Error: ";
  switch(ex.type()) {
  case ExceptionTCP::NOT_CONNECTED:
    s << "NOT CONNECTED";
    break;
  case ExceptionTCP::NOT_LISTENING:
    s << "NOT LISTENING";
    break;
  case ExceptionTCP::NOT_AVAILABLE:
    s << "NOT AVAILABLE";
    break;
  case ExceptionTCP::UNKNOWN_HOST:
    s << "UNKNOWN HOST";
    break;
  case ExceptionTCP::UNREACHABLE_HOST:
    s << "UNREACHABLE HOST";
    break;
  case ExceptionTCP::ACCESS_DENIED:
    s << "ACCESS DENIED";
    break;
  case ExceptionTCP::ROOT_ONLY:
    s << "ROOT ONLY";
    break;
  case ExceptionTCP::CONNECTION_CLOSED:
    s << "CONNECTION CLOSED";
    break;
  case ExceptionTCP::MEMORY_ERROR:
    s << "MEMORY ERROR";
    break;
  case ExceptionTCP::OTHER_ERROR:
    s << "Unhandled exception"
      << " in file " << ex.m_filename
      << " at line " << ex.m_lineno;
    break;
  }
  s << std::endl;
  return s;
}

TCPConnection::TCPConnection(string hostname, int port) throw (ExceptionTCP) :
  m_socket(createSocket()),
  m_isConnected(false), 
  m_isListening(false),
  m_isListenerTypeSocket(false)
{

  try {
  addressLookup(hostname, &m_remote_address);
  // specify the port
  // We have to reformat the port so it matches 
  // Network byte order
  m_remote_address.sin_port=htons(port);

  // the funny cast here is because C++ does not
  // like these sockaddr_in -> sockaddr casts.
  // the type scheme for BSD address structures
  // is pretty strange
  pthread_testcancel();
  int ret = connect(m_socket, (sockaddr *) &m_remote_address,
                    sizeof(sockaddr_in));
  pthread_testcancel();

  if(ret<0){
    ExceptionTCP TCPfailure(ExceptionTCP::OTHER_ERROR);
    switch(errno){
      // special cases
      // we shouldn't 
      // get here unless 
      // we did something wrong 

      // These three imply that we built the socket wrong
    case EBADF:     
    case ENOTSOCK:  
    case EAFNOSUPPORT:
      // the next two only apply
      // for non-blocking sockets
    case EALREADY:  
    case EINPROGRESS:
      // The following imply that we 
      // are already connected on 
      // this socket, but we just made it
    case EISCONN:
 
    case EFAULT:
      TCPfailure=ExceptionTCP(__FILE__, 
			      __LINE__, 
			      string("Unexpected error in connect(): ")+string(strerror(errno))); 
      break;
    case ETIMEDOUT:
    case ENETUNREACH:
    case EADDRINUSE:
    case EAGAIN:
      TCPfailure=ExceptionTCP(ExceptionTCP::NOT_AVAILABLE);
      break;
    case EACCES:
    case EPERM:
    case ECONNREFUSED:
      TCPfailure=ExceptionTCP(ExceptionTCP::ACCESS_DENIED);
      break;
    default:
      TCPfailure=ExceptionTCP(__FILE__, __LINE__,
			      "undocumented error code in system connect()");
    }
    throw TCPfailure;
  };
  
  // we are now successfully connected
  m_isConnected=true;

  socklen_t length=sizeof(struct sockaddr_in);
  if(getsockname(m_socket, 
		 (struct sockaddr *) &m_local_address,
		 &length)<0){
    ExceptionTCP TCPfailure(ExceptionTCP::OTHER_ERROR); 
    switch (errno){
      // TMP
    case EBADF:
    case ENOTSOCK:
      TCPfailure=ExceptionTCP(__FILE__, __LINE__, 
			      string("Unexpected error: ") + string(strerror(errno)));
      break;
    case ENOBUFS:
      TCPfailure=ExceptionTCP(ExceptionTCP::NOT_AVAILABLE);
      break;
    case EFAULT:
      TCPfailure=ExceptionTCP(ExceptionTCP::MEMORY_ERROR);
      break;
    default:
      TCPfailure=ExceptionTCP(__FILE__, __LINE__,
			      "undocumented error code in system getsockname()");
      break;
    };
    throw TCPfailure;
  };

  } catch(ExceptionTCP& err) {
    if(m_socket >= 0) {
      ::shutdown(m_socket, SHUT_RDWR);
      ::close(m_socket);
      m_socket = -1;
    }
    throw err;
  }
}

TCPConnection::TCPConnection() :
  m_socket(createSocket()), 
  m_isConnected(false),
  m_isListening(false),
  m_isListenerTypeSocket(false)
{ 

}

TCPConnection::TCPConnection(int connectedSocket, struct sockaddr_in remote_address) :
  m_socket(connectedSocket), 
  m_isConnected(true),
  m_isListening(false),
  m_isListenerTypeSocket(false)
{ 

  m_remote_address.sin_family=AF_INET;
  m_remote_address.sin_port=remote_address.sin_port;
  m_remote_address.sin_addr.s_addr=remote_address.sin_addr.s_addr;
}

TCPConnection::~TCPConnection() {
  
  if(m_socket >=0) ::close(m_socket);
}

/* create a socket
 * and make it non-blocking */

int TCPConnection::createSocket() throw (ExceptionTCP) {
  int newSocket=socket(AF_INET, SOCK_STREAM, 0); 
  if(newSocket<0){
    switch(errno){
    case EPROTONOSUPPORT:
    case EAFNOSUPPORT:
    case EINVAL:
      throw ExceptionTCP(__FILE__, 
			 __LINE__, 
			 string("Protocol not supported: ")+string(strerror(errno))); 
      break;
    case ENFILE:
    case EMFILE:
    case EACCES:
    case ENOMEM:
      throw ExceptionTCP(__FILE__, 
			 __LINE__, 
			 string("Not Enough Resources: ")+string(strerror(errno))); 
      break;
    default:
      throw ExceptionTCP(__FILE__, 
			 __LINE__, 
			 string("Other error: ")+string(strerror(errno))); 
    }
  }


  return newSocket;
}


/* Accept uses this to create a copy of a
 * socket.  at the end of the method block,
 * it automatically calls the destructor,
 * which closes the socket we need.
 *
 * fix: use dup() (man dup for more info).
 */

TCPConnection::TCPConnection(const TCPConnection& other) :
  m_socket(dup(other.m_socket)),
  m_isConnected(other.m_isConnected),
  m_isListening(other.m_isListening),
  m_isListenerTypeSocket(other.m_isListenerTypeSocket)
{ 

  if(m_isConnected){
    m_remote_address.sin_family=AF_INET;
    m_remote_address.sin_port=other.m_remote_address.sin_port;
    m_remote_address.sin_addr.s_addr=other.m_remote_address.sin_addr.s_addr;
    
    m_local_address.sin_family=AF_INET;
    m_local_address.sin_port=other.m_local_address.sin_port;
    m_local_address.sin_addr.s_addr=other.m_local_address.sin_addr.s_addr;
  }
}

TCPConnection &TCPConnection::operator=(TCPConnection other){
  m_isConnected=other.m_isConnected;
  m_isListening=other.m_isListening;
  m_isListenerTypeSocket=other.m_isListenerTypeSocket;
 
  ::close(m_socket);
	
  m_socket=dup(other.m_socket);
  
  m_remote_address.sin_family=AF_INET;
  m_remote_address.sin_port=other.m_remote_address.sin_port;
  m_remote_address.sin_addr.s_addr=other.m_remote_address.sin_addr.s_addr;

  m_local_address.sin_family=AF_INET;
  m_local_address.sin_port=other.m_local_address.sin_port;
  m_local_address.sin_addr.s_addr=other.m_local_address.sin_addr.s_addr;

  return *this;
}

bool TCPConnection::operator==(TCPConnection &other){

  return  m_remote_address.sin_family==AF_INET &&
  m_remote_address.sin_port==other.m_remote_address.sin_port &&
  m_remote_address.sin_addr.s_addr==other.m_remote_address.sin_addr.s_addr &&
  m_local_address.sin_family==AF_INET &&
  m_local_address.sin_port==other.m_local_address.sin_port &&
  m_local_address.sin_addr.s_addr==other.m_local_address.sin_addr.s_addr;

}

void TCPConnection::addressLookup(string hostname, struct sockaddr_in *address) throw (ExceptionTCP) {
  /*
   * about IP address structures
   * 
   * We store our address information in 
   * same structure as used by the linux functions
   * bind(), and connect().
   *
   * These structures are of type struct sockaddr_in, which 
   * is a special type of struct sockaddr. (In principle 
   * sockaddr structures can have many forms, and can be 
   * used for a variety of networking protocols such as appletalk,
   * UNIX domain sockets, or IPv.6 (128 bit IP-addresses)
   * These other addressing protocols have different addressing 
   * schemes and thus different address stuctures.
   *
   * This class however deals with just IPv.4 (Plain old 
   * 32 bit-Internet) sockets
   * so we store things as sockaddr_in structures.
   *
   */
 
  address->sin_family=AF_INET;
 
  // Memory management note:
  // as I understand it gethostbyname returns
  // a STATIC pointer, thus we should not
  // try to free it.
  // M. Lueker Jun 29, 2004.
  struct hostent *hostlookup = gethostbyname(hostname.c_str());
 
  //TMP
  if(!hostlookup) throw ExceptionTCP(ExceptionTCP::UNKNOWN_HOST);
 
  address->sin_addr = *((struct in_addr *)hostlookup->h_addr);

}

unsigned int TCPConnection::cRecv(unsigned int size, 
				  void *data) 
  throw (ExceptionTCP){

  if(size==0){
    return 0;
  }

  if(!isConnected()){
    //cerr << "You are not Connected!" << std::endl;
    throw ExceptionTCP(ExceptionTCP::NOT_CONNECTED);
  }
  if(m_isListenerTypeSocket){
    // Listeners are never connected socket so we should never
    // get here.
    throw (ExceptionTCP(__FILE__, __LINE__, 
			"Not only are you not connected, but you are a listener!\n"));
  }

  int bytes_read=0;
 reread:
  pthread_testcancel();
  if( (bytes_read=::recv(m_socket, data, size, 0)) <0){
    pthread_testcancel();
    switch(errno){
    case EINTR:
      // received a signal while waiting
      // reread repeat.
      goto reread;
      break;

    case EFAULT:
      throw ExceptionTCP(ExceptionTCP::MEMORY_ERROR);
      break;
    case EINVAL:
    case ENOTSOCK:
    case ENOTCONN:
    case ECONNREFUSED:
    case EBADF:
      //cerr << "Error no. " << errno << " (" << strerror(errno) << ") !!!" << std::endl;
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("System claims bad socket: ")
			 +string(strerror(errno)));
      break;
    case EPIPE:
    case ECONNRESET:
      m_isConnected=false;
      throw ExceptionTCP(ExceptionTCP::CONNECTION_CLOSED);
      return 0;
      break;
    default:
     throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			+string(strerror(errno)));
    };
  };
  if (bytes_read==0) {
    // means the remote side has disconnected.
     m_isConnected=false;
     throw ExceptionTCP(ExceptionTCP::CONNECTION_CLOSED);
    //XXX return 0;
  } else {
    return bytes_read;
  }
  return bytes_read;
}


unsigned int TCPConnection::cSend(unsigned int size, 
				  const void *data) 
  throw (ExceptionTCP) {
  if(!isConnected()){
    //cerr << "You are not Connected!" << std::endl;
    throw ExceptionTCP(ExceptionTCP::NOT_CONNECTED);
  }
  if(m_isListenerTypeSocket){
    // Listeners are never connected socket so we should never
    // get here.
    throw (ExceptionTCP(__FILE__, __LINE__, 
			"Not only are you not connected, but you are a listener!\n"));
  }


  int bytes_sent=0;
  int bytes_per_send=size;
 
  /*
   * send the data
   *
   * Note the MSG_NOSIGNAL flag.
   *
   * We do not want asynchronous (UNIX signals) notification
   * if the socket is disconnected. Hence the MSG_NOSIGNAL flag
   * If the socket disconnects we will receive an error (EPIPE) from send()
   */
  pthread_testcancel();
  if((bytes_sent=::send(m_socket, data, bytes_per_send, MSG_NOSIGNAL))<0){
    pthread_testcancel();
    switch(errno){
    case EPIPE:
      // TMP
        // connection has been closed from our side
      m_isConnected=false;
      throw ExceptionTCP(ExceptionTCP::CONNECTION_CLOSED);
      break;
    case ETIMEDOUT:
      // TMP
      // connection has been closed from the remote side
      m_isConnected=false;
      throw ExceptionTCP(ExceptionTCP::CONNECTION_CLOSED);
      break;
    case EINTR:      // received a signal while waiting
    case EAGAIN:     // socket is busy
      bytes_sent=0;
      break;
    case EFAULT:
      // TMP
        // memory leak?
      throw ExceptionTCP(ExceptionTCP::MEMORY_ERROR);
      break;
    case EMSGSIZE:
      //TMP
      // Here we should lookup the buffer size and
      // send just that much.
      // for now just throw an exception
      throw ExceptionTCP(__FILE__, __LINE__, 
			 "TCP atomicity error? Your data is too big!");
    case EINVAL: 
    case ENOTSOCK:
    case EBADF: 
      // TMP
      // connection was never properly instantiated
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("System claims bad socket: ")
			 +string(strerror(errno)));	
      break;
    default:
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			 +string(strerror(errno)));
    };
  };
  return bytes_sent;
}

vector<TCPConnection> 
TCPConnection::waitForEvents(std::vector<TCPConnection> &connections, float timeout) throw (ExceptionTCP){

  // setup for call to select
  int maxSocket=0;
  fd_set toRead;
  FD_ZERO(&toRead);

  struct timeval tv;
  struct timeval *tv_p=NULL;

  // if the timeout is less than zero 
  // take that as a "wait forever"
  if(timeout<0)
    { tv_p = NULL; }
  else {
    tv.tv_sec=(int)floor(timeout);
    tv.tv_usec=(int)((timeout-tv.tv_sec)*1e6);
    tv_p=&tv;   
  }
  
  for(std::vector<TCPConnection>::iterator iter=connections.begin();
      iter!=connections.end();
      iter++){
    if((*iter).isConnected() || (*iter).isListening()){
      FD_SET((*iter).m_socket, &toRead);
      // Increase maxSocket if necessary
      maxSocket=(((*iter).m_socket>maxSocket) ? (*iter).m_socket : maxSocket);
    } else {
      std::cout << "Not Setting fd!" << std::endl;
    }
  }

  vector <TCPConnection> results;
  results.clear();
  
 call_select:
  // select will do the waiting for us
  pthread_testcancel();

  int numEvents=select(maxSocket+1, &toRead, NULL, NULL, tv_p);

  pthread_testcancel();
  if(numEvents <0){
    switch(errno){
    case EINTR:
      
      goto call_select; //try_again
    case EINVAL:
    case EBADF:
    case ENOMEM:
      //TMP
      // something has gone seriously wrong.
      // get out of here.
      throw ExceptionTCP(__FILE__, __LINE__, string("Error: ")+string(strerror(errno)));
      break;
    }
  } else if (numEvents==0){
    // no items found, timeout occurred first
    return results;
  }

  // Now we just need to sort out which sockets have Events
 

  for(std::vector<TCPConnection>::iterator iter=connections.begin();
      iter!=connections.end();
      iter++){
    
    if(FD_ISSET((*iter).m_socket, &toRead)){
      results.push_back(*iter);
    }
  }

  return results;
}


TCPListener::TCPListener() throw (ExceptionTCP) : 
  TCPConnection(),  // opens socket 

  m_port(-1)  
{ 
  m_isListenerTypeSocket=true;
  listenOnPort(); 
}

TCPListener::TCPListener(int port) throw (ExceptionTCP) :
  TCPConnection(), 

  m_port(port)
{
  m_isListenerTypeSocket=true;
  listenOnPort();
}

void TCPListener::listenOnPort() throw (ExceptionTCP){

  int doReuse=1;

  setsockopt(m_socket, SOL_SOCKET, SO_REUSEADDR, &doReuse, sizeof(int));


  if(m_port != -1){  // skip the bind step if m_port=-1;
    struct sockaddr_in address;
    address.sin_family=AF_INET;          // is an internet address
    address.sin_port=htons(m_port);      // bind to m_port
    address.sin_addr.s_addr=INADDR_ANY;  /* bind to any and all 
					  * available addresses
					  * (listen on all networks)
					  */


    if(bind(m_socket, (sockaddr *) &address, sizeof(sockaddr_in))<0){
      switch(errno){
      case EBADF:
      case ENOTSOCK:
	throw ExceptionTCP(__FILE__, __LINE__, 
			   string("Poorly initialized socket: ")
			   +string(strerror(errno)));
	break;
      case EINVAL:
	throw ExceptionTCP(ExceptionTCP::NOT_AVAILABLE);
	break;

      case EACCES:
	throw ExceptionTCP(ExceptionTCP::ROOT_ONLY);
	break;
	
	// Other error codes listed in the bind man 
	// page apply only to UNIX domain (AF_UNIX) sockets.
	// Since we are Talking about Internet/TCP-IP (AF_INET) 
	// sockets we don't check for these here.
	
      default:
        //cerr << "Error no. " << errno << " (" << strerror(errno) << ") !!!" << std::endl;
	throw ExceptionTCP(__FILE__, __LINE__, 
			   string("Unexplained Error: ")
			   +string(strerror(errno)));
      };
    };
  }

  pthread_testcancel();
  if(listen(m_socket, SOMAXCONN)<0){
    pthread_testcancel();
    switch(errno){
    case EADDRINUSE:
	throw ExceptionTCP(ExceptionTCP::NOT_AVAILABLE);
	break;

    case EBADF:
    case ENOTSOCK:
    case EOPNOTSUPP:
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Poorly initialized TCP socket: ")
			 +string(strerror(errno)));

    default:
	throw ExceptionTCP(__FILE__, __LINE__, 
			   string("Unexplained Error: ")
			   +string(strerror(errno)));

    }
  }
  

  socklen_t length=sizeof(sockaddr_in);
  getsockname(m_socket, 
	      (struct sockaddr *) &m_local_address,
	      &length);

  m_isListening=true;

  /*	
   * make sure that listening sockets are
   * non-blocking. That way if we try to 
   * accept when no one is trying to connect
   * then we don't wait forever, 
   * this could be a problem if waitForEvent 
   * said someone was connecting, but they 
   * gave up or died before 
   * we accepted()
   */
  if(fcntl(m_socket, F_SETFL, O_NONBLOCK)<0){
    
    // fcntl has a wide variety of functions
    // most of the error codes do not apply to 
    // the F_SETFL operation. 
    // but we check for them here just because they are listed
    // in the man page.
    
    switch(errno){
    case EFAULT:
      throw ExceptionTCP(ExceptionTCP::MEMORY_ERROR);
      break;

    case EBADF:
      // socket is invalid?
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Pooly initialized TCP socket: ")
			 +string(strerror(errno)));

    case EACCES:
    case EDEADLK:
    case EINTR:
    case ENOLCK:
      // The above flags should only 
      // happen when using fcntl for locking, not setting flags
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			 +string(strerror(errno)));
      

    case EINVAL:
    case EMFILE:
     // The above flags should only 
      // happen when using fcntl for duplicating files, not setting flags
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			 +string(strerror(errno)));


    case EPERM:
      // The above flags should only 
      // happen when adjusting the O_ACCESS flag, not the O_NONBLOCK Flag
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			 +string(strerror(errno)));

    default:
	throw ExceptionTCP(__FILE__, __LINE__, 
			   string("Unexplained Error: ")
			   +string(strerror(errno)));


    }
  }; 

  return;

}


TCPConnection TCPListener::accept(string hostname) throw(ExceptionTCP){
  struct sockaddr_in remote_address;
  socklen_t length=sizeof(sockaddr_in);
 
  if(!isListening()){
    throw ExceptionTCP(ExceptionTCP::NOT_LISTENING);
  }

  TCPConnection unconnected; // an unconnected socket.

 try_again:
  pthread_testcancel();
  int connectionSocket=::accept(m_socket, (sockaddr*) &remote_address, 
				&length);
  pthread_testcancel();

  if(connectionSocket<0){
    switch(errno){

    case EINTR: 
      // we were interupted by a signal, give it another go.
      goto try_again;

    case ECONNABORTED:
      // There was a connection pending but it has since aborted.
    case EAGAIN:  
      // There are no connections pending.
    case ENETDOWN: 
    case EPROTO: 
    case ENOPROTOOPT: 
    case EHOSTDOWN: 
    case ENONET:
    case EHOSTUNREACH: 
    case ENETUNREACH:
      /* from the man page:
       * 
       * "Linux accept passes already-pending network errors on the new socket as
       * an error code from accept.   This  behaviour  differs  from  other  BSD
       * socket  implementations.  For reliable operation the application should
       * detect the network errors defined for the  protocol  after  accept  and
       * treat  them  like EAGAIN by __retrying__ In case of TCP/IP these are ENET-
       * DOWN, EPROTO, ENOPROTOOPT, EHOSTDOWN, ENONET, EHOSTUNREACH, EOPNOTSUPP,
       * and ENETUNREACH."
       *
       * Following this advice we treat all these errors like EAGAIN.  Namely
       * We assume that nobody is trying to 
       * contact us at this exact moment and so we return for now.
       */
      


      // return an invalid (not connected) TCP Connection
      return unconnected;

    case EFAULT:
      throw ExceptionTCP(ExceptionTCP::MEMORY_ERROR);
      break;
    case EMFILE:     // too many file descriptors in this process
    case ENFILE:     // too many open file descriptors on this system
      throw ExceptionTCP(ExceptionTCP::NOT_AVAILABLE);
      break;
    case EPERM:
      //std::cerr << "Connection from " << inet_ntoa(remote_address.sin_addr) << " rejected by Firewall" << std::endl; 
      throw ExceptionTCP(ExceptionTCP::ACCESS_DENIED);
      break;
    case EBADF:      // socket is not a file descriptor
    case ENOTSOCK:   // socket is not a socket
    case EOPNOTSUPP: // socket is not a tcp socket
    case EINVAL:     // socket is not listening
      //std::cerr << "Socket was not correctly configured: " << strerror(errno) << std::endl;
      /* these are all errors that say our socket is not correctly configured */
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Poorly initialized TCP socket: ")
			 +string(strerror(errno)));
      break;

    default:
      throw ExceptionTCP(__FILE__, __LINE__, 
			 string("Unexplained Error: ")
			 +string(strerror(errno)));
      break;
    };
  };
  
  TCPConnection result(connectionSocket, remote_address);

  if(hostname != ""){
    struct sockaddr_in desiredAddress;
    addressLookup(hostname, &desiredAddress);
    // the user has specified to only accept connections
    // from a particular client
    if(remote_address.sin_addr.s_addr != desiredAddress.sin_addr.s_addr){
      //cerr << "Connection request received from wrong host, will try again\n";
      goto try_again;
    }
  }

  // record the local address
  if(getsockname(connectionSocket, 
		 (struct sockaddr *) &(result.m_local_address),
		 &length)<0){
    switch (errno){
      // TMP
    case EBADF:
    case ENOTSOCK:
      break;
    case ENOBUFS:
      break;
    case EFAULT:
      break;
    default:
      break;
    };

  };

  return result;
}



/*
 * Recv has same functionality as cRecv only it has the 
 * additional step of converting the data to a string
 * afterwards 
 */

string TCPConnection::Recv(unsigned int size) 
  throw (ExceptionTCP){


  //cerr << "SIZE: " << size <<endl;

  char *char_data=new char[size+1];
  unsigned int bytes_received=0;
  try{
    bytes_received=cRecv(size, (void *) char_data);
  } catch (...){
    delete[] char_data;
    throw;
  }
  //cerr << "BYTES: " << bytes_received <<endl;

  char_data[bytes_received]='\0';

  string result="";
  result+=char_data;

  delete[] char_data;

  return result;
}


unsigned int TCPConnection::Send(const string &data) 
  throw (ExceptionTCP){
  // size may be smaller than the vector we've been 
  // or vice-versa, here we will just send 
  // the smaller of the two.

  unsigned int bytes_sent=cSend(data.length(), (void *) data.c_str());
  return bytes_sent;
}


/*
 *  Make a call to waitForEvents, passing only 'this'
 *
 *  If any events happened (i.e. waitForEvents returns
 *  some non-empty vector) then return true;
 *
 */
bool TCPConnection::waitForEvent(float timeout)
  throw (ExceptionTCP) {
  
  vector<TCPConnection> connList(1, *this);
  vector<TCPConnection> result=waitForEvents(connList, timeout);
  bool retval=(result.size() != 0);
  return retval;
}

void TCPConnection::setWaitState(bool wait1_nowait0) throw (ExceptionTCP) {
  int delaystate=!(wait1_nowait0);
  if(setsockopt(m_socket, SOL_SOCKET, TCP_NODELAY, &delaystate, sizeof(int))<0){
    throw ExceptionTCP(__FILE__, __LINE__, 
		       string("Unable to set socket in TCP_NOWAIT state: ")+
		       string(strerror(errno))
		       );
  }
}

int TCPConnection::setMaxSegmentSize(int size) {
  if(setsockopt(m_socket, SOL_SOCKET, TCP_MAXSEG, &size, sizeof(int))<0){
    throw ExceptionTCP(__FILE__, __LINE__, 
		       string("Unable to set maximum segment size for socket: ")+
		       string(strerror(errno))
		       );
  }
  return getMaxSegmentSize();
}

int TCPConnection::setSendBufferSize(int size) {
  if(setsockopt(m_socket, SOL_SOCKET, SO_SNDBUF, &size, sizeof(int))<0){
    throw ExceptionTCP(__FILE__, __LINE__, 
		       string("Unable to set send buffer size for socket: ")+
		       string(strerror(errno))
		       );
  }
  return getSendBufferSize();
}

int TCPConnection::getSendBufferSize() {
  int size;
  socklen_t length=sizeof(int);
  if(getsockopt(m_socket, SOL_SOCKET, SO_SNDBUF, &size, &length)<0){
    throw ExceptionTCP(__FILE__, __LINE__, 
		       string("Unable to get send buffer size for socket: ")+
		       string(strerror(errno))
		       );
  }
  return size;
}

int TCPConnection::getMaxSegmentSize() {
  int size;
  socklen_t length=sizeof(int);
  if(getsockopt(m_socket, SOL_SOCKET, TCP_MAXSEG, &size, &length)<0){
    throw ExceptionTCP(__FILE__, __LINE__, 
		       string("Unable to get maximum segment size for socket: ")+
		       string(strerror(errno))
		       );
  }
  return size;
}

string TCPConnection::getRemoteAddress(){
  return string(inet_ntoa(m_remote_address.sin_addr));
}

string TCPConnection::getLocalAddress(){
  return string(inet_ntoa(m_local_address.sin_addr));
}

int TCPConnection::getRemotePort(){
  return ntohs(m_remote_address.sin_port);
}

int TCPConnection::getLocalPort(){
    return ntohs(m_local_address.sin_port);
}

void TCPConnection::close(){
  pthread_testcancel();
  ::close(m_socket);
  pthread_testcancel();
  m_socket=-1;
  m_isConnected=false;

}

void TCPConnection::shutdown(){
  pthread_testcancel();
  ::shutdown(m_socket, 2);
  pthread_testcancel();
  m_socket=-1;
  m_isConnected=false;

}
