// OPTIMIZE
/* 
 * class TCPConnection
 * class TCPListener : public TCPConnection
 * 
 * A Simplified Wrapper class for Linux TCP interface
 *
 * Martin Lueker 
 * Sept, 2004
 *
 * Synopsis:
 *
 * Supports either "client" or "server" mode.
 *
 * Definitions:
 * Host: a Physical Computer. Each computer has a unique address. 
 * Port: a number which routes network requests 
 *       to the appropriate program on some host.
 *       (If you think of the Host Address
 *        as company telephone number, think of the 
 *        port as an extension).
 * Socket:  One half of a network connection. Two sockets
 *          are needed to make a connection, one on each host.
 *          In any TCPConnection one socket is a "server" 
 *          or "listener",  which waits for incoming requests.  
 *          The other side, called a "client", initiates a 
 *          connection to a listening server.
 *
 * Client Mode:
 * In client mode we specify the host address and port of the 
 * program we are trying to connect to.
 * The host and port may be passed in the constructor.
 * If the remote host is not listening then the socket
 * will not be connected.  
 * 
 * One should always check to see if a socket is indeed 
 * connected.
 *
 * Server mode:
 * In server mode we "listen" for incoming requests, if 
 * another computer tries to connect we may accept the 
 * request, with the accept() method.
 * 
 * With the TCPListen class you can specify a port to listen on. 
 * You can alternatively let listen pick a port for you.
 * This is only useful though if you have some way of 
 * letting other programs know which port has been chosen.
 *
 * By default the accept method will allow connections 
 * from any host.  For access protection you should use "ipchains" 
 * or some similar firewall program to restrict access.
 *  
 * You can also perform simple access control by passing 
 * a hostname to accept() 
 * 
 * TCPListeners vs. TCPConn
 *
 * TCPListeners are not connected sockets, though they can accept sockets
 * and make connections, they cannot transmit data.
 *
 * TCPConnections can transmit and receive data.
 */ 

#ifndef TCP_CONNECTION_H
#define TCP_CONNECTION_H

#include <exception>
#include <string>
#include <vector>

// system includes
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <math.h>
//#include "ExceptionMux.h"

#include <pthread.h>

    class ExceptionTCP {
    public:
      enum failure_mode {
	NOT_CONNECTED,     /* Connection was never established */
	NOT_LISTENING,     /* This socket is a normal socket not
			    * a listening socket */
	NOT_AVAILABLE,     /* The network or there are temporarily not
			    * enough resources (something is down or busy) */
	UNKNOWN_HOST,      /* Couldn't find the name of the host */
	UNREACHABLE_HOST,  /* The Host did not respond */
	ACCESS_DENIED,     /* Normal Connections:
			    * The other side was contacted 
			    * However we are not allowed to 
			    * access that port.
			    *
			    * Listeners:
			    * We cannot accept this connection 
			    * because of firewall rules
			    */
	ROOT_ONLY,         /* Only root can listen on the 
			    * port we are trying to access */
	CONNECTION_CLOSED, /* The other side has hung up 
			    */
	MEMORY_ERROR,      /* Somebody messed up and 
			    * passed a bad pointer */
	OTHER_ERROR        /* Some other problem 
			    * probably means a bug
			    * in TCPConnection class, 
			    * or some error in the setup
			    * or system configuration
			    */
      };
      ExceptionTCP(failure_mode fm) : m_failureMode(fm) {};
      ExceptionTCP(char *file, int line, std::string comment) :
	m_failureMode(OTHER_ERROR), m_filename(file), m_lineno(line), m_comment(comment) {};
      //virtual ~ExceptionTCP() {};

      failure_mode type() const { return m_failureMode; };
    private:
      failure_mode m_failureMode;
    public:
      char *m_filename;
      int m_lineno;

      friend std::ostream &operator<<(std::ostream&, const ExceptionTCP&);

    private:
      std::string m_comment;
      ExceptionTCP();
    };


    class TCPConnection  {
      // for some reason child classes
      // cannot explicitly access 
      // protected constructors?
  
      // make TCPListener a freind for now
      friend class TCPListener; 

    public:
      TCPConnection(std::string hostname, int port) throw (ExceptionTCP);
      TCPConnection(const TCPConnection&);

      // create an unconnected socket.
      TCPConnection(); 

      virtual ~TCPConnection();

      void close(void);

      //REALLY CLOSE-- ignore the fact that there
      //may be any other copies of this socket lying 
      //around.
      void shutdown(void);

      // For sending ascii strings

      virtual std::string Recv(unsigned int byteCount) 
	throw(ExceptionTCP);

      virtual unsigned int Send(const std::string& data)
	throw (ExceptionTCP);

      int getFd() { return m_socket;};

      /* For sending binary data. 
       *
       * It is provided here for convenience,
       * though the user should be warned that it
       * has limited applications.  There
       * are some restraints on the 
       * class T.
       *
       * Warning when transmitting binary data:
       * Only use these calls
       * for primitive types or simple 
       * structures.  Use no Standard C++
       * classes, only ints, chars, shorts, 
       * and the like.
       *
       * Be very careful when using 
       * these calls for 
       * structures containing pointers,
       * The data refered to by the 
       * pointer is not sent.
       * 
       * When sending data between 
       * different computer 
       * architectures (i.e. from an 
       * intel to a mac) 
       * 
       * It is recommended that ints 
       * and shorts are stored in 
       * "Network byte order".
       * 
       * Data can be translated to
       * network byte order using 
       * linux 
       * using the htons, and htonl
       * functions.  (See the man page
       * for details.)
       *
       */

      template<class T> int binRecv(T& data) 
	throw(ExceptionTCP){
	return cRecv(sizeof(T), (void *)&data);
      }

      template<class T> int binSend(const T& data)
	throw (ExceptionTCP){
	return cSend(sizeof(T), (void *)&data);
      }

#if 0
      template<class T> int binRecv(vector<T>& data, int number=0) 
	throw(ExceptionTCP){
	int tmp=0;
	int bytes_recd=0;    
	bytes_read+=ntohl(binRecv(tmp));
    
	data.resize(size);
	for(int i=0; i<size; i++){
	  bytes_recd+=binRecv(data[i]);      
	}
	return bytes_recd;
      }
#endif

      template<class T> int binSend(const std::vector<T>& data)
	throw (ExceptionTCP){
    
	int size=data.size();
	int bytes_sent=0;    
	int tmp=htonl(size);
    
	bytes_sent+=binSend(tmp);
	for(int i=0; i<data.size(); i++){
	  bytes_sent+=binSend(data[i]);      
	}
	return bytes_sent;
      }



      /* 
       * C-style functional equivalents for 
       * use in performance applications
       * 
       * Buffer must already be allocated, either 
       * as a static array or by calling new().
       *
       * Can only be used by normal TCPConnections
       * (Not Listeners).
       * 
       */

      virtual unsigned int cRecv(unsigned int size,
				 void *buffer) 
	throw (ExceptionTCP);

      virtual unsigned int cSend(unsigned int size, 
				 const void *buffer) 
	throw (ExceptionTCP);
  

      /*
       * waitForEvent:
       *
       * Does not return until there is activity on 
       * the socket.  (or there is a timeout) 
       * For normal TCPConnection objects this
       * means that there is data to be read.
       *
       * For TCPListener objects this means that there is
       * an incoming connection pending.
       * 
       * Ignores unconnected sockets
       * 
       * returns true if something happened, false if a timeout occurs.
       *
       * If Timeout==-1 this function will wait forever
       *
       */
      bool waitForEvent(float timeout=-1) throw (ExceptionTCP);
      static std::vector<TCPConnection> 
	waitForEvents(std::vector<TCPConnection> &, float timeout=-1) 
	throw (ExceptionTCP);

      bool isListening() { return m_isListening; };
      bool isConnected() { return m_isConnected; };

      /* 
       * retreive some information about the connection
       *
       * non Connected sockets return either 0 or empty strings
       *
       */

      /* get the IP-address of the remote host in nnn.nnn.nnn.nnn format */
      std::string getRemoteAddress();
      /* get the address this machine is using for this connection
       * (many machines may have more than one address) */
      std::string getLocalAddress();
      /* Get the port number used on this socket--eventually both
       * clients and servers are assigned a socket.  It may be sometimes
       * useful to know the client's port number as well as the servers'
       * port */
      int getRemotePort();
      int getLocalPort();

      /*
       * some useful operators
       */
      TCPConnection& operator=(TCPConnection other);

      bool operator==(TCPConnection &other); 


      /***************************** 
       *                           *  
       *  Other socket properties  *
       *                           *
       *****************************/
  
      /*
       * By default TCP gathers many small packets and sends them only when there
       * is a large amount of data.  This reduces network traffic, but if puts 
       * large delays in protocols with small packets (i.e. modbus).
       *
       * setWaitState(0) makes small packets move faster at the expense of
       * extra network traffic.  This is done by turning on the 
       * TCP "NO_DELAY" flag.
       */
      void setWaitState(bool wait1_nowait0) throw (ExceptionTCP);
  
      /*
       * The Maximum Segment Size determines 
       * The largest possible amount that can be sent
       * in a single TCP packet.  
       *
       * Since TCP is a streaming protocol, it is not 
       * important to know this size when performing 
       * socket send or recv operations.
       * If one sends data in one very large
       * call to send, the data will be automatically
       * broken up into reasonably sized segments.
       * The segment size is also completely transparent
       * at the receiving end.
       *
       * However it is often useful to adjust the 
       * segment size in order to manage the
       * network load in a given subsystem.
       *
       */
      int setMaxSegmentSize(int size);
      int getMaxSegmentSize();
      int setSendBufferSize(int size);
      int getSendBufferSize();

    protected:
      int m_socket;
      bool m_isConnected;
      bool m_isListening;
      bool m_isListenerTypeSocket;
  
      struct sockaddr_in m_remote_address;
      struct sockaddr_in m_local_address;

      pthread_mutex_t m_mutex;
  

      // helper function for connect.
      // is protected to hide linux address format.
      static void addressLookup(std::string hostname, struct sockaddr_in *) throw (ExceptionTCP);
  
      int createSocket(void) throw (ExceptionTCP); 
  
      // for use only by TCPListener
      TCPConnection(int Socket, struct sockaddr_in remote_address);
  
    };


    class TCPListener : public TCPConnection { 
    public: 
      TCPListener(void) throw (ExceptionTCP);
      TCPListener(int port) throw (ExceptionTCP);
  
      TCPConnection accept(std::string hostname="") throw (ExceptionTCP);

    protected:
  
      int m_port;
      // opens m_port for listening
      // if m_port is -1 this picks an arbitrary port
      void listenOnPort() throw (ExceptionTCP);
  
    };

#endif
