#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include "TCPConnection.h"

using namespace std;

//
// Globals concerning ROACH-internal data format as of 14 June 2011
//

const unsigned int STD_HEADER = 598070372;

// size of the shared FPGA/PPC memory in bytes
const size_t registerSize = 16384;

// number of integers in each timesample. There is some padding with
// useless data that we will remove.
const size_t integersPerSample = 256;
const size_t actualDataPerSample = 192;

// # of timesamples in each register
size_t numTimesamplesPerPacket = 4;
size_t numTimesamplesPerRegister = registerSize / sizeof(int) /
  integersPerSample;
size_t numPacketsPerRegister = numTimesamplesPerRegister /
  numTimesamplesPerPacket;

// Each datagram will contain numTimesamples of data from two
// registers combined. Want to know the datagram size in bytes and
// in integers.
const size_t headerSize = 2;
const size_t timestampSize = 2;
size_t timesampleSize = headerSize + timestampSize + actualDataPerSample*2;
size_t datagramBufferSize = timesampleSize * numTimesamplesPerPacket *
  numPacketsPerRegister;
size_t datagramBufferSizeInBytes = timesampleSize * numTimesamplesPerPacket *
  numPacketsPerRegister * sizeof(int);
size_t datagramSizeInIntegers = numTimesamplesPerPacket * timesampleSize;
size_t datagramSizeInBytes = datagramSizeInIntegers * sizeof(int);

inline bool read_bram(ifstream* bram_file, char* buffer,
		       int offset, int nToRead) {

  bram_file->seekg(offset, ios::beg);
  bram_file->read(buffer, nToRead);

  // These lines convert 'buffer' to signed value
  // if(buffer > 2048) {
  // buffer = 4096-buffer;
  // }
  // buffer = (int) 0xFFFC0000 | buffer;

  return bram_file->good();
}

// We're no longer using UDP because of lost packets, but we'll still 
// refer to the data package as a "datagram". This function produces
// a datagram from the data read out from the two chunks of shared 
// FPGA/PPC memory.
inline bool package_data(signed int* bottom, signed int* top,
			 signed int* datagram) {
  size_t skipToReachData = integersPerSample - actualDataPerSample;
  for(size_t j=0; j < numTimesamplesPerRegister; ++j) {
    datagram[j*timesampleSize] = STD_HEADER;
    datagram[j*timesampleSize+1] = 0; // dummy for checksum
    datagram[j*timesampleSize+2] = top[j*integersPerSample];
    datagram[j*timesampleSize+3] = bottom[j*integersPerSample];
    
    for(size_t k=0; k < actualDataPerSample; ++k) {
      datagram[j*timesampleSize + 3 + k*2 + 1] =
	top[j*integersPerSample + skipToReachData + k];
      datagram[j*timesampleSize + 3 + k*2 + 2] = 
	bottom[j*integersPerSample + skipToReachData + k];
    }
  }

  return true;
}

inline bool isfile(string filename) {
  ifstream lock(filename.c_str());
  bool exists = lock.is_open();

  if(exists) {
    lock.close();
  }

  return exists;
}

int main(int argc, char** argv) {
  stringstream topReg1Filename, btmReg1Filename;
  stringstream topReg2Filename, btmReg2Filename;
  stringstream writing1RegFilename, writing2RegFilename;

  if(argc != 3) {
    fprintf(stderr, "Please provide process ID and board #s as arguments!\n");
    exit(1);
  }

  int pid = atoi(argv[1]);
  int board = atoi(argv[2]);
  //Modified for new firmware (a119_23_6a84_2012_Feb_12_0225.bof)
  string topRegister1("Top_part_1");
  string btmRegister1("Bottom_part_1");
  //string topRegister1("qdr0_memory");
  //string btmRegister1("qdr1_memory");
  string topRegister2("Top_part_2");
  string btmRegister2("Bottom_part_2");
  string writing1Register("start_write_1");
  string writing2Register("start_write_2");
  string lockfile("/root/data_server_lock");

  topReg1Filename << "/proc/" << pid << "/hw/ioreg/" << topRegister1;
  btmReg1Filename << "/proc/" << pid << "/hw/ioreg/" << btmRegister1;
  topReg2Filename << "/proc/" << pid << "/hw/ioreg/" << topRegister2;
  btmReg2Filename << "/proc/" << pid << "/hw/ioreg/" << btmRegister2;
  writing1RegFilename << "/proc/" << pid << "/hw/ioreg/" << writing1Register;
  writing2RegFilename << "/proc/" << pid << "/hw/ioreg/" << writing2Register;

  ifstream top_file1(topReg1Filename.str().c_str(), ios::binary);
  ifstream btm_file1(btmReg1Filename.str().c_str(), ios::binary);
  ifstream top_file2(topReg2Filename.str().c_str(), ios::binary);
  ifstream btm_file2(btmReg2Filename.str().c_str(), ios::binary);
  ifstream writing_file1(writing1RegFilename.str().c_str(), ios::binary);
  ifstream writing_file2(writing2RegFilename.str().c_str(), ios::binary);

  if(!top_file1.is_open() || !btm_file1.is_open() ||
     !top_file2.is_open() || !btm_file2.is_open() ||
     !writing_file1.is_open() || !writing_file2.is_open()) {
    cerr << "[ERROR] couldn't open one of the registers: " <<
      topReg1Filename.str() << " or " <<
      btmReg1Filename.str() << " or " <<
      topReg2Filename.str() << " or " <<
      btmReg2Filename.str() << " or " <<
      writing1RegFilename.str() << " or " <<
      writing2RegFilename.str() << endl;
    exit(1);
  }

  size_t numInts = registerSize / sizeof(signed int);
  signed int* btm_buffer = new signed int[numInts];
  signed int* top_buffer = new signed int[numInts];
  signed int* datagram = new signed int[datagramBufferSize];

  ssize_t bytesSent = 0;

  //string daqHostname("10.5.5.32");
  string daqHostname("192.168.40.77");
  TCPConnection* dataConnection = NULL;

  // volatile ensures that these get updated every loop iteration
  // compilers can sometimes try to optimize in a way that is bad
  static volatile int wrt1 = 0;
  static volatile int wrt2 = 0;
  bool success = false;
  static volatile bool takeData = isfile(lockfile);
  while(takeData) {

    // first ensure that we have an active connection
    while(dataConnection == NULL || !dataConnection->isConnected()) {
      takeData = isfile(lockfile);
      if(!takeData) {
	goto SHUTDOWN;
      }

      delete dataConnection; // deallocate lost connection (NULL-safe!)
      dataConnection = NULL;

      try {
	dataConnection = new TCPConnection(daqHostname, 51700 + board);
      }	catch (ExceptionTCP etcp) {
	// cerr << etcp << endl;
	continue;
      }
    }
    

    do {
      writing_file1.sync();
      writing_file1.seekg(0);
      writing_file1.read((char*) &wrt1, 4);
    } while(wrt1 == 1);

    // read both bottom and top parts
    success = read_bram(&top_file1, (char*) top_buffer, 0, registerSize);
    success = read_bram(&btm_file1, (char*) btm_buffer, 0, registerSize);

    if(!success) {
      break;
    }

    success = package_data(btm_buffer, top_buffer, datagram);
    try {
      bytesSent = dataConnection->cSend(datagramBufferSizeInBytes, datagram);
    } catch (ExceptionTCP etcp) {
      // cerr << etcp << endl;
      takeData = isfile(lockfile);
      continue;
    }

    do {
      writing_file2.sync();
      writing_file2.seekg(0);
      writing_file2.read((char*) &wrt2, 4);
    } while(wrt2 == 1);

    // read both bottom and top parts
    success = read_bram(&top_file2, (char*) top_buffer, 0, registerSize);
    success = read_bram(&btm_file2, (char*) btm_buffer, 0, registerSize);

    if(!success) {
      break;
    }

    success = package_data(btm_buffer, top_buffer, datagram);
    try {
      bytesSent = dataConnection->cSend(datagramBufferSizeInBytes, datagram);
    } catch (ExceptionTCP etcp) {
      // cerr << etcp << endl;
      takeData = isfile(lockfile);
      continue;
    }

    takeData = isfile(lockfile);
  }

 SHUTDOWN:

  if(dataConnection != NULL) {
    dataConnection->shutdown();
    dataConnection->close();
  }

  top_file1.close();
  btm_file1.close();
  top_file2.close();
  btm_file2.close();
  writing_file1.close();
  writing_file2.close();

  delete dataConnection;
  delete[] datagram;
  delete[] btm_buffer;
  delete[] top_buffer;
}
