#include <iostream>
#include <sstream>
#include <fstream>
#include <ctime>
#include <iomanip>
#include <queue>

#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/scoped_array.hpp>
#include <boost/thread.hpp>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <cryptopp/sha.h>

#define IOTESTER_BUFSIZE 512

namespace fs = boost::filesystem;

boost::mutex bigmutex;
uint32_t count = 0;
uint32_t max = 0;
uint32_t wrong_digests;
boost::posix_time::ptime hundred_started;
boost::posix_time::time_duration max_write_time(0, 0, 0, 0);
boost::posix_time::time_duration max_read_time(0, 0, 0, 0);

class DirDispatcher
	{
	public:
	DirDispatcher(fs::path datadir)
		{
		for(fs::directory_iterator i(datadir) ; i != fs::directory_iterator() ; ++i)
			{
			if(i->path().filename().string().size() == 2)
				{
				for(fs::directory_iterator j(i->path()) ; j != fs::directory_iterator() ; ++j)
					if(i->path().filename().string().size() == 2)
						m_dirs.push(j->path());
				}
			}
		}
	~DirDispatcher() {}

	fs::path dispatch_dir()
		{
		boost::lock_guard<boost::mutex> lock(m_mutex);
		fs::path out;
		if(!m_dirs.empty())
			{
			out = m_dirs.front();
			m_dirs.pop();
			}

		return out;
		}

	private:
	std::queue<fs::path> m_dirs;
	boost::mutex m_mutex;
	};

void fill_buffer_randomly(unsigned char* buffer, uint32_t len)
	{
	uint64_t* intbuf = (uint64_t*) buffer;
	for(int i = 0 ; i < len / 8 ; ++i)
		intbuf[i] = (uint64_t) rand();
	}

std::string digest_to_hex(const boost::scoped_array<unsigned char>& digest_buffer, std::size_t len)
	{
	std::ostringstream digeststr;
	for(int i = 0 ; i < len ; ++i)
		digeststr << std::hex << std::setw(2) << std::setfill('0') << (((unsigned int)digest_buffer[i]) & 0xff);

	return digeststr.str();
	}

std::string random_file_name()
	{
	std::ostringstream out;
	for(int i = 0 ; i < 16 ; ++i)
		out << std::hex << std::setw(2) << std::setfill('0') << (uint32_t(rand()) & 0xff);

	return out.str();
	}

void create_files(const fs::path& datadir)
	{
	fs::space_info dataspace = fs::space(datadir);
	unsigned char buffer[IOTESTER_BUFSIZE];

	fs::create_directories(datadir / "tmp");
	while(dataspace.available > dataspace.capacity * 0.01 && (max == 0 || count < max))
		{
		CryptoPP::SHA1 checksum;
		// TODO global list of open tmp files to avoid filename collisions - probably not a big deal with 16 char random filenames...
		fs::path tmp_path = datadir / "tmp" / random_file_name();
		fs::ofstream file(tmp_path, std::ios::out);
		for(int i = 0 ; i < (1024 * 1024) / IOTESTER_BUFSIZE ; ++i)
			{
			fill_buffer_randomly(buffer, IOTESTER_BUFSIZE);
			checksum.Update(buffer, IOTESTER_BUFSIZE);
			file.write((char*)buffer, IOTESTER_BUFSIZE);
			}
		file.close();

		boost::scoped_array<unsigned char> digest_buffer(new unsigned char[checksum.DigestSize()]);
		checksum.Final(digest_buffer.get());
		std::string digest = digest_to_hex(digest_buffer, checksum.DigestSize());

		std::string byte1 = digest.substr(0,2);
		std::string byte2 = digest.substr(2,2);

		fs::create_directories(datadir / byte1 / byte2);
		fs::rename(tmp_path, datadir / byte1 / byte2 / digest);

			{
			boost::lock_guard<boost::mutex> biglock(bigmutex);
			++count;
			if(count % 100 == 0)
				{
				boost::posix_time::time_duration write_time = boost::posix_time::microsec_clock::universal_time() - hundred_started;
				std::cout << "Wrote 100 MiB of data in " << write_time.total_milliseconds() << " milliseconds";
				std::cout << " (" << (100.0 / write_time.total_milliseconds()) * 1000 << " Megabytes/second) total files written: " << count << std::endl;
				hundred_started = boost::posix_time::microsec_clock::universal_time();
				if(write_time > max_write_time)
					max_write_time = write_time;

				}
			}
		dataspace = fs::space(datadir);
		}
	}

void read_files(const fs::path& subpath)
	{
	unsigned char buffer[IOTESTER_BUFSIZE];
	std::vector<fs::path> todelete;
	for(fs::directory_iterator i(subpath) ; i != fs::directory_iterator() ; ++i)
        	{
		if(i->path().filename().string().size() == 40)
        		{
	                std::string filename = i->path().filename().string();

			std::ifstream file(i->path().c_str(), std::ios::in);
			CryptoPP::SHA1 checksum;
			for(int j = 0 ; j < (1024 * 1024) / IOTESTER_BUFSIZE ; ++j)
				{
				file.read((char *)buffer, IOTESTER_BUFSIZE);
				checksum.Update(buffer, IOTESTER_BUFSIZE);
				}
			boost::scoped_array<unsigned char> digest_buffer(new unsigned char[checksum.DigestSize()]);
			checksum.Final(digest_buffer.get());
			std::string digest = digest_to_hex(digest_buffer, checksum.DigestSize());

			if(digest == filename)
				{
				todelete.push_back(i->path());
				}
			else
				{
				boost::lock_guard<boost::mutex> biglock(bigmutex);
				std::cout << "Digest wrong for file " << i->path() << std::endl;
				++wrong_digests;
				}

				{
				boost::lock_guard<boost::mutex> biglock(bigmutex);
				++count;
				if(count % 100 == 0)
					{
					boost::posix_time::time_duration read_time = boost::posix_time::microsec_clock::universal_time() - hundred_started;
					std::cout << "Read 100 MiB of data in " << read_time.total_milliseconds() << " milliseconds";
					std::cout << " (" << (100.0 / read_time.total_milliseconds()) * 1000 << " Megabytes/second)" << std::endl;
					hundred_started = boost::posix_time::microsec_clock::universal_time();
					if(read_time > max_read_time)
						max_read_time = read_time;
					}
				}
			}
		}
	for(std::vector<fs::path>::iterator i = todelete.begin() ; i != todelete.end() ; ++i)
		fs::remove(*i);
	}

void reader_thread(DirDispatcher* dispatcher)
	{
	fs::path curdir = dispatcher->dispatch_dir();

	while(!curdir.empty())
		{
		read_files(curdir);
		curdir = dispatcher->dispatch_dir();
		}
	}

int main(int argc, char **argv)
	{
	std::string datadir = "iotest";
	uint32_t num_threads = 100;
	if(argc == 2 || argc == 3 || argc == 4)
		datadir = argv[1];
	if(argc == 3 || argc == 4)
		{
		std::istringstream thr_str(argv[2]);
		thr_str >> num_threads;
		if(thr_str.fail())
			throw std::runtime_error(std::string("Could not convert ") + argv[2] + " to int");
		}

	if(argc == 4)
		{
		std::istringstream max_str(argv[3]);
		max_str >> max;
		if(max_str.fail())
			throw std::runtime_error(std::string("Could not convert ") + argv[3] + " to int");
		}

	if(argc > 4)
		{
		std::cerr << "Usage: iotester[ <datadir>[ <numthreads>[ <maxfiles>]]]" << std::endl;
		exit(-1);
		}

	srand(std::time(0));
	hundred_started = boost::posix_time::microsec_clock::universal_time();

	boost::thread_group write_threads;

	while(write_threads.size() < num_threads)
		{
		write_threads.create_thread(boost::bind(create_files, datadir));
		}
	write_threads.join_all();
	count = 0;
	std::cout << "Wrote all files, now checking integrity" << std::endl;

	DirDispatcher * dispatcher = new DirDispatcher(fs::path(datadir));

	boost::thread_group read_threads;
	while(read_threads.size() < num_threads)
		read_threads.create_thread(boost::bind(reader_thread, dispatcher));
	read_threads.join_all();

	std::cout << "Digests wrong (all loops):  " << wrong_digests << std::endl;
	return 0;
	}
