<cfcomponent displayname="facade" output="false" extends="CommonValues">
		
	<cfproperty name="beanService" displayname="private" required="true"
		default=""
		hint="service object for generating the beans code" />
		
	<cfproperty name="daoService" displayname="private" required="true"
		default=""
		hint="service object for generating the daos code" />
		
	<cfproperty name="testCaseService" displayname="private" required="true"
		default=""
		hint="service object for generating the test cases code" />
		
	<cfproperty name="EINPUT" displayname="error code" required="true"
		type="string" default="EINPUT"
		hint="error code for invalid input" />
		
	<cfproperty name="EINPUTMESSAGE" displayname="private" required="true"
		type="string" default=""
		hint="basic message for the error" />
	
	<cfproperty name="EPARSED" displayname="error code" requires="true"
		type="string" default="EPARSED"
		hint="erro code for invalid parsed structure" />
	
	<cfproperty name="EPARSEDMESSAGE" displayname="private" required="true"
		type="string" default=""
		hint="An informative message about how the parsed structure is setup">

	<!-------------------------------------------------------------->

	<cffunction name="init" access="public" output="false"
		returntype="BeanCreatorFacade">
		<cfscript>
			super.init();
			
			EINPUT = 'EINPUT';
			EINPUTMESSAGE = 'the input string was improperly formatted. please see the explination section of the UI.';
			EPARSED = 'EPARSED';
			EPARSEDMESSAGE = 'the parsed struct must conatin
			<ul>
				<li>TITLE - bean name</li>
				<li>L_TITLE - bean name with first letter lower case</li>
				<li>VARLIST - query with fields NAME, TYPE, DEFAULT</li>
			</ul>';
			
			variables.beanService = newObject( 'BeanService' );
			variables.daoService = newObject( 'DaoService' );
			variables.testCaseService = newObject( 'TestCaseService' );
			
			return this;
		</cfscript>
	</cffunction>
	
	<!-------------------------------------------------------------->
	
	<!---------------------------------------------------------------
		createAll
	---------------------------------------------------------------->
	<cffunction name="createAll" access="public" output="false"
		returntype="BeanCreator"
		hint="Generates bean, dao, and test case.<br /><br />
		Requires:<br />
		BeanCreator object must have a valid input string (see explination from UI)
		<br /><br />
		Returns:<br />
		BeanCreator object with bean, dao, and test case code.<br /><br />
		Throws:<br />
		EINPUT - invalid input string">
		<cfargument name="bc" required="yes" type="BeanCreator" />
		<cfscript>
			var parsed = parseInput( arguments.bc.getInput() );
			var bean = 0;
			var dao = 0;
			var testCase = 0;
			
			// generate
			bean = createBean( parsed, arguments.bc.getBeanExtends() );
			dao = createDao( parsed, arguments.bc.getDaoExtends() );
			testCase = createTestCase( parsed, arguments.bc.getTestCaseExtends() );
			
			// set
			arguments.bc.setBean( bean );
			arguments.bc.setDao( dao );
			arguments.bc.setTestCase( testCase );
			
			return arguments.bc;
		</cfscript>
	</cffunction>
	
	<!---------------------------------------------------------------
		parseInput
	---------------------------------------------------------------->
	<cffunction name="parseInput" access="public" output="false" returntype="struct"
		hint="Converts an input string into a more usable structure.<br /><br />
		Requires:<br />
		A valid input string (see explination from UI)<br /><br />
		Returns:<br />
		Structure with keys:<br />
		<ol>
			<li>TITLE - the bean name</li>
			<li>L_TITLE - the bean name with the first letter in lowercase</li>
			<li>VARLIST - a query with fields NAME, TYPE, DEFAULT</li>
		</ol>
		Throws:<br />
		EINPUT - invalid input string">
		<cfargument name="input" type="string" required="yes" />
		<cfscript>
			var parsed = structNew();
			var i = 1;
			var inputLineCount = 0;
			var lineCnt = 0;
			var line = 0;
			var type = 0;
			
			// need input
			if( not len(arguments.input) )
				throw( EINPUT, EINPUTMESSAGE );
				
			// setup the varlist query
			parsed.varList = queryNew( 'name,type,default' );
			
			// convert the input string to the parsed structure
			// line by line
			inputLineCount = listlen( arguments.input, CR );
			for( ; i lte inputLineCount; i = i + 1 ) {
				line = listGetAt( arguments.input, i, CR );
				
				if( i eq 1 ) {
					// parsing the name of the bean
					parsed.title = uCaseFirstLetter( listFirst( line, SP ) );
					parsed.l_title = lCaseFirstLetter( parsed.title );
					
				} else {
					// setup the variables list
					
					// get a term count
					lineCnt = listlen( line, SP );
					
					// move to the next line if this one is empty
					if( not lineCnt ) continue;
					
					// add a new row to query result set
					queryAddRow( parsed.varList );
					querySetCell( parsed.varList, 'name', lCaseFirstLetter( listfirst( line, SP ) ) );
					
					// setup variable information based on the information given
					switch( lineCnt ) {
						case 1:
							querySetCell( parsed.varList, 'type', 'string' );
							querySetCell( parsed.varList, 'default', '' );
							break;
						case 2:
							type = listlast( line, SP );
							querySetCell( parsed.varList, 'type', type );
							if( structKeyExists( defaults, type ) )
								querySetCell( parsed.varList, 'default', '#defaults[type]#' );
							else
								querySetCell( parsed.varList, 'default', "##createObject('component','#type#').init()##" );
							break;
						case 3:
							querySetCell( parsed.varList, 'type', listGetAt( line, 2, SP ) );
							querySetCell( parsed.varList, 'default', listlast( line, SP ) );
							break;
					}
				}
			}
			
			// should be setup
			return parsed;
		</cfscript>
	</cffunction>
	
	<!---------------------------------------------------------------
		createBean
	---------------------------------------------------------------->
	<cffunction name="createBean" access="public" output="false" returntype="string"
		hint="Creates bean code.<br /><br />
		Requires:<br /><br />
		Returns:<br />
		bean code<br /><br />
		Throws:<br />
		EPARSED - if the parsed structure doesnt have all of the necessary values">
		<cfargument name="parsed" type="struct" required="yes" />
		<cfargument name="extends" type="string" default="" />
		<cfset assertParsed( arguments.parsed ) />
		<cfreturn variables.beanService.create( argumentcollection = arguments ) />
	</cffunction>
	
	<!---------------------------------------------------------------
		createDao
	---------------------------------------------------------------->
	<cffunction name="createDao" access="public" output="false" returntype="string">
		hint="Creates dao code.<br /><br />
		Requires:<br /><br />
		Returns:<br />
		dao code<br /><br />
		Throws:<br />
		EPARSED - if the parsed structure doesnt have all of the necessary values">
		<cfargument name="parsed" type="struct" required="yes" />
		<cfargument name="extends" type="string" default="" />
		<cfset assertParsed( arguments.parsed ) />
		<cfreturn variables.daoService.create( argumentcollection = arguments ) />
	</cffunction>
	
	<!---------------------------------------------------------------
		createTestCase
	---------------------------------------------------------------->
	<cffunction name="createTestCase" access="public" output="false" returntype="string"
		hint="Creates test case code.<br /><br />
		Requires:<br /><br />
		Returns:<br />
		test case code<br /><br />
		Throws:<br />
		EPARSED - if the parsed structure doesnt have all of the necessary values">
		<cfargument name="parsed" type="struct" required="yes" />
		<cfargument name="extends" type="string" default="" />
		<cfset assertParsed( arguments.parsed ) />
		<cfreturn variables.testCaseService.create( argumentcollection = arguments ) />
	</cffunction>

	<!-------------------------------------------------------------->
	
	<!---------------------------------------------------------------
		assertParsed
	---------------------------------------------------------------->
	<cffunction name="assertParsed" access="private" output="false" returntype="void"
		hint="Validates the parsed structure. If the structure does not conform
		to standards this throws an error.<br /><br />
		Requires:<br /><br />
		Returns:<br /><br />
		Throws:<br />
		EPARSED - if the parsed structure does not conform to standards">
		<cfargument name="parsed" type="struct" requried="yes" />
		<cfscript>
			var failure = '';
			
			if( not structKeyExists( arguments.parsed, 'title' ) or
				not isSimpleValue( arguments.parsed.title ) )
				failure = listAppend( failure, 'title missing/malformed ' );
			
			if( not structKeyExists( arguments.parsed, 'l_title' ) or
				not isSimpleValue( arguments.parsed.title ) )
				failure = listAppend( failure, 'l_title missing/malformed' );
			
			if( not structKeyExists( arguments.parsed, 'varlist' ) or
				not isQuery( arguments.parsed.varlist ) )
				failure = listAppend( failure, 'varlist missing/malformed ' );
			
			if( not len(failure) )
				if( listsort( arguments.parsed.varlist.columnList, 'text' ) neq
					'DEFAULT,NAME,TYPE' )
					failure = listAppend( failure, 'varlist columns malformed ' );
				
			if( len(failure) )
				throw( EPARSED, EPARSEDMESSAGE & failure );
		</cfscript>
	</cffunction>
	
</cfcomponent>
