/* DecoderUTF8.java -- Decoder for the UTF-8 character encoding.
   Copyright (C) 1998 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
 
GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

As a special exception, if you link this library with other files to
produce an executable, this library does not by itself cause the
resulting executable to be covered by the GNU General Public License.
This exception does not however invalidate any other reasons why the
executable file might be covered by the GNU General Public License. */


package gnu.java.nio.charset.decoder;

import java.io.InputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharsetDecoder;
import java.nio.BufferUnderflowException;
import java.nio.charset.CharacterCodingException;

/**
  * This class implements character decoding in the UCS Transformation
  * Format 8 (UTF-8) encoding scheme.  This is defined in RFC-2279.
  * We only handle the 1-3 byte encodings for characters in the Unicode
  * set.
  *
  * @version 0.0
  *
  * @author Aaron M. Renn (arenn@urbanophile.com)
  */
public class DecoderUTF8 extends CharsetDecoder
{
/*
static
{
    scheme_name = "UTF8";
    scheme_description = "UCS Transformation Format 8 (see RFC-2279)";
}
*/

  public DecoderUTF8(){
    super(1.0F, 2.0F);
  }

//public DecoderUTF8(){}
/*************************************************************************/

public void decodeStep(ByteBuffer in, CharBuffer out) 
  throws CharacterCodingException {
  int val;
  /*Note that this method of reading a single byte at a time is 
    inefficient and should probably be replaced Read a byte
    */

    int b = in.get();
    try {
      // Three byte encoding case.
	if ((b & 0xE0) == 0xE0) // 224
	  {
	    val = (b & 0x0F) << 12;
	    
	    b = in.get();
	    
	    if ((b & 0x80) != 0x80)
	      throw new CharacterCodingException("Bad input byte: " + b);
	    val |= (b & 0x3F) << 6; 
	    
	    b = in.get();
	    
	    if ((b & 0x80) != 0x80)
	      throw new CharacterCodingException("Bad input byte: " + b);
	    val |= (b & 0x3F);
	  }
      // Two byte encoding case
	else if ((b & 0xC0) == 0xC0) // 192
	  {
	    val = (b & 0x1F) << 6;
	    b = in.get();
	    
	    if ((b & 0x80) != 0x80)
	      throw new CharacterCodingException("Bad input byte: " + b);
	    val |= (b & 0x3F);
	  }
      // One byte encoding case
	else if (b < 0x80) // 128
	  {
	    val = b;
	  }
	else
	  {
	    throw new CharacterCodingException("Bad input encountered: " + b);
	  }
      
      out.put((char)val);

    } catch (BufferUnderflowException e){
      throw new CharacterCodingException("Pre-mature end of input"); 
    }
}


} // class DecoderUTF8

