/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included  with this distribution in *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.commons.simplestore;

import com.coyotegulch.jisp.BTreeIndex;
import com.coyotegulch.jisp.IndexedObjectDatabase;
import com.coyotegulch.jisp.KeyNotFound;
import com.coyotegulch.jisp.KeyObject;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Map;
import java.util.Set;

/**
 * This store is based on the Jisp library 
 * (http://www.coyotegulch.com/jisp/index.html). This store uses B-Tree indexes
 *  to access variable-length serialized data stored in files.
 *
 * @author Gerhard Froehlich <a href="mailto:g-froehlich@gmx.de">
 *      g-froehlich@gmx.de</a>
 * @version $Id: JispFilesystemStore.java,v 1.12 2002/01/21 22:02:36 froehlich Exp $
 */

public final class JispFilesystemStore
implements Store {

    /**
     *  The directory repository
     */
    protected File mDirectoryFile;
    protected volatile String mDirectoryPath;

    /**
     *  The database
     */
    private String mDatabaseName = "default.dat";
    private String mIndexName = "default.idx";;
    private int mOrder = 101;
    private IndexedObjectDatabase mDatabase;
    private BTreeIndex mIndex;
    
    public JispFilesystemStore(String Directory,
                               String DatabaseName, 
                               String IndexName, 
                               int Order) {
        this.mDatabaseName = DatabaseName;
        this.mIndexName = IndexName;
        this.mOrder = Order;
        try {
            this.setDirectory(Directory);
        } catch (IOException e) {
            e.printStackTrace(); 
        }
        this.initialize();
    }

   /**
     * This method returns the name of the data file. Default
     * is default.dat.
     *
     * @return name of the data file
     */
    public String getDatabaseName() {
        return this.mDatabaseName;
    }

    /**
     * This method returns the name of the index file. Default
     * is default.idx.
     *
     * @return name of the index file
     */
    public String getIndexfileName() {
        return this.mIndexName;
    }

    public int getOrder() {
        return this.mOrder;
    }

    /**
     * Sets the repository's location
     *
     * @param directory the new directory value
     * @exception IOException
     */
    public void setDirectory(final String directory)
        throws IOException {
        this.setDirectory(new File(directory));
    }

    /**
     * Sets the repository's location
     *
     * @param directory the new directory value
     * @exception IOException
     */
    public void setDirectory(final File directory)
        throws IOException {
        this.mDirectoryFile = directory;

        this.mDirectoryPath =  this.mDirectoryFile.getCanonicalPath();
        this.mDirectoryPath += File.separator;

        if (!this.mDirectoryFile.exists()) {
            /* Create it new */
            if (!this.mDirectoryFile.mkdir()) {
                throw new IOException(
                        "Error creating store directory '" 
                        + this.mDirectoryPath + "': ");
            }
        }

        /* Is given file actually a directory? */
        if (!this.mDirectoryFile.isDirectory()) {
            throw new IOException("'" + this.mDirectoryPath 
                                  + "' is not a directory");
        }

        /* Is directory readable and writable? */
        if (!(this.mDirectoryFile.canRead()
              && this.mDirectoryFile.canWrite())) {
            throw new IOException(
                    "Directory '" + this.mDirectoryPath 
                    + "' is not readable/writable"
                    );
        }
    }


    /**
     * Returns the repository's full pathname
     *
     * @return the directory as String
     */
    public String getDirectoryPath() {
        return this.mDirectoryPath;
    }

    /**
     *  Returns a Object from the store associated with the Key Object
     *
     * @param key the Key object
     * @return the Object associated with Key Object
     */
    public Object get(Object key) {
        Object readObj = null;

        try {
            readObj = mDatabase.read(this.wrapKeyObject(key), mIndex);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return readObj;
    }

    /**
     *  Initialize the Component
     */
    public void initialize() {
        try {
            File myFile = new File(this.getDirectoryPath() + mDatabaseName);
            if (myFile.exists()) {
                mDatabase = new IndexedObjectDatabase(getDirectoryPath() 
                                + mDatabaseName, false);
                mIndex = new BTreeIndex(this.getDirectoryPath() + mIndexName);
                mDatabase.attachIndex(mIndex);
            } else {
                mDatabase = new IndexedObjectDatabase(getDirectoryPath() 
                            + mDatabaseName, false);
                mIndex = new BTreeIndex(this.getDirectoryPath() + mIndexName,
                        mOrder, new JispStringKey(), false);
                mDatabase.attachIndex(mIndex);
            }
        } catch (KeyNotFound ignore) {
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     *  Store the given Object in the indexed data file.
     *
     * @param key the Key Object
     * @param value the Value Object 
     * @exception IOException
     */
    public void put(Object key, Object value) {
        if (value instanceof Serializable) {
            try {
                KeyObject[] keyArray = new KeyObject[1];
                keyArray[0] = this.wrapKeyObject(key);
                mDatabase.write(keyArray, (Serializable) value);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            throw new IllegalArgumentException("Object not Serializable");
        }
    }

    /**
     * Frees some values of the data file. NOTE:
     * not implemented, yet.
     */
    public void free() { 
       throw new UnsupportedOperationException("method not implemented yet"); 
    }

    /**
     * Removes a value from the data file with the given key.
     *
     * @param key the Key Object
     */
    public Object remove(Object key) {
        try {
            KeyObject[] keyArray = new KeyObject[1];
            keyArray[0] = this.wrapKeyObject(key);
            mDatabase.remove(keyArray);
        } catch (KeyNotFound ignore) {
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Test if the the index file contains the given key
     *
     * @param - the Key Object
     * @return - true if Key exists and false if not
     */
    public boolean containsKey(Object key) {
        long res = -1;

        try {
            res = mIndex.findKey(this.wrapKeyObject(key));
        } catch (KeyNotFound ignore) {
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (res > 0) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * This method wraps around the key Object a Jisp KeyObject.
     * 
     * @param key the key Object
     * @return the wraped key Object
     */
    private KeyObject wrapKeyObject(Object key) {

        if(key.toString() instanceof String) {
            return new JispStringKey(key.toString());
        } else {
            //TODO: Implementation of Integer and Long keys
            return null;
        }
    }
    
    public boolean isEmpty()  {
        throw new UnsupportedOperationException("method not implemented yet");  
    }
    
    public int size()  {
        throw new UnsupportedOperationException("method not implemented yet");  
    }
    
    public void clear()  {
        throw new UnsupportedOperationException("method not implemented yet");  
    }

    public Store getNextStore() {
        throw new UnsupportedOperationException("method not implemented yet");  
    }
}



