/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included  with this distribution in *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.commons.simplestore;

import java.io.IOException;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Enumeration;
import java.util.Set;

/**
 *
 * @author Juozas Baliuka <a href="mailto:baliuka@mwm.lt">
 *      baliuka@mwm.lt</a>
 * @author Gerhard Froehlich <a href="mailto:g-froehlich@gmx.de">
 *      g-froehlich@gmx.de</a>
 * @version $Id: SoftRefMemoryStore.java,v 1.9 2002/01/27 17:41:48 froehlich Exp $
 */
public class SoftRefMemoryStore
implements Store {
    
    private static boolean DEBUG = true;
    private int m_maxStrongRefCount;
    private Object [] m_strongRefs;
    private int m_current = 0;
    private Map m_map;
    private ReferenceQueue m_queue = new ReferenceQueue();
    private Swap m_swap;
    
    static class SoftRef extends SoftReference {
        Object key;
        private SoftRef(Object key, Object object, ReferenceQueue queue) {
            super(object, queue);
            this.key = key;
        }
    }
    
    static class StrongRef extends Object {
        Reachable object;
        Object key;
        Swap queue;
        
        private  StrongRef(Object key, Reachable object, Swap queue) {
            this.queue = queue;// used in finalize
            this.object = object;// add strong reference to value
            this.key = key;
        }
        
        public Object get(){
            return object;
        }
        
        protected void finalize() throws Throwable {
            super.finalize();
            queue.add(key, object);
        }
    }
    
    private SoftRef makeValue(Object key, Object value, ReferenceQueue queue, Swap swap) {
        if(swap == null || value == null) {
            return new SoftRef(key, value, queue);
        } else {
            if (!(value instanceof Reachable)) {
                throw new java.lang.IllegalStateException("Value not Reachable in Swap ");
            }
            Reachable val    = (Reachable)value;
            StrongRef strong = new StrongRef(key, val, swap);
            SoftRef   ref    = new SoftRef(key, strong, queue);
            val.setReference(strong);
            return ref;
        }
    }
    
    public static Store getInstance(Map map, Swap swap, int maxStrongRef) {
        return new SynchronizedStore(new SoftRefMemoryStore(map, swap, maxStrongRef));
    }
    
    /** Creates new SoftRefMemoryStore */
    protected SoftRefMemoryStore(Map map, Swap swap, int maxStrongRefCount) {
        this.m_swap = swap;
        this.m_map = map;
        
        if(maxStrongRefCount < 0) {
            throw new java.lang.IllegalArgumentException();
        }
        this.m_maxStrongRefCount = maxStrongRefCount;
        
        if(maxStrongRefCount > 0) {
            m_strongRefs = new Object[maxStrongRefCount];
        }
    }
    
    // remove unused keys
    private void removeSoftRef(){
        SoftRef ref = (SoftRef)m_queue.poll();
        
        while(ref != null) {
            m_map.remove(ref.key);
            
            if(DEBUG) {
                System.out.println( "Key " + ref.key + " removed from Reference queue, map size is " + m_map.size());
            }
            ref = (SoftRef)m_queue.poll();
        }
    }
    
    private void addStrongRef(Object object) {
        if(m_strongRefs != null) {
            m_strongRefs[(m_current++) % m_maxStrongRefCount] = object;
        }
    }
    
    private void internalStoreObject(Object key, Object object) {
        SoftRef ref = makeValue(key, object, m_queue, m_swap);
        addStrongRef(ref.get());
        m_map.put(key,ref);
    }
    
    /**
     * Remove the object associated to the given key.
     *
     * @param key the Key Object
     */
    public Object remove(Object key) {
        removeSoftRef();
        return m_map.remove(key);
    }
    
    /**
     * Indicates if the given key is associated to a contained object.
     *
     * @param key the Key Object
     */
    public boolean containsKey(Object key) {
        removeSoftRef();
        return m_map.containsKey(key);
    }
    
    /**
     * Frees some object out of the Store.
     */
    public void free() {
        if( m_strongRefs != null ) {
            Arrays.fill(m_strongRefs,null);
        }
        removeSoftRef();
    }
    
    /**
     * Store the given object in a persistent state. It is up to the
     * caller to ensure that the key has a persistent state across
     * different JVM executions.
     *
     * @param key the Key Object
     * @param value the Value Object
     */
    public void put(Object key, Object object) {
        removeSoftRef();
        internalStoreObject(key,object);
    }
    
    /**
     * Get the object associated to the given unique key.
     *
     * @param key the Key Object
     */
    public Object get(Object key) {
        removeSoftRef();
        Object object = null;
        SoftRef ref = (SoftRef)m_map.get(key);
        
        if(ref != null) {
            Object value = ref.get();

            if(value != null && value instanceof StrongRef) {
                object = ((StrongRef)value).object;
            } else {
                object = value;
            }
        }
        addStrongRef(object);
        return object;
    }
    
    public boolean isEmpty()  {
        return m_map.isEmpty();
    }
    
    public int size()  {
        return m_map.size();
    }
    
    public void clear()  {
        m_map.clear();
    }
    
    public Store getNextStore() {
        throw new UnsupportedOperationException("method not implemented yet");
    }
}
