/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included  with this distribution in *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.commons.simplestore.cleanup;

import org.apache.commons.simplestore.Store;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * This class is a implentation of a StoreJanitor. Store classes
 * can register to the StoreJanitor. When memory is too low,
 * the StoreJanitor frees the registered caches until memory is normal.
 * <br>
 * NOTE:<br>
 * Be careful with the setFreeMemory and setHeapsize methods. Wrong values can
 * cause high cpu usage.<br>
 * Example configuration:<br>
 * Jvm settings:<br>
 *    -Xms100000000 -Xmx200000000<br>
 * StoreJanitor settings:<br>
 *   setFreeMemory(50000000)<br>
 *   setHeapsize(150000000)<br>
 *
 * Heapsize must be higher then the -Xms parameter and freememory
 * between those both.<br>    
 *
 * @author Gerhard Froehlich <a href="mailto:g-froehlich@gmx.de">
 *      g-froehlich@gmx.de</a>
 * @version $Id: StoreJanitorImpl.java,v 1.2 2002/01/20 12:24:50 froehlich Exp $
 */
public class StoreJanitorImpl 
implements StoreJanitor,
           Runnable {

    private int mFreeMemory = -1;
    private int mHeapSize = -1;
    private int mThreadInterval = -1;
    private int mThreadPriority = -1;
    private Runtime mJVM;
    private ArrayList mStoreList;
    private int mIndex = -1;
    private static boolean mDoRun = false;

    /**
     * Initialize the StoreJanitorImpl.
     *
     */
    public void initialize() {
        this.mJVM = Runtime.getRuntime();
        this.mStoreList = new ArrayList();
    }

    /**
     * This method starts the background Thread, which
     * checks periodic if memory is running low.
     */
    public void start() {
        mDoRun = true;
        Thread checker = new Thread(this);
        checker.setPriority(this.getThreadPriority());
        checker.setDaemon(true);
        checker.setName("checker");
        checker.start();
    }

    /**
     * This method stops the background Thread.
     */
    public void stop() {
        mDoRun = false;
    }

    /**
     * The "checker" thread checks if memory is running low in the jvm.
     */
    public void run() {
        while (mDoRun) {
            // amount of memory used is greater then heapsize
            if (this.memoryLow()) {
                this.freePhysicalMemory();
                synchronized (this) {
                    while (this.memoryLow() && this.mStoreList.size() > 0) {
                        this.freeMemory();
                    }
                    this.resetIndex();
                }
            }
            try {
                Thread.currentThread().sleep(this.mThreadInterval * 1000);
            } catch (InterruptedException ignore) {}
        }
    }

    /**
     * Method to check if memory is running low in the JVM.
     *
     * @return true if memory is low
     */
    private boolean memoryLow() {
        return this.mJVM.totalMemory() > this.getHeapsize() && this.mJVM.freeMemory() < this.getFreememory();
    }

    /**
     * This method register the stores
     *
     * @param store the store to be registered
     */
    public void register(Store store) {
        this.mStoreList.add(store);
    }

    /**
     * This method unregister the stores
     *
     * @param store the store to be unregistered
     */
    public void unregister(Store store) {
        this.mStoreList.remove(store);
    }

    /**
     * This method return a java.util.Iterator of every registered stores
     *
     * @return a java.util.Iterator
     */
    public Iterator iterator() {
        return this.mStoreList.iterator();
     }
     
    /**
     * Round Robin alghorithm for freeing the registerd caches.
     */
    private void freeMemory() {
        try {
            if (this.mIndex < this.mStoreList.size()) {
                if(this.mIndex == -1) {
                    ((Store)this.mStoreList.get(0)).free();
                    this.mIndex = 0;
                } else {
                    ((Store)this.mStoreList.get(this.mIndex)).free();
                    this.mIndex = this.mIndex + 1;
                }
            } else {
                this.resetIndex();
                ((Store)this.mStoreList.get(0)).free();
                this.mIndex = 0;
            }
            this.freePhysicalMemory();
        } catch(Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * This method forces the garbage collector
     */
    private void freePhysicalMemory() {
        this.mJVM.runFinalization();
        this.mJVM.gc();
    }

    /**
     * This method returns the current free memory
     * setting.
     *
     * @return current value of free memory in bytes
     */
    public int getFreememory() {
        return mFreeMemory;
    }

    /**
     * This method sets how much memory should be available
     * in the JVM.
     *
     * @param freememory free memory in bytes
     */
    public void setFreeMemory(int freememory) {
        this.mFreeMemory = freememory;
    }

    /**
     * This method returns current Heapsize setting.
     *
     * @return current value of the Heapsize in bytes
     */
    public int getHeapsize() {
        return this.mHeapSize;
    }

    /**
     * This method sets the maximum JVM consumptions, called
     * Heapsize.
     *
     * @param heapsize maximum JVM consumption in bytes.
     */
    public void setHeapsize(int heapsize) {
        this.mHeapSize = heapsize;
    }

    /**
     * This method return the current setting of the background
     * Thread interval.
     * 
     * @return current interval in seconds
     */
    public int getThreadInterval() {
        return this.mThreadInterval;
    }

    /**
     * This method sets the background Thread interval in
     * seconds.
     * 
     * @param threadinterval current interval in seconds
     */
    public void setThreadInterval(int threadinterval) {
        this.mThreadInterval = threadinterval;
    }

    /**
     * This method returns the priority of the background
     * Thread.
     * 
     * @return current Thread priority
     */
    public int getThreadPriority() {
        return this.mThreadPriority;
    }

    /**
     * This method sets the priority of the background
     * Thread.
     * 
     * @param priority current Thread priority
     */
    public void setThreadPriority(int priority) {
        this.mThreadPriority = priority;
    }

    private void resetIndex() {
        this.mIndex = -1;
    }
}
