/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included  with this distribution in *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.commons.simplestore.jdbc;

import java.sql.*;
import java.util.*;


/**
 *
 * @author Juozas Baliuka <a href="mailto:baliuka@mwm.lt">
 *      baliuka@mwm.lt</a>
 * @version $Id$
 *
 * This is light weight Read Only Datasource implementation, asumes Driver returns reentrant connections
 * it tested and designed for PostgresSQL JDBC driver only.
 *
 */
public class DriverDataSource implements javax.sql.DataSource {
    
    private final java.util.Properties properties = new java.util.Properties();
    
    private  ConnectionWrapper connection[] = null;
    
    private int counter = 0;
    
    private java.sql.Driver driver;
    
    /** Holds value of property user. */
    private String user;
    
    /** Holds value of property password. */
    private String password;
    
    /** Holds value of property url. */
    private String url;
    
    /** Holds value of property maxConnections. */
    private int maxConnections = 5;
    
    /** Creates new DriverDataSourceFactoryImpl */
    public DriverDataSource() {
        
    }
    
    
    /** Getter for property user.
     * @return Value of property user.
     */
    public String getUser() {
        return user;
    }
    
    /** Setter for property user.
     * @param user New value of property user.
     */
    public void setUser(String user) {
        if( this.user != null )
            throw new java.lang.IllegalStateException();
        this.user = user;
        properties.setProperty("user",user);
    }
    
    /** Getter for property password.
     * @return Value of property password.
     */
    public String getPassword() {
        return password;
    }
    
    /** Setter for property password.
     * @param password New value of property password.
     */
    public void setPassword(String password) {
        if( this.password != null )
            throw new java.lang.IllegalStateException();
        properties.setProperty("password",password);
        this.password = password;
    }
    
    /** Getter for property url.
     * @return Value of property url.
     */
    public String getUrl() {
        return url;
    }
    
    /** Setter for property url.
     * @param url New value of property url.
     */
    public void setUrl(String url) {
        if( this.url != null )
            throw new java.lang.IllegalStateException();
        this.url = url;
    }
    
    private ConnectionWrapper newConnection() throws java.sql.SQLException{
        java.sql.Connection con = driver.connect(url,properties);
        con.setAutoCommit(false);
        return new ConnectionWrapper(con);
    }
    
    public java.sql.Connection getConnection() throws java.sql.SQLException{
        
        synchronized(this){
            
            if( connection == null )
                connection = new ConnectionWrapper[maxConnections];
            
            
            counter  = (counter + 1)%maxConnections;
            
            for(int i = 0; i < maxConnections; i++ )
                if(connection[i] != null)
                    if(!connection[i].isUsed()){
                        counter = i;
                        break;
                    }
            
            if(connection[counter] == null)
                connection[counter] =  newConnection();
            
            if(connection[counter].isClosed()){
                try{
                    
                    connection[counter].release();
                    
                }catch(Exception ignore){
                    
                }
                connection[counter] =  newConnection();
            }
            
            connection[counter].setUsed(true);
        }
        return connection[counter];
    }
    
    public java.io.PrintWriter getLogWriter() throws java.sql.SQLException {
        return new java.io.PrintWriter(System.out);
    }
    
    public void setLogWriter(java.io.PrintWriter printWriter) throws java.sql.SQLException {
    }
    
    public void setLoginTimeout(int param) throws java.sql.SQLException {
    }
    
    public java.sql.Connection getConnection(java.lang.String str, java.lang.String str1) throws java.sql.SQLException {
        return getConnection();
    }
    
    public int getLoginTimeout() throws java.sql.SQLException {
        return 0;
    }
    
    /** Getter for property maxConnections.
     * @return Value of property maxConnections.
     */
    public int getMaxConnections() {
        return maxConnections;
    }
    
    /** Setter for property maxConnections.
     * @param maxConnections New value of property maxConnections.
     */
    public void setMaxConnections(int maxConnections) {
        if(connection != null)
            throw new java.lang.IllegalStateException();
        this.maxConnections = maxConnections;
        
    }
    
    
    /** Setter for property driver.
     * @param driver New value of property driver.
     */
    public void setDriver(String driver) {
        if( this.driver != null )
            throw new java.lang.IllegalStateException();
        
        try{
            this.driver = (java.sql.Driver)Class.forName(driver).newInstance();
        }catch(Exception e){
            e.printStackTrace();
            throw new RuntimeException(e.getMessage());
        }
    }
    
}

/**
 *
 * @author  Juozas Baliuka
 * @version $Id$
 */
class ConnectionWrapper  implements Connection {
    
    
    boolean used = true;
    private Connection connection;
    public void release()throws SQLException {
        connection.close();
    }
    
    public boolean isUsed(){
        return used;
    }
    
    public void setUsed(boolean used){
        this.used = used;
        
    }
    
    public ConnectionWrapper(Connection connection ) {
        this.connection = connection;
        
    }
    
    public void close() throws SQLException {
        setUsed(false);
    }
    
    
    public Statement createStatement() throws SQLException {
        return connection.createStatement();
        
    }
    
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        return connection.prepareStatement(sql);
        
    }
    
    public CallableStatement prepareCall(String sql) throws SQLException {
        return connection.prepareCall(sql);
    }
    
    public String nativeSQL(String sql) throws SQLException {
        return connection.nativeSQL(sql);
    }
    
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        connection.setAutoCommit(autoCommit);
    }
    
    public boolean getAutoCommit() throws SQLException {
        return connection.getAutoCommit();
    }
    
    public void commit() throws SQLException {
        connection.commit();
        
    }
    
    public void rollback() throws SQLException {
        
        connection.rollback();
    }
    
    
    public boolean isClosed() throws SQLException {
        if(connection == null){
            return true;
        }
        try{
            Statement stmt =  connection.createStatement();
            ResultSet rs   =   stmt.executeQuery("SELECT 1");
            rs.next();
            rs.getInt(1);
            rs.close();
            stmt.close();
            
        }catch(Exception sqle){
            try{
                connection.close();
            }catch(Exception ignore){}
            
            return true;
        }
        
        return connection.isClosed();
    }
    
    public DatabaseMetaData getMetaData() throws SQLException {
        return connection.getMetaData();
    }
    
    public void setReadOnly(boolean readOnly) throws SQLException {
        connection.setReadOnly(readOnly);
    }
    
    public boolean isReadOnly() throws SQLException {
        return connection.isReadOnly();
    }
    
    public void setCatalog(String catalog) throws SQLException {
        connection.setCatalog(catalog);
    }
    
    public String getCatalog() throws SQLException {
        return connection.getCatalog();
    }
    
    public void setTransactionIsolation(int level) throws SQLException {
        connection.setTransactionIsolation(level);
    }
    
    public int getTransactionIsolation() throws SQLException {
        return connection.getTransactionIsolation();
    }
    
    public SQLWarning getWarnings() throws SQLException {
        return connection.getWarnings();
    }
    
    public void clearWarnings() throws SQLException {
        connection.clearWarnings();
    }
    
    public Statement createStatement(int resultSetType, int resultSetConcurrency)
    throws SQLException {
        return connection.createStatement(resultSetType,resultSetConcurrency);
        
    }
    
    public PreparedStatement prepareStatement(String sql, int resultSetType,
    int resultSetConcurrency) throws SQLException {
        return connection.prepareStatement(sql, resultSetType, resultSetConcurrency);
    }
    
    public CallableStatement prepareCall(String sql, int resultSetType,
    int resultSetConcurrency) throws SQLException {
        return connection.prepareCall(sql, resultSetType, resultSetConcurrency);
    }
    
    public java.util.Map getTypeMap() throws java.sql.SQLException {
        return connection.getTypeMap();
    }
    
    public void setTypeMap(Map map) throws SQLException {
        connection.setTypeMap(map);
    }
    
    
    
    
}


