/*
 * $Header: /home/cvs/jakarta-commons/collections/src/java/org/apache/commons/collections/FastArrayList.java,v 1.4 2002/02/10 08:07:42 jstrachan Exp $
 * $Revision: 1.4 $
 * $Date: 2002/02/10 08:07:42 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */


package org.apache.commons.collections;


import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;


/**
 * <p>A customized implementation of <code>java.util.ArrayList</code> designed
 * to operate in a multithreaded environment where the large majority of
 * method calls are read-only, instead of structural changes.
 *
 *  <P>This class can operate in two modes, fast mode and slow mode.  In
 *  slow mode, all method invocations are synchronized.  This is appropriate
 *  for initially populating the collection.
 *
 *  <P>Previous releases of this class provided unsychronized read access
 *  in fast mode.  Write operations would clone an internal collection, update
 *  the clone, then reassign the internal field to the clone.  Read operations
 *  would simply obtain the most recent reference and use that.
 *
 *  <P>However, that technique does not work on some platforms, particularly 
 *  multiprocessor architectures.  Even on a single processor, however, an
 *  optimizing compiler or a pipeline could alter the order of operations, so
 *  that the reference would be reassigned to the clone before the clone was
 *  fully updated. 
 *
 *  <P>This release doesn't guarantee unsynchronized read access, but offers
 *  the next best thing:  The critical section of a read operation is the shortest
 *  imaginable.  A read operation is synchronized only long enough to fetch the
 *  reference to the internal collection.  (Although on some platforms, you may
 *  be able to safely avoid the synchronization on reads.  See {@link SafeReferenceFactory}
 *  for more information.)
 *
 *  <P>Write operations are still fully synchronized, and still clone the 
 *  internal collection.  Only one write can occur at a time; multiple reads can
 *  be in progress at the same time.
 *
 *  <P>Sublists produced by this class are also thread-safe, and also provide
 *  fast read access but slow write access.
 *
 *  <P>Note that the iterators produced by this class are <I>not</I> fail-fast:
 *  Read operations on an iterator <I>never</I> fail.  A {@link ConcurrentModificationException}
 *  will only be raised if the original list is modified after the iterator is 
 *  created, and the iterator then attempts to modify the list.
 *
 * <p><strong>NOTE</strong>: If you are creating and accessing an
 * <code>ArrayList</code> only within a single thread, you should use
 * <code>java.util.ArrayList</code> directly (with no synchronization), for
 * maximum performance.</p>
 *
 * @author Craig R. McClanahan, Paul Jack
 * @version $Revision: 1.4 $ $Date: 2002/02/10 08:07:42 $
 */

public class FastArrayList extends ArrayList {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a an empty list.
     */
    public FastArrayList() {
        this(new ArrayList());
    }


    /**
     * Construct an empty list with the specified capacity.
     *
     * @param capacity The initial capacity of the empty list
     */
    public FastArrayList(int capacity) {
        this(new ArrayList(capacity));
    }


    /**
     * Construct a list containing the elements of the specified collection,
     * in the order they are returned by the collection's iterator.
     *
     * @param collection The collection whose elements initialize the contents
     *  of this list
     */
    public FastArrayList(Collection collection) {
        this(new ArrayList(collection));
    }


    private FastArrayList(ArrayList list) {
        super(0); // minimize space taken up by unused superclass fields
        this.list = list;
        listRef = SafeReference.create(list);
    }

    // ----------------------------------------------------- Instance Variables


    /**
     * The underlying list we are managing.
     *
     * @deprecated  use the getList() and setList(ArrayList) methods instead.
     */
    transient private ArrayList list = null;


    /**
     *  Safe reference to the list.
     */
    transient private SafeReference listRef;


    // ------------------------------------------------------------- Properties


    /**
     * Are we operating in "fast" mode?
     */
    protected boolean fast = false;

    public boolean getFast() {
        return (this.fast);
    }

    public void setFast(boolean fast) {
        this.fast = fast;
    }


    protected ArrayList getList() {
        return (ArrayList)listRef.get();
    }


    protected void setList(ArrayList list) {
        listRef.set(list);
	this.list = list;
    }


    // --------------------------------------------------------- Public Methods


    /**
     * Appends the specified element to the end of this list.
     *
     * @param element The element to be appended
     */
    public boolean add(Object element) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.add(element);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.add(element));
            }
        }

    }


    /**
     * Insert the specified element at the specified position in this list,
     * and shift all remaining elements up one position.
     *
     * @param index Index at which to insert this element
     * @param element The element to be inserted
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public void add(int index, Object element) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                temp.add(index, element);
                setList(temp);
            }
        } else {
            synchronized (list) {
                list.add(index, element);
            }
        }

    }


    /**
     * Append all of the elements in the specified Collection to the end
     * of this list, in the order that they are returned by the specified
     * Collection's Iterator.
     *
     * @param collection The collection to be appended
     */
    public boolean addAll(Collection collection) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.addAll(collection);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.addAll(collection));
            }
        }

    }


    /**
     * Insert all of the elements in the specified Collection at the specified
     * position in this list, and shift any previous elements upwards as
     * needed.
     *
     * @param index Index at which insertion takes place
     * @param collection The collection to be added
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public boolean addAll(int index, Collection collection) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.addAll(index, collection);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.addAll(index, collection));
            }
        }

    }


    /**
     * Remove all of the elements from this list.  The list will be empty
     * after this call returns.
     *
     * @exception UnsupportedOperationException if <code>clear()</code>
     *  is not supported by this list
     */
    public void clear() {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                temp.clear();
                setList(temp);
            }
        } else {
            synchronized (list) {
                list.clear();
            }
        }

    }


    /**
     * Return a shallow copy of this <code>FastArrayList</code> instance.
     * The elements themselves are not copied.
     */
    public Object clone() {

        FastArrayList results = null;
        if (fast) {
            results = new FastArrayList(list);
        } else {
            synchronized (list) {
                results = new FastArrayList(list);
            }
        }
        results.setFast(getFast());
        return (results);

    }


    /**
     * Return <code>true</code> if this list contains the specified element.
     *
     * @param element The element to test for
     */
    public boolean contains(Object element) {

        if (fast) {
            return (getList().contains(element));
        } else {
            synchronized (list) {
                return (list.contains(element));
            }
        }

    }


    /**
     * Return <code>true</code> if this list contains all of the elements
     * in the specified Collection.
     *
     * @param collection Collection whose elements are to be checked
     */
    public boolean containsAll(Collection collection) {

        if (fast) {
            return (getList().containsAll(collection));
        } else {
            synchronized (list) {
                return (list.containsAll(collection));
            }
        }

    }


    /**
     * Increase the capacity of this <code>ArrayList</code> instance, if
     * necessary, to ensure that it can hold at least the number of elements
     * specified by the minimum capacity argument.
     *
     * @param capacity The new minimum capacity
     */
    public void ensureCapacity(int capacity) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                temp.ensureCapacity(capacity);
                setList(temp);
            }
        } else {
            synchronized (list) {
                list.ensureCapacity(capacity);
            }
        }

    }


    /**
     * Compare the specified object with this list for equality.  This
     * implementation uses exactly the code that is used to define the
     * list equals function in the documentation for the
     * <code>List.equals</code> method.
     *
     * @param o Object to be compared to this list
     */
    public boolean equals(Object o) {

        // Simple tests that require no synchronization
        if (o == this)
            return (true);
        else if (!(o instanceof List))
            return (false);
        List lo = (List) o;

        // Compare the sets of elements for equality
        if (fast) {
            ListIterator li1 = getList().listIterator();
            ListIterator li2 = lo.listIterator();
            while (li1.hasNext() && li2.hasNext()) {
                Object o1 = li1.next();
                Object o2 = li2.next();
                if (!(o1 == null ? o2 == null : o1.equals(o2)))
                    return (false);
            }
            return (!(li1.hasNext() || li2.hasNext()));
        } else {
            synchronized (list) {
                ListIterator li1 = list.listIterator();
                ListIterator li2 = lo.listIterator();
                while (li1.hasNext() && li2.hasNext()) {
                    Object o1 = li1.next();
                    Object o2 = li2.next();
                    if (!(o1 == null ? o2 == null : o1.equals(o2)))
                        return (false);
                }
                return (!(li1.hasNext() || li2.hasNext()));
            }
        }

    }


    /**
     * Return the element at the specified position in the list.
     *
     * @param index The index of the element to return
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public Object get(int index) {

        if (fast) {
            return (getList().get(index));
        } else {
            synchronized (list) {
                return (list.get(index));
            }
        }

    }


    /**
     * Return the hash code value for this list.  This implementation uses
     * exactly the code that is used to define the list hash function in the
     * documentation for the <code>List.hashCode</code> method.
     */
    public int hashCode() {

        if (fast) {
            int hashCode = 1;
            java.util.Iterator i = getList().iterator();
            while (i.hasNext()) {
                Object o = i.next();
                hashCode = 31 * hashCode + (o == null ? 0 : o.hashCode());
            }
            return (hashCode);
        } else {
            synchronized (list) {
                int hashCode = 1;
                java.util.Iterator i = list.iterator();
                while (i.hasNext()) {
                    Object o = i.next();
                    hashCode = 31 * hashCode + (o == null ? 0 : o.hashCode());
                }
                return (hashCode);
            }
        }

    }


    /**
     * Search for the first occurrence of the given argument, testing
     * for equality using the <code>equals()</code> method, and return
     * the corresponding index, or -1 if the object is not found.
     *
     * @param element The element to search for
     */
    public int indexOf(Object element) {

        if (fast) {
            return (getList().indexOf(element));
        } else {
            synchronized (list) {
                return (list.indexOf(element));
            }
        }

    }


    /**
     * Test if this list has no elements.
     */
    public boolean isEmpty() {

        if (fast) {
            return (getList().isEmpty());
        } else {
            synchronized (list) {
                return (list.isEmpty());
            }
        }

    }


    /**
     * Return an iterator over the elements in this list in proper sequence.
     * <br><br>
     * <strong>IMPLEMENTATION NOTE</strong> - If the list is operating in fast
     * mode, an Iterator is returned, and a structural modification to the
     * list is made, then the Iterator will continue over the previous contents
     * of the list (at the time that the Iterator was created), rather than
     * failing due to concurrent modifications.
     */
    public Iterator iterator() {

        if (fast) {
            return new FastArrayListIterator(this, getList(), 0);
        } else {
            synchronized (list) {
                return (list.iterator());
            }
        }

    }


    /**
     * Search for the last occurrence of the given argument, testing
     * for equality using the <code>equals()</code> method, and return
     * the corresponding index, or -1 if the object is not found.
     *
     * @param element The element to search for
     */
    public int lastIndexOf(Object element) {

        if (fast) {
            return (getList().lastIndexOf(element));
        } else {
            synchronized (list) {
                return (list.lastIndexOf(element));
            }
        }

    }


    /**
     * Return an iterator of the elements of this list, in proper sequence.
     * See the implementation note on <code>iterator()</code>.
     */
    public ListIterator listIterator() {

        if (fast) {
            return new FastArrayListIterator(this, getList(), 0);
        } else {
            synchronized (list) {
                return (list.listIterator());
            }
        }

    }


    /**
     * Return an iterator of the elements of this list, in proper sequence,
     * starting at the specified position.
     * See the implementation note on <code>iterator()</code>.
     *
     * @param index The starting position of the iterator to return
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public ListIterator listIterator(int index) {

        if (fast) {
	    ArrayList list = getList();
            if ((index < 0) || (index > list.size()))
            {
                throw new IndexOutOfBoundsException();
            }
            return new FastArrayListIterator(this, list, index);
        } else {
            synchronized (list) {
                return (list.listIterator(index));
            }
        }

    }


    /**
     * Remove the element at the specified position in the list, and shift
     * any subsequent elements down one position.
     *
     * @param index Index of the element to be removed
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public Object remove(int index) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                Object result = temp.remove(index);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.remove(index));
            }
        }

    }


    /**
     * Remove the first occurrence of the specified element from the list,
     * and shift any subsequent elements down one position.
     *
     * @param element Element to be removed
     */
    public boolean remove(Object element) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.remove(element);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.remove(element));
            }
        }

    }


    /**
     * Remove from this collection all of its elements that are contained
     * in the specified collection.
     *
     * @param collection Collection containing elements to be removed
     *
     * @exception UnsupportedOperationException if this optional operation
     *  is not supported by this list
     */
    public boolean removeAll(Collection collection) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.removeAll(collection);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.removeAll(collection));
            }
        }

    }


    /**
     * Remove from this collection all of its elements except those that are
     * contained in the specified collection.
     *
     * @param collection Collection containing elements to be retained
     *
     * @exception UnsupportedOperationException if this optional operation
     *  is not supported by this list
     */
    public boolean retainAll(Collection collection) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                boolean result = temp.retainAll(collection);
                setList(temp);
                return (result);
            }
        } else {
            synchronized (list) {
                return (list.retainAll(collection));
            }
        }

    }


    /**
     * Replace the element at the specified position in this list with
     * the specified element.  Returns the previous object at that position.
     * <br><br>
     * <strong>IMPLEMENTATION NOTE</strong> - This operation is specifically
     * documented to not be a structural change, so it is safe to be performed
     * without cloning.
     *
     * @param index Index of the element to replace
     * @param element The new element to be stored
     *
     * @exception IndexOutOfBoundsException if the index is out of range
     */
    public Object set(int index, Object element) {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList)getList().clone();
                Object result = temp.set(index, element);
                setList(temp);
                return result;
            }
        } else {
            synchronized (list) {
                return (list.set(index, element));
            }
        }

    }


    /**
     * Return the number of elements in this list.
     */
    public int size() {

        if (fast) {
            return (getList().size());
        } else {
            synchronized (list) {
                return (list.size());
            }
        }

    }


    /**
     * Return a view of the portion of this list between fromIndex
     * (inclusive) and toIndex (exclusive).  The returned list is backed
     * by this list, so non-structural changes in the returned list are
     * reflected in this list.  The returned list supports
     * all of the optional list operations supported by this list.
     *
     * @param fromIndex The starting index of the sublist view
     * @param toIndex The index after the end of the sublist view
     *
     * @exception IndexOutOfBoundsException if an index is out of range
     */
    public List subList(int fromIndex, int toIndex) {

        if (fast) {
            ArrayList current = getList();
            if (toIndex < fromIndex) {
                throw new IllegalArgumentException("subList(" + fromIndex + ", " + toIndex + "): Invalid range.");
            }
            if ((fromIndex < 0) || (fromIndex > current.size()) || (toIndex < 0) || (toIndex > current.size())) {
                throw new IndexOutOfBoundsException();
            }
            return new SubFastArrayList(this, current, fromIndex, toIndex);
        } else {
            synchronized (list) {
                return (list.subList(fromIndex, toIndex));
            }
        }

    }


    /**
     * Return an array containing all of the elements in this list in the
     * correct order.
     */
    public Object[] toArray() {

        if (fast) {
            return (getList().toArray());
        } else {
            synchronized (list) {
                return (list.toArray());
            }
        }

    }


    /**
     * Return an array containing all of the elements in this list in the
     * correct order.  The runtime type of the returned array is that of
     * the specified array.  If the list fits in the specified array, it is
     * returned therein.  Otherwise, a new array is allocated with the
     * runtime type of the specified array, and the size of this list.
     *
     * @param array Array defining the element type of the returned list
     *
     * @exception ArrayStoreException if the runtime type of <code>array</code>
     *  is not a supertype of the runtime type of every element in this list
     */
    public Object[] toArray(Object array[]) {

        if (fast) {
            return (getList().toArray(array));
        } else {
            synchronized (list) {
                return (list.toArray(array));
            }
        }

    }


    /**
     * Return a String representation of this object.
     */
    public String toString() {

        StringBuffer sb = new StringBuffer("FastArrayList[");
        sb.append(getList().toString());
        sb.append("]");
        return (sb.toString());

    }


    /**
     * Trim the capacity of this <code>ArrayList</code> instance to be the
     * list's current size.  An application can use this operation to minimize
     * the storage of an <code>ArrayList</code> instance.
     */
    public void trimToSize() {

        if (fast) {
            synchronized (this) {
                ArrayList temp = (ArrayList) getList().clone();
                temp.trimToSize();
                setList(temp);
            }
        } else {
            synchronized (list) {
                list.trimToSize();
            }
        }

    }


    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        ArrayList temp = (ArrayList)getList();
        out.writeObject(temp);
    }


    private void readObject(ObjectInputStream inp) throws IOException, ClassNotFoundException {
        inp.defaultReadObject();
	ArrayList list = (ArrayList)inp.readObject();
        this.listRef = SafeReference.create(list);
	this.list = list;
    }

}


class SubFastArrayList implements List
{

	static class State
	{
		ArrayList expected;
		int first;
		int last;

		State(ArrayList expected, int first, int last)
		{
			this.expected = expected;
			this.first = first;
			this.last = last;
		}


		public List subList()
		{
			return expected.subList(first, last);
		}


		public int checkBounds(int index)
		{
			index += first;
			if ((index < first) || (index >= last))
			{
				throw new IndexOutOfBoundsException(); // FIXME
			}
			return index;
		}


		public int checkBounds2(int index)
		{
			index += first;
			if ((index < first) || (index > last))
			{
				throw new IndexOutOfBoundsException(); // FIXME
			}
			return index;
		}


	}


	FastArrayList fast;
	SafeReference stateRef;


	public SubFastArrayList(FastArrayList fast, ArrayList expected, int first, int last)
	{
		this.fast = fast;
		State state = new State(expected, first, last);
		stateRef = SafeReference.create(state);
	}


	State getState()
	{
		return (State)stateRef.get();
	}


	private State checkMod()
	{
		State state = (State)stateRef.get();
		if (state.expected != fast.getList())
		{
			throw new ConcurrentModificationException();
		}
		return state;
	}


	private void update(State state, int delta)
	{
		stateRef.set(new State(fast.getList(), state.first, state.last + delta));
	}


	private void updateFirst(State state, int delta)
	{
		stateRef.set(new State(fast.getList(), state.first, state.first + delta));
	}


	public int size()
	{
		State state = (State)stateRef.get();
		return state.last - state.first;
	}


	public boolean isEmpty()
	{
		State state = (State)stateRef.get();
		return state.last == state.first;
	}


	public boolean contains(Object value)
	{
		return indexOf(value) >= 0;
	}


	public Iterator iterator()
	{
		State state = (State)stateRef.get();
		return listIterator();
	}


	public Object[] toArray()
	{
		State state = (State)stateRef.get();
		return state.subList().toArray();
	}


	public Object[] toArray(Object[] target)
	{
		State state = (State)stateRef.get();
		return state.subList().toArray(target);
	}


	public boolean add(Object value)
	{
		synchronized (fast)
		{
			State state = checkMod();
			fast.add(state.last, value);
			update(state, 1);
			return true;
		}
	}


	public boolean remove(Object value)
	{
		synchronized (fast)
		{
			State state = checkMod();
			int i = indexOf(value);
			if (i < 0) return false;
			fast.remove(state.first + i);
			update(state, -1);
			return true;
		}
	}


	public boolean containsAll(Collection c)
	{
		State state = (State)stateRef.get();
		return state.subList().containsAll(c);
	}


	public boolean addAll(Collection c)
	{
		synchronized (fast)
		{
			State state = checkMod();
			fast.addAll(state.last, c);
			update(state, c.size());
			return true;
		}
	}


	public boolean addAll(int index, Collection c)
	{
		synchronized (fast)
		{
			State state = checkMod();
			fast.addAll(state.first + index, c);
			update(state, c.size());
			return true;
		}
	}


	public boolean removeAll(Collection c)
	{
		synchronized (fast)
		{
			State state = checkMod();
			ArrayList temp = (ArrayList)state.expected.clone();
			List sub = temp.subList(state.first, state.last);
			boolean result = sub.removeAll(c);
			fast.setList(temp);
			updateFirst(state, sub.size());
			return result;
		}
	}


	public boolean retainAll(Collection c)
	{
		synchronized (fast)
		{
			State state = checkMod();
			ArrayList temp = (ArrayList)state.expected.clone();
			List sub = temp.subList(state.first, state.last);
			boolean result = sub.retainAll(c);
			fast.setList(temp);
			updateFirst(state, sub.size());
			return result;
		}
	}


	public void clear()
	{
		synchronized (fast)
		{
			State state = checkMod();
			ArrayList temp = (ArrayList)state.expected.clone();
			List sub = temp.subList(state.first, state.last);
			sub.clear();
			fast.setList(temp);
			updateFirst(state, sub.size());
		}
	}


	public boolean equals(Object o)
	{
		State state = (State)stateRef.get();
		return state.subList().equals(o);
	}


	public int hashCode()
	{
		State state = (State)stateRef.get();
		return state.subList().hashCode();
	}


	public Object get(int index)
	{
		State state = (State)stateRef.get();
		return state.expected.get(index + state.first);
	}


	public Object set(int index, Object value)
	{
		synchronized (fast)
		{
			State state = checkMod();
			index = state.checkBounds(index);
			return state.expected.set(index, value);
		}
	}


	public void add(int index, Object value)
	{
		synchronized (fast)
		{
			State state = checkMod();
			index = state.checkBounds2(index);
			fast.add(index, value);
			update(state, 1);
		}
	}


	public Object remove(int index)
	{
		synchronized (fast)
		{
			State state = checkMod();
			index = state.checkBounds(index);
			Object result = fast.remove(index);
			update(state, -1);
			return result;
		}
	}


	public int indexOf(Object value)
	{
		State state = (State)stateRef.get();
		return state.subList().indexOf(value);
	}


	public int lastIndexOf(Object value)
	{
		State state = (State)stateRef.get();
		return state.subList().lastIndexOf(value);
	}


	public ListIterator listIterator()
	{
		return listIterator(0);
	}


	public ListIterator listIterator(int index)
	{
		State state = (State)stateRef.get();
		state.checkBounds2(index);
		return new SubFastArrayListIterator(this, state, index);
	}


	public List subList(int first, int last)
	{
		if (last < first)
		{
			throw new IllegalArgumentException("subList(" + first + ", " + last + ": Invalid range.");
		}
		State state = (State)stateRef.get();
		first = state.checkBounds2(first);
		last = state.checkBounds2(last);
		return new SubFastArrayList(fast, state.expected, first, last);
	}


	public String toString()
	{
		State state = (State)stateRef.get();
		return state.subList().toString();
	}



}


class SubFastArrayListIterator implements ListIterator
{

	SubFastArrayList sub;
	SubFastArrayList.State expected;
	int cursor;
	int lastReturnedIndex = -1;


	SubFastArrayListIterator(SubFastArrayList sub, SubFastArrayList.State expected, int cursor)
	{
		this.sub = sub;
		this.expected = expected;
		this.cursor = cursor;
	}


	private void checkMod()
	{
		checkNull();
		if ((expected.expected != sub.fast.getList()) || (sub.stateRef.get() != expected))
		{
			expected = null;
			throw new ConcurrentModificationException();
		}
		if (lastReturnedIndex == -1)
		{
			throw new IllegalStateException();
		}
	}

	private void checkNull()
	{
		if (expected == null)
		{
			throw new ConcurrentModificationException();
		}
	}


	public boolean hasNext()
	{
		checkNull();
		return cursor != expected.last - expected.first;
	}


	public Object next()
	{
		checkNull();
		Object r = expected.expected.get(cursor + expected.first);
		lastReturnedIndex = cursor;
		cursor++;
		return r;
	}


	public boolean hasPrevious()
	{
		checkNull();
		return cursor != 0;
	}


	public Object previous()
	{
		checkNull();
		cursor--;
		Object r = expected.expected.get(cursor + expected.first);
		lastReturnedIndex = cursor;
		return r;
	}


	public int nextIndex()
	{
		checkNull();
		return cursor;
	}


	public int previousIndex()
	{
		checkNull();
		return cursor - 1;
	}


	public void remove()
	{
		synchronized (sub.fast)
		{
			checkMod();
			sub.remove(lastReturnedIndex);
			if (lastReturnedIndex < cursor)
			{
				cursor--;
			}
			lastReturnedIndex = -1;
			expected = sub.getState();
		}
	}


	public void set(Object v)
	{
		synchronized (sub.fast)
		{
			checkMod();
			sub.set(lastReturnedIndex, v);
			lastReturnedIndex = -1;
			expected = sub.getState();
		}
	}


	public void add(Object v)
	{
		synchronized (sub.fast)
		{
			checkMod();
			sub.add(cursor, v);
			cursor++;
			lastReturnedIndex = -1;
			expected = sub.getState();
		}
	}


}


class FastArrayListIterator implements ListIterator
{

	FastArrayList fast;
	ArrayList expected;
	int cursor;
	int lastReturnedIndex = -1;


	FastArrayListIterator(FastArrayList fast, ArrayList expected, int cursor)
	{
		this.fast = fast;
		this.expected = expected;
		this.cursor = cursor;
	}


	private void checkMod()
	{
		checkNull();
		if (expected != fast.getList())
		{
			expected = null;
			throw new ConcurrentModificationException();
		}
		if (lastReturnedIndex == -1)
		{
			throw new IllegalStateException();
		}
	}

	private void checkNull()
	{
		if (expected == null)
		{
			throw new ConcurrentModificationException();
		}
	}


	public boolean hasNext()
	{
		checkNull();
		return cursor != expected.size();
	}


	public Object next()
	{
		checkNull();
		Object r = expected.get(cursor);
		lastReturnedIndex = cursor;
		cursor++;
		return r;
	}


	public boolean hasPrevious()
	{
		checkNull();
		return cursor != 0;
	}


	public Object previous()
	{
		checkNull();
		cursor--;
		Object r = expected.get(cursor);
		lastReturnedIndex = cursor;
		return r;
	}


	public int nextIndex()
	{
		return cursor;
	}


	public int previousIndex()
	{
		return cursor - 1;
	}


	public void remove()
	{
		synchronized (fast)
		{
			checkMod();
			fast.remove(lastReturnedIndex);
			if (lastReturnedIndex < cursor)
			{
				cursor--;
			}
			lastReturnedIndex = -1;
			expected = fast.getList();
		}
	}


	public void set(Object v)
	{
		synchronized (fast)
		{
			checkMod();
			fast.set(lastReturnedIndex, v);
			lastReturnedIndex = -1;
			expected = fast.getList();
		}
	}


	public void add(Object v)
	{
		synchronized (fast)
		{
			checkMod();
			fast.add(cursor, v);
			cursor++;
			lastReturnedIndex = -1;
			expected = fast.getList();
		}
	}


}
