/*
 * $Header: /home/cvs/jakarta-commons/collections/src/java/org/apache/commons/collections/FastHashMap.java,v 1.5 2002/02/10 08:07:42 jstrachan Exp $
 * $Revision: 1.5 $
 * $Date: 2002/02/10 08:07:42 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */


package org.apache.commons.collections;


import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;


/**
 * <p>A customized implementation of <code>java.util.ArrayList</code> designed
 * to operate in a multithreaded environment where the large majority of
 * method calls are read-only, instead of structural changes.
 *
 *  <P>This class can operate in two modes, fast mode and slow mode.  In
 *  slow mode, all method invocations are synchronized.  This is appropriate
 *  for initially populating the collection.
 *
 *  <P>Previous releases of this class provided unsychronized read access
 *  in fast mode.  Write operations would clone an internal map, update
 *  the clone, then reassign the internal field to the clone.  Read operations
 *  would simply obtain the most recent reference and use that.
 *
 *  <P>However, that technique does not work on some platforms, particularly 
 *  multiprocessor architectures.  Even on a single processor, however, an
 *  optimizing compiler or a pipeline could alter the order of operations, so
 *  that the reference would be reassigned to the clone before the clone was
 *  fully updated. 
 *
 *  <P>This release doesn't guarantee unsynchronized read access, but offers
 *  the next best thing:  The critical section of a read operation is the shortest
 *  imaginable.  A read operation is synchronized only long enough to fetch the
 *  reference to the internal map.  (Although on some platforms, you may
 *  be able to safely avoid the synchronization on reads.  See {@link SafeReferenceFactory}
 *  for more information.)
 *
 *  <P>Write operations are still fully synchronized, and still clone the 
 *  internal collection.  Only one write can occur at a time; multiple reads can
 *  be in progress at the same time.
 *
 *  <P>The collection views produced by this class are also thread-safe, and also provide
 *  fast read access but slow write access.
 *
 *  <P>Note that the iterators produced by this class are <I>not</I> fail-fast:
 *  Read operations on an iterator <I>never</I> fail.  A {@link ConcurrentModificationException}
 *  will only be raised if the original list is modified after the iterator is 
 *  created, and the iterator then attempts to modify the list.
 *
 * <p><strong>NOTE</strong>: If you are creating and accessing a
 * <code>HashMap</code> only within a single thread, you should use
 * <code>java.util.HashMap</code> directly (with no synchronization), for
 * maximum performance.</p>
 *
 * @author Craig R. McClanahan, Paul Jack
 * @version $Revision: 1.5 $ $Date: 2002/02/10 08:07:42 $
 */

public class FastHashMap extends HashMap {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a an empty map.
     */
    public FastHashMap() {

        this(new HashMap());

    }


    /**
     * Construct an empty map with the specified capacity.
     *
     * @param capacity The initial capacity of the empty map
     */
    public FastHashMap(int capacity) {

        this(new HashMap(capacity));
    }


    /**
     * Construct an empty map with the specified capacity and load factor.
     *
     * @param capacity The initial capacity of the empty map
     * @param factor The load factor of the new map
     */
    public FastHashMap(int capacity, float factor) {

        this(new HashMap(capacity, factor));

    }


    /**
     * Construct a new map with the same mappings as the specified map.
     *
     * @param map The map whose mappings are to be copied
     */
    public FastHashMap(Map map) {

        this(new HashMap(map));

    }


    private FastHashMap(HashMap map) {
        super(0);
        this.map = map;
        this.mapRef = SafeReference.create(map);
    }


    // ----------------------------------------------------- Instance Variables


    /**
     * The underlying map we are managing.
     *
     *  @deprecated  use setMap(HashMap map) and getMap() instead
     */
    private HashMap map = null;


    private transient SafeReference mapRef;


    // ------------------------------------------------------------- Properties


    /**
     * Are we operating in "fast" mode?
     */
    protected transient boolean fast = true;

    public boolean getFast() {
        return (this.fast);
    }

    public void setFast(boolean fast) {
        this.fast = fast;
    }


    protected HashMap getMap() {
        return (HashMap)mapRef.get();
    }


    protected void setMap(HashMap map) {
        mapRef.set(map);
        this.map = map;
    }


    HashMap getInternalClone() {
        return (HashMap)getMap().clone();
    }

    // --------------------------------------------------------- Public Methods


    /**
     * Remove all mappings from this map.
     */
    public void clear() {

        if (fast) {
            synchronized (this) {
                HashMap temp = (HashMap) getMap().clone();
                temp.clear();
                setMap(temp);;
            }
        } else {
            synchronized (map) {
                map.clear();
            }
        }

    }


    /**
     * Return a shallow copy of this <code>FastHashMap</code> instance.
     * The keys and values themselves are not copied.
     */
    public Object clone() {

        FastHashMap results = null;
        if (fast) {
            results = new FastHashMap(getMap());
        } else {
            synchronized (map) {
                results = new FastHashMap(map);
            }
        }
        results.setFast(getFast());
        return (results);

    }


    /**
     * Return <code>true</code> if this map contains a mapping for the
     * specified key.
     *
     * @param key Key to be searched for
     */
    public boolean containsKey(Object key) {

        if (fast) {
            return (getMap().containsKey(key));
        } else {
            synchronized (map) {
                return (map.containsKey(key));
            }
        }

    }


    /**
     * Return <code>true</code> if this map contains one or more keys mapping
     * to the specified value.
     *
     * @param value Value to be searched for
     */
    public boolean containsValue(Object value) {

        if (fast) {
            return (getMap().containsValue(value));
        } else {
            synchronized (map) {
                return (map.containsValue(value));
            }
        }

    }


    /**
     * Return a collection view of the mappings contained in this map.  Each
     * element in the returned collection is a <code>Map.Entry</code>.
     */
    public Set entrySet() {

        if (fast) {
            return new FastMapEntrySet(new FastHashMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.entrySet());
            }
        }

    }


    /**
     * Compare the specified object with this list for equality.  This
     * implementation uses exactly the code that is used to define the
     * list equals function in the documentation for the
     * <code>Map.equals</code> method.
     *
     * @param o Object to be compared to this list
     */
    public boolean equals(Object o) {

        // Simple tests that require no synchronization
        if (o == this)
            return (true);
        else if (!(o instanceof Map))
            return (false);
        Map mo = (Map) o;

        // Compare the two maps for equality
        if (fast) {
            if (mo.size() != map.size())
                return (false);
            java.util.Iterator i = getMap().entrySet().iterator();
            while (i.hasNext()) {
                Map.Entry e = (Map.Entry) i.next();
                Object key = e.getKey();
                Object value = e.getValue();
                if (value == null) {
                    if (!(mo.get(key) == null && mo.containsKey(key)))
                        return (false);
                } else {
                    if (!value.equals(mo.get(key)))
                        return (false);
                }
            }
            return (true);
        } else {
            synchronized (map) {
                if (mo.size() != map.size())
                    return (false);
                java.util.Iterator i = map.entrySet().iterator();
                while (i.hasNext()) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (value == null) {
                        if (!(mo.get(key) == null && mo.containsKey(key)))
                            return (false);
                    } else {
                        if (!value.equals(mo.get(key)))
                            return (false);
                    }
                }
                return (true);
            }
        }

    }


    /**
     * Return the value to which this map maps the specified key.  Returns
     * <code>null</code> if the map contains no mapping for this key, or if
     * there is a mapping with a value of <code>null</code>.  Use the
     * <code>containsKey()</code> method to disambiguate these cases.
     *
     * @param key Key whose value is to be returned
     */
    public Object get(Object key) {

        if (fast) {
            return (getMap().get(key));
        } else {
            synchronized (map) {
                return (map.get(key));
            }
        }

    }


    /**
     * Return the hash code value for this map.  This implementation uses
     * exactly the code that is used to define the list hash function in the
     * documentation for the <code>Map.hashCode</code> method.
     */
    public int hashCode() {

        if (fast) {
            int h = 0;
            java.util.Iterator i = getMap().entrySet().iterator();
            while (i.hasNext())
                h += i.next().hashCode();
            return (h);
        } else {
            synchronized (map) {
                int h = 0;
                java.util.Iterator i = map.entrySet().iterator();
                while (i.hasNext())
                    h += i.next().hashCode();
                return (h);
            }
        }

    }


    /**
     * Return <code>true</code> if this map contains no mappings.
     */
    public boolean isEmpty() {

        if (fast) {
            return (getMap().isEmpty());
        } else {
            synchronized (map) {
                return (map.isEmpty());
            }
        }

    }


    /**
     * Return a set view of the keys contained in this map.
     */
    public Set keySet() {

        if (fast) {
            return new FastMapKeySet(new FastHashMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.keySet());
            }
        }

    }


    /**
     * Associate the specified value with the specified key in this map.
     * If the map previously contained a mapping for this key, the old
     * value is replaced and returned.
     *
     * @param key The key with which the value is to be associated
     * @param value The value to be associated with this key
     */
    public Object put(Object key, Object value) {

        if (fast) {
            synchronized (this) {
                HashMap temp = (HashMap) getMap().clone();
                Object result = temp.put(key, value);
                setMap(temp);
                return (result);
            }
        } else {
            synchronized (map) {
                return (map.put(key, value));
            }
        }

    }


    /**
     * Copy all of the mappings from the specified map to this one, replacing
     * any mappings with the same keys.
     *
     * @param in Map whose mappings are to be copied
     */
    public void putAll(Map in) {

        if (fast) {
            synchronized (this) {
                HashMap temp = (HashMap) getMap().clone();
                temp.putAll(in);
                setMap(temp);;
            }
        } else {
            synchronized (map) {
                map.putAll(in);
            }
        }

    }


    /**
     * Remove any mapping for this key, and return any previously
     * mapped value.
     *
     * @param key Key whose mapping is to be removed
     */
    public Object remove(Object key) {

        if (fast) {
            synchronized (this) {
                HashMap temp = (HashMap) getMap().clone();
                Object result = temp.remove(key);
                setMap(temp);;
                return (result);
            }
        } else {
            synchronized (map) {
                return (map.remove(key));
            }
        }

    }


    /**
     * Return the number of key-value mappings in this map.
     */
    public int size() {

        if (fast) {
            return (getMap().size());
        } else {
            synchronized (map) {
                return (map.size());
            }
        }

    }


    /**
     * Return a collection view of the values contained in this map.
     */
    public Collection values() {

        if (fast) {
            return new FastMapValues(new FastHashMapInnerCollectionState(this));
        } else {
            synchronized (map) {
                return (map.values());
            }
        }

    }


    public String toString() {
        return getMap().toString();
    }


    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        HashMap temp = (HashMap)getMap();
        out.writeObject(temp);
    }


    private void readObject(ObjectInputStream inp) throws IOException, ClassNotFoundException {
        inp.defaultReadObject();
	HashMap map = (HashMap)inp.readObject();
        this.mapRef = SafeReference.create(map);
	this.map = map;
    }


}


class FastHashMapInnerCollectionState extends FastMapInnerCollectionState
{

	public FastHashMapInnerCollectionState(FastHashMap fast)
	{
		super(fast, fast);
	}


	public Map getInternalMap()
	{
		return ((FastHashMap)fast).getMap();
	}


	public void setInternalMap(Map map)
	{
		((FastHashMap)fast).setMap((HashMap)map);
	}


	public Map getInternalClone()
	{
		HashMap hash = (HashMap)getInternalMap();
		Map clone = (Map)hash.clone();
		return clone;
	}


}
