package org.apache.commons.collections;


import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;


/**
 *  Defines common functionality found in both {@link FastHashMap}
 *  and {@link FastTreeMap}.  Since those two classes don't have
 *  a common superclass, and since I don't want these methods to 
 *  be defined by a public interface, this class is needed to 
 *  provide a common implementation for the collection views.
 *
 *  @author Paul Jack
 */
abstract class FastMapInnerCollectionState
{

	/**
	 *  The monitor to enter during modifications.
	 */
	final Object monitor;


	/**
	 *  The map implementation; will be a FastHashMap, a
	 *  FastTreeMap or a SubFastTreeMap.
	 */ 
	final Map fast;


	/**
	 *  Constructor.
	 *
	 *  @param monitor  the monitor to enter during modifications
	 *  @param fast  the fast map 
	 */
	FastMapInnerCollectionState(Object monitor, Map fast)
	{
		this.monitor = monitor;
		this.fast = fast;
	}


	/**
	 *  Returns a clone of the fast map's internal map.
	 *
	 *  @return a clone of the fast map's internal map
	 */
	abstract Map getInternalClone();

	/**
	 *  Returns the fast map's internal map.
	 *
	 *  @return the fast map's internal map.
	 */
	abstract Map getInternalMap();


	/**
	 *  Sets the fast map's internal map.
	 *
	 *  @param clone the new internal map for the fast map
	 */
	abstract void setInternalMap(Map clone);


	/**
	 *  Returns a sub map.  This is really only used for 
	 *  SubFastTreeMaps; the default implementation just returns
	 *  the given map.
	 *
	 *  @param map  the fast map that will generate the submap
	 *  @return  the sub map
	 */
	Map getSubMap(Map map)
	{
		return map;
	}


	/**
	 *  Returns a sub map.  This is a convienence for getting the
	 *  sub map of the fast map's internal map.
	 *
	 *  @return a sub map
	 */
	Map getSubMap()
	{
		return getSubMap(getInternalMap());
	}

}


/**
 *  Common superclass for key sets, entry sets and value collections
 *  for the fast maps.
 */
abstract class FastMapInnerCollection implements Collection
{

	/**
	 *  Provides the means to get the internal map, clone it, set it,
	 *  or create a sub map from it.
	 */
	FastMapInnerCollectionState state;


	/**
	 *  Constructor.
	 *
	 *  @param state  the fast map class-specific implementation
	 */
	public FastMapInnerCollection(FastMapInnerCollectionState state) 
	{
		this.state = state;
	}


	/**
	 *  Returns the "original" collection.  This returns the direct
	 *  collection view suitable for read operations.
	 */
	Collection getOriginal()
	{
		return getOriginal(state.getSubMap());
	}


	/**
	 *  Returns the "original" collection.  For instance, if this
	 *  collections represents a fast map's key set, this method will
	 *  return the key set of the given map.
	 *
	 *  @param clone  the map whose collection view to return
	 *  @return a collection view of that map
	 */
	abstract Collection getOriginal(Map clone);


	// Collection interface implementation:
	//
	// There are two basic idioms at work here.  For read operations:
	//
	// 1.  Fetch the most recent reference to the fast map's internal map.
	// 2.  If necessary, constrain the internal map to a sub map.
	// 3.  Create the collection view for the sub map.
	// 4.  Perform the read operation on the collection view.
	//
	// Since both HashMap and TreeMap cache their collection views, the
	// idiom isn't that slow.
	//
	// For write operations:
	//
	// 1. Enter the monitor.
	// 2. Fetch the most recent reference to the fast map's internal map.
	// 3. Clone the internal map.
	// 4. If necessary, constrain the clone to a sub map.
	// 5. Create the collection view for the sub map.
	// 6. Perform the write operation on the clone's sub map's collection view.
	// 7. Set the reference to the fast map's internal map to the clone.
	
	public boolean contains(Object object)
	{
		return getOriginal().contains(object);
	}


	public boolean add(Object object)
	{
		// As per spec.
		throw new UnsupportedOperationException();
	}


	public boolean addAll(Collection col)
	{
		// as per spec.
		throw new UnsupportedOperationException();
	}


	public boolean isEmpty()
	{
		return state.getSubMap().isEmpty();
	}


	public void clear()
	{
		synchronized (state.monitor)
		{
			Map clone = state.getInternalClone();
			Map sub = state.getSubMap(clone);
			if (sub.size() > 0)
			{
				sub.clear();
				state.setInternalMap(clone);
			}
		}
	}


	public int size()
	{
		return state.getSubMap().size();
	}


	public boolean removeAll(Collection c)
	{
		synchronized (state.monitor)
		{
			Map clone = state.getInternalClone();
			Map sub = state.getSubMap(clone);
			boolean r = getOriginal(sub).removeAll(c);
			if (r) state.setInternalMap(clone);
			return r;
		}
	}


	public boolean retainAll(Collection c)
	{
		synchronized (state.monitor)
		{
			Map clone = state.getInternalClone();
			Map sub = state.getSubMap(clone);
			boolean r = getOriginal(sub).retainAll(c);
			if (r) state.setInternalMap(clone);
			return r;
		}
	}


	public boolean containsAll(Collection c)
	{
		return getOriginal().containsAll(c);
	}


	public boolean remove(Object value)
	{
		synchronized (state.monitor)
		{
			Map clone = state.getInternalClone();
			Map sub = state.getSubMap(clone);
			boolean r = getOriginal(sub).remove(value);
			if (r) state.setInternalMap(clone);
			return r;
		}
	}


	public Iterator iterator()
	{
		Map original = state.getInternalMap();
		Iterator iterator = getOriginal(state.getSubMap(original)).iterator();
		return new FastMapIterator(this, original, iterator);
	}


	public Object[] toArray()
	{
		return getOriginal().toArray();
	}


	public Object[] toArray(Object[] arr)
	{
		return getOriginal().toArray(arr);
	}


	public String toString()
	{
		return getOriginal().toString();
	}


	public boolean equals(Object o)
	{
		return getOriginal().equals(o);
	}


	public int hashCode()
	{
		return getOriginal().hashCode();
	}


}


class FastMapKeySet extends FastMapInnerCollection implements Set
{

	public FastMapKeySet(FastMapInnerCollectionState state)
	{
		super(state);
	}


	public Collection getOriginal(Map map)
	{
		return map.keySet();
	}

}


class FastMapEntrySet extends FastMapInnerCollection implements Set
{

	public FastMapEntrySet(FastMapInnerCollectionState state)
	{
		super(state);
	}


	public Collection getOriginal(Map map)
	{
		return map.entrySet();
	}

}


class FastMapValues extends FastMapInnerCollection
{

	public FastMapValues(FastMapInnerCollectionState state)
	{
		super(state);
	}


	public Collection getOriginal(Map map)
	{
		return map.values();
	}

}


class FastMapIterator implements Iterator
{

	private FastMapInnerCollection col;
	private Map original;
	private Iterator iterator;
	private int count;
	private Object mostRecentlyReturned;


	public FastMapIterator(FastMapInnerCollection col, Map original, Iterator iterator)
	{
		this.col = col;
		this.original = original;
		this.iterator = iterator;
	}
	

	public boolean hasNext()
	{
		return iterator.hasNext();
	}


	public Object next()
	{
		count++;
		mostRecentlyReturned = iterator.next();
		return mostRecentlyReturned;
	}


	public void remove()
	{
		synchronized (col.state.monitor)
		{
			checkMod();
			if (col instanceof Set)
			{
				col.remove(mostRecentlyReturned);
				count--;
				mostRecentlyReturned = null;
				return;
			}
			Map clone = col.state.getInternalClone();
			Map sub = col.state.getSubMap(clone);
			Collection c = col.getOriginal(sub);
			Iterator iter = c.iterator();
			for (int i = 0; i < count; i++)
			{
				iter.next();
			}
			iter.remove();
			count--;
			col.state.setInternalMap(clone);
			this.original = clone;
			this.iterator = iter;
		}
	}


	private void checkMod()
	{
		if (col.state.getInternalMap() != original)
		{
			throw new ConcurrentModificationException();
		}
	}

}
